import os
from concurrent.futures import ProcessPoolExecutor, as_completed
from tqdm import tqdm
import numpy as np


# ---------- GLOBAL READ-ONLY SHARED ARRAYS -----------------
# You *inject* these from your main program once, just after loading them.
JUDGE_SCORES       = None
WEAK_JUDGE_SCORES  = None

def _init_globals(judge_scores, weak_judge_scores):
    """Call this exactly once in the main process *before* spawning workers."""
    global JUDGE_SCORES, WEAK_JUDGE_SCORES
    JUDGE_SCORES      = judge_scores
    WEAK_JUDGE_SCORES = weak_judge_scores
    

def _run_parallel(num_iterations, worker_fn, *, max_workers=None, desc=""):
    """
    Run `worker_fn(s)` for all s in [0, num_iterations) in parallel.
    The helper collects results in order and returns a NumPy array.
    """
    max_workers = max_workers or os.cpu_count() or 1

    # --- Pass _init_globals to each worker on spawn-based systems ---
    init_kw = {}
    if os.name == "nt":          # Windows needs the initializer
        init_kw.update(
            initializer=_init_globals,
            initargs=(JUDGE_SCORES, WEAK_JUDGE_SCORES),
        )

    out = [None] * num_iterations
    with ProcessPoolExecutor(max_workers=max_workers, **init_kw) as pool:
        fs = {pool.submit(worker_fn, s): s for s in range(num_iterations)}
        for f in tqdm(as_completed(fs), total=num_iterations, desc=desc):
            out[fs[f]] = f.result()

    return np.asarray(out)