import os
import pandas as pd
import numpy as np
from glob import glob
from scipy.stats import bootstrap
import argparse
from tabulate import tabulate


def format_ci_or_minmax(series, mode='ci'):
    if len(series) < 2:
        return f"{series.mean() * 100:.3f}% [N/A]"
    
    mean = series.mean() * 100
    if mode == 'ci':
        data = series.values.reshape(1, -1)
        ci = bootstrap(data, np.mean, confidence_level=0.95, method='percentile').confidence_interval
        ci_low = ci.low * 100
        ci_high = ci.high * 100
        return f"{mean:.3f}% [{ci_low:.3f}%, {ci_high:.3f}%]"
    elif mode == 'minmax':
        min_val = series.min() * 100
        max_val = series.max() * 100
        return f"{mean:.3f}% [{min_val:.3f}%, {max_val:.3f}%]"
    else:
        raise ValueError("mode must be 'ci' or 'minmax'")


def analyze_experiment(exp_name, all_files, mode='ci'):
    df = pd.concat([pd.read_csv(f) for f in all_files])
    df_grouped = df.groupby('train_size')
    metrics_to_display = [col for col in df.columns if col not in ['exp', 'seed', 'train_size']]

    table = []
    for train_size, group in df_grouped:
        row = [train_size]
        for col in metrics_to_display:
            try:
                row.append(format_ci_or_minmax(group[col].astype(float), mode=mode))
            except Exception:
                row.append("N/A")
        table.append(row)

    header = ["train_size"] + metrics_to_display
    section = f"### Experiment {exp_name}\n\n"
    section += tabulate(table, headers=header, tablefmt="github") + "\n"
    return section

def main(mode='ci'):
    result_dir = "results"
    exps = ['exp1', 'exp2', 'exp3', 'exp4']
    output = []

    for exp in exps:
        files = sorted(glob(os.path.join(result_dir, f"{exp}_size*_seed*.csv")))
        if not files:
            continue
        section = analyze_experiment(exp, files, mode=mode)
        output.append(section)

    print("\n\n".join(output))

if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Summarize experiment results with CI or min/max.")
    parser.add_argument('--mode', type=str, default='ci', choices=['ci', 'minmax'],
                        help="Display mode: 'ci' for confidence interval, 'minmax' for min-max range")
    args = parser.parse_args()
    main(mode=args.mode)
