"""
    ResNet for CIFAR/SVHN, implemented in PyTorch.
    Original paper: 'Deep Residual Learning for Image Recognition,' https://arxiv.org/abs/1512.03385.
    Adapted from https://github.com/osmr/imgclsmob/blob/master/pytorch/pytorchcv/models/resnet_cifar.py
"""

import os
import torch
import torch.nn as nn
import torch.nn.init as init
from Layers.layers import conv3x3_block, Conv2d, BatchNorm2d, Linear, sparse_initialize
from Models.imagenet_resnet import ResUnit, ResBlock


class CIFARResNet(nn.Module):
    def __init__(self, channels, init_block_channels, bottleneck, in_channels=3, in_size=(32, 32), num_classes=10):
        super(CIFARResNet, self).__init__()
        self.in_size = in_size
        self.num_classes = num_classes

        self.features = nn.Sequential()
        self.features.add_module("init_block", conv3x3_block(in_channels=in_channels, out_channels=init_block_channels))
        in_channels = init_block_channels
        for i, channels_per_stage in enumerate(channels):
            stage = nn.Sequential()
            for j, out_channels in enumerate(channels_per_stage):
                stride = 2 if (j == 0) and (i != 0) else 1
                stage.add_module(
                    "unit{}".format(j + 1),
                    ResUnit(in_channels=in_channels, out_channels=out_channels, stride=stride, bottleneck=bottleneck, conv1_stride=False))
                in_channels = out_channels
            self.features.add_module("stage{}".format(i + 1), stage)
        self.features.add_module("final_pool", nn.AvgPool2d(kernel_size=8, stride=1))

        self.output = Linear(in_features=in_channels, out_features=num_classes)
        self._initialize_weights(sparse_init=False)
        self.set_prune_types()

    def _initialize_weights(self, sparse_init=True):
        if sparse_init:
            sparse_initialize(self)
        else:
            for name, module in self.named_modules():
                if isinstance(module, Conv2d):
                    init.kaiming_uniform_(module.weight)
                    if module.bias is not None:
                        init.constant_(module.bias, 0)

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.output(x)
        return x

    def clean(self):
        with torch.no_grad():
            self.cuda()
            self._initialize_weights()
            self(torch.randn(64, 3, 32, 32).cuda())

    def set_prune_types(self):
        self.pruned_types = {}
        unit_list = []
        for name, module in self.named_modules():
            if isinstance(module, ResUnit):
                unit_list.append(module)
        length = len(unit_list)
        self.pruned_types[self.features.init_block] = 'vgg_out'
        for index, unit in enumerate(unit_list):
            self.pruned_types[unit.body.conv1] = 'in'
            self.pruned_types[unit.body.conv2] = 'vgg_out'
            if unit.resize_identity:
                self.pruned_types[unit.identity_conv] = 'in'


def get_resnet_cifar(blocks, bottleneck, width_scale=1.0, **kwargs):

    if bottleneck:
        assert ((blocks - 2) % 9 == 0)
        layers = [(blocks - 2) // 9] * 3
    else:
        assert ((blocks - 2) % 6 == 0)
        layers = [(blocks - 2) // 6] * 3

    channels_per_layers = [16, 32, 64]
    init_block_channels = 16

    channels = [[ci] * li for (ci, li) in zip(channels_per_layers, layers)]

    if bottleneck:
        channels = [[cij * 4 for cij in ci] for ci in channels]

    if width_scale != 1.0:
        channels = [[int(cij * width_scale) if (i != len(channels) - 1) or (j != len(ci) - 1) else cij for j, cij in enumerate(ci)]
                    for i, ci in enumerate(channels)]
        init_block_channels = int(init_block_channels * width_scale)

    net = CIFARResNet(channels=channels, init_block_channels=init_block_channels, bottleneck=bottleneck, **kwargs)

    return net


def resnet20_cifar10(groups=1, width_factor=1, **kwargs):
    assert groups == 1
    return get_resnet_cifar(blocks=20, bottleneck=False, width_scale=width_factor, **kwargs)


def resnet56_cifar10(groups=1, width_factor=1, **kwargs):
    assert groups == 1
    return get_resnet_cifar(blocks=56, bottleneck=False, width_scale=width_factor, **kwargs)


def _test():
    import torch

    models = [
        (resnet20_cifar10, 10),
        (resnet56_cifar10, 10),
    ]

    for model, num_classes in models:

        net = model(width_factor=3)

        # net.train()
        net.eval()
        print(model.__name__)
        x = torch.randn(1, 3, 32, 32)
        y = net(x)
        y.sum().backward()
        print(y)
        assert (tuple(y.size()) == (1, num_classes))


if __name__ == "__main__":
    _test()