"""
    EfficientNet for ImageNet-1K, implemented in PyTorch.
    Original papers:
    - 'EfficientNet: Rethinking Model Scaling for Convolutional Neural Networks,' https://arxiv.org/abs/1905.11946,
    - 'Adversarial Examples Improve Image Recognition,' https://arxiv.org/abs/1911.09665.
    Adapted from https://github.com/osmr/imgclsmob/blob/master/pytorch/pytorchcv/models/efficientnet.py
"""

import os
import math
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.nn.init as init
from Layers.layers import conv1x1_block, conv3x3_block, dwconv3x3_block, dwconv5x5_block, Conv2d, sparse_initialize, SEBlock, Linear, BatchNorm2d, PaddedIdentityAdd
from Layers.common import round_channels


class EffiDwsConvUnit(nn.Module):
    def __init__(self, in_channels, out_channels, stride, activation):
        super(EffiDwsConvUnit, self).__init__()
        self.residual = (in_channels == out_channels) and (stride == 1)

        self.dw_conv = dwconv3x3_block(in_channels=in_channels,
                                       out_channels=in_channels,
                                       padding=1,
                                       activation=activation)
        self.se = SEBlock(channels=in_channels,
                          reduction=4,
                          mid_activation=activation,
                          use_conv=False)
        self.pw_conv = conv1x1_block(in_channels=in_channels,
                                     out_channels=out_channels,
                                     activation=None)
        self.identity = PaddedIdentityAdd()

    def forward(self, x):
        if self.residual:
            identity = x
        x = self.dw_conv(x)
        x = self.se(x)
        x = self.pw_conv(x)
        if self.residual:
            # x = x + identity
            x = self.identity(x, identity)
        # print(self, x.shape)
        return x


class EffiInvResUnit(nn.Module):
    def __init__(self, in_channels, out_channels, kernel_size, stride,
                 exp_factor, se_factor, activation):
        super(EffiInvResUnit, self).__init__()
        self.kernel_size = kernel_size
        self.stride = stride
        self.activation = activation
        self.residual = (stride == 1)
        self.use_se = se_factor > 0
        mid_channels = in_channels * exp_factor
        dwconv_block_fn = dwconv3x3_block if kernel_size == 3 else (
            dwconv5x5_block if kernel_size == 5 else None)

        self.conv1 = conv1x1_block(in_channels=in_channels,
                                   out_channels=mid_channels,
                                   activation=activation,
                                   shuffle=True)
        self.conv2 = dwconv_block_fn(in_channels=mid_channels,
                                     out_channels=mid_channels,
                                     stride=stride,
                                     padding=kernel_size // 2,
                                     activation=activation)
        if self.use_se:
            self.se = SEBlock(channels=mid_channels,
                              reduction=(exp_factor * se_factor),
                              mid_activation=activation,
                              use_conv=False)
        self.conv3 = conv1x1_block(in_channels=mid_channels,
                                   out_channels=out_channels,
                                   activation=None)
        if self.residual:
            self.identity = PaddedIdentityAdd()

    def forward(self, x):
        if self.residual:
            identity = x
        x = self.conv1(x)
        x = self.conv2(x)
        if self.use_se:
            x = self.se(x)
        x = self.conv3(x)
        if self.residual:
            # x = x + identity
            x = self.identity(x, identity)
        # print(self, x.shape)
        return x


class EffiInitBlock(nn.Module):
    def __init__(self, in_channels, out_channels, activation):
        super(EffiInitBlock, self).__init__()

        self.conv = conv3x3_block(in_channels=in_channels,
                                  out_channels=out_channels,
                                  stride=2,
                                  padding=1,
                                  activation=activation)

    def forward(self, x):
        x = self.conv(x)
        return x


class EfficientNet(nn.Module):
    def __init__(self,
                 channels,
                 init_block_channels,
                 final_block_channels,
                 kernel_sizes,
                 strides_per_stage,
                 expansion_factors,
                 dropout_rate=0.2,
                 in_channels=3,
                 in_size=(224, 224),
                 num_classes=1000):
        super(EfficientNet, self).__init__()
        self.in_size = in_size
        self.num_classes = num_classes
        activation = "swish"

        self.features = nn.Sequential()
        self.features.add_module(
            "init_block",
            EffiInitBlock(in_channels=in_channels,
                          out_channels=init_block_channels,
                          activation=activation))
        in_channels = init_block_channels
        for i, channels_per_stage in enumerate(channels):
            kernel_sizes_per_stage = kernel_sizes[i]
            expansion_factors_per_stage = expansion_factors[i]
            stage = nn.Sequential()
            for j, out_channels in enumerate(channels_per_stage):
                kernel_size = kernel_sizes_per_stage[j]
                expansion_factor = expansion_factors_per_stage[j]
                stride = strides_per_stage[i] if (j == 0) else 1
                if i == 0:
                    stage.add_module(
                        "unit{}".format(j + 1),
                        EffiDwsConvUnit(in_channels=in_channels,
                                        out_channels=out_channels,
                                        stride=stride,
                                        activation=activation))
                else:
                    stage.add_module(
                        "unit{}".format(j + 1),
                        EffiInvResUnit(in_channels=in_channels,
                                       out_channels=out_channels,
                                       kernel_size=kernel_size,
                                       stride=stride,
                                       exp_factor=expansion_factor,
                                       se_factor=4,
                                       activation=activation))
                in_channels = out_channels
            self.features.add_module("stage{}".format(i + 1), stage)
        self.features.add_module(
            "final_block",
            conv1x1_block(in_channels=in_channels,
                          out_channels=final_block_channels,
                          activation=activation))
        in_channels = final_block_channels
        self.features.add_module("final_pool",
                                 nn.AdaptiveAvgPool2d(output_size=1))

        self.output = nn.Sequential()
        if dropout_rate > 0.0:
            self.output.add_module("dropout", nn.Dropout(p=dropout_rate))
        self.output.add_module(
            "fc", Linear(in_features=in_channels, out_features=num_classes))

        self._initialize_weights(sparse_init=False)
        self._set_prune_types()

    def _initialize_weights(self, sparse_init=True):
        if sparse_init:
            sparse_initialize(self)
        else:
            for name, module in self.named_modules():
                if isinstance(module, Conv2d):
                    init.kaiming_uniform_(module.weight)
                    if module.bias is not None:
                        init.constant_(module.bias, 0)

    def set_eps(self, eps):
        for module in self.modules():
            if isinstance(module, BatchNorm2d):
                module.eps = eps

    def _set_prune_types(self):
        self.pruned_types = {}
        dws_unit_list = []
        inv_unit_list = []
        for name, module in self.named_modules():
            if isinstance(module, EffiDwsConvUnit):
                dws_unit_list.append(module)
            if isinstance(module, EffiInvResUnit):
                inv_unit_list.append(module)
        #self.pruned_types[self.features.init_block.conv] = 'skip'
        for index, unit in enumerate(dws_unit_list):
            self.pruned_types[unit.pw_conv] = 'only_out'
        for index, unit in enumerate(inv_unit_list):
            if unit.residual:
                self.pruned_types[unit.conv1] = 'in'
            else:
                self.pruned_types[unit.conv1] = 'nonresidual_in'
            self.pruned_types[unit.conv3] = 'vgg_out'
        # self.pruned_types[self.features.final_block] = 'only_out'

    def clean(self):
        prev_out = 32
        for n, m in self.named_modules():
            if isinstance(m, EffiInvResUnit):
                if not m.residual and prev_out is not None:
                    in_channels = prev_out
                    out_channels = m.conv1.conv.out_channels
                    activation = None if not hasattr(m.conv1,
                                                     'activ') else m.conv1.activ
                    m.conv1.__init__(in_channels, out_channels,
                                     m.conv1.conv.kernel_size,
                                     m.conv1.conv.stride, m.conv1.conv.groups,
                                     m.conv1.conv.padding, m.conv1.conv.bias,
                                     activation, m.conv1.shuffle)
                    prev_out = m.conv3.conv.out_channels
                else:
                    prev_out = m.identity.out_channels

        m = self.features.final_block
        in_channels = prev_out
        out_channels = m.conv.out_channels
        activation = None if not hasattr(m, 'activ') else m.activ
        m.__init__(in_channels, out_channels, m.conv.kernel_size, m.conv.stride,
                   m.conv.groups, m.conv.padding, m.conv.bias, activation,
                   m.shuffle)
        # in_channels = self.features.stage5.unit7.identity.out_channels
        # print(in_channels)
        # m = self.features.final_block
        # activation = None if not hasattr(m, 'activ') else m.activ
        # m.__init__(in_channels, m.conv.out_channels, m.conv.kernel_size,
        #            m.conv.stride, m.conv.groups, m.conv.padding, m.conv.bias,
        #            activation, m.shuffle)
        with torch.no_grad():
            self.cuda()
            self._initialize_weights()
            self(torch.randn(64, 3, 224, 224).cuda())

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.output(x)
        return x


def get_efficientnet(version, in_size, **kwargs):
    assert version in ["b0", "b2"]
    # change the input_shape for both model to 224 x 224
    assert (in_size == (224, 224))
    if version == "b0":
        depth_factor = 1.0
        width_factor = 1.0
        dropout_rate = 0.2
    elif version == "b2":
        depth_factor = 1.2
        width_factor = 1.1
        dropout_rate = 0.3

    init_block_channels = 32
    layers = [1, 2, 2, 3, 3, 4, 1]
    downsample = [1, 1, 1, 1, 0, 1, 0]
    channels_per_layers = [16, 24, 40, 80, 112, 192, 320]
    expansion_factors_per_layers = [1, 6, 6, 6, 6, 6, 6]
    kernel_sizes_per_layers = [3, 3, 5, 3, 5, 5, 3]
    strides_per_stage = [1, 2, 2, 2, 1, 2, 1]
    final_block_channels = 1280

    layers = [int(math.ceil(li * depth_factor)) for li in layers]
    channels_per_layers = [
        round_channels(ci * width_factor) for ci in channels_per_layers
    ]

    from functools import reduce
    channels = reduce(
        lambda x, y: x + [[y[0]] * y[1]]
        if y[2] != 0 else x[:-1] + [x[-1] + [y[0]] * y[1]],
        zip(channels_per_layers, layers, downsample), [])
    kernel_sizes = reduce(
        lambda x, y: x + [[y[0]] * y[1]]
        if y[2] != 0 else x[:-1] + [x[-1] + [y[0]] * y[1]],
        zip(kernel_sizes_per_layers, layers, downsample), [])
    expansion_factors = reduce(
        lambda x, y: x + [[y[0]] * y[1]]
        if y[2] != 0 else x[:-1] + [x[-1] + [y[0]] * y[1]],
        zip(expansion_factors_per_layers, layers, downsample), [])
    strides_per_stage = reduce(
        lambda x, y: x + [[y[0]] * y[1]]
        if y[2] != 0 else x[:-1] + [x[-1] + [y[0]] * y[1]],
        zip(strides_per_stage, layers, downsample), [])
    strides_per_stage = [si[0] for si in strides_per_stage]

    init_block_channels = round_channels(init_block_channels * width_factor)

    if width_factor > 1.0:
        assert (int(final_block_channels * width_factor) == round_channels(
            final_block_channels * width_factor))
        final_block_channels = round_channels(final_block_channels *
                                              width_factor)

    net = EfficientNet(channels=channels,
                       init_block_channels=init_block_channels,
                       final_block_channels=final_block_channels,
                       kernel_sizes=kernel_sizes,
                       strides_per_stage=strides_per_stage,
                       expansion_factors=expansion_factors,
                       dropout_rate=dropout_rate,
                       in_size=in_size,
                       **kwargs)

    return net


def efficientnet_b0(input_shape=(224, 224),
                    num_classes=1000,
                    groups=1,
                    width_factor=1.,
                    **kwargs):
    # change the input_shape from 260 to 224
    assert groups == 1
    assert width_factor == 1
    if len(input_shape) == 3:
        input_shape = input_shape[1:]
    model = get_efficientnet(version="b0", in_size=input_shape, **kwargs)
    # model.set_eps(1e-3)
    return model


def efficientnet_b2(input_shape=(224, 224),
                    num_classes=1000,
                    groups=1,
                    width_factor=1.,
                    **kwargs):
    # change the input_shape from 260 to 224
    assert groups == 1
    assert width_factor == 1
    if len(input_shape) == 3:
        input_shape = input_shape[1:]
    model = get_efficientnet(version="b2", in_size=input_shape, **kwargs)
    # model.set_eps(1e-3)
    return model


def _test():
    import torch

    models = [
        efficientnet_b0,
    ]

    for model in models:

        net = model()

        # net.train()
        net.eval()
        x = torch.randn(1, 3, net.in_size[0], net.in_size[1])
        y = net(x)
        y.sum().backward()
        print(y.shape)
        assert (tuple(y.size()) == (1, 1000))


if __name__ == "__main__":
    _test()
