from dataclasses import dataclass

def get_category_restriction(features):
    category_strings = []
    
    for key, values in features.items():
        quoted_values = [f"'{value}'" for value in values]
        
        if len(quoted_values) > 1:
            last_value = quoted_values[-1]
            remaining_values = ", ".join(quoted_values[:-1])
            category_strings.append(f'The <{key}> can only be either {remaining_values}, or {last_value}')
        
        else: 
            category_strings.append(f'The <{key}> can only be {quoted_values[0]}')

    return "\n".join(category_strings)

@dataclass
class Config:
    features: dict
    string_templates: dict
    MQ: list
    EQ: list
    error_threshold: int
    verbose: bool


configs = [
    {
        'features': {
            'continent'  : ['Africa', 'Americas', 'Asia', 'Europe', 'North America', 'Oceania', 'South America', 'Australia', 'Eurasia'], 
            'time period': ['before the year 1875', 'between the years 1875 and 1925', 'between the years 1925 and 1951', 'between the years 1951 and 1970', 'after the year 1970'], 
            'occupation' : ['fashion designer', 'nurse', 'dancer', 'priest', 'footballer', 'banker', 'singer', 'lawyer', 'mathematician', 'diplomat'], 
            'gender'     : ['woman', 'man']
        },

        'string_templates':{
            'feature_string': "<continent>, <time period>, <occupation>, and <gender>",
            'reply_format'   : "A person that is born in <continent>, <time period> and is a <occupation> and is a <gender> is possible",
            'base_hypothesis': "'A person born in any continent, in any time period, who is any occupation, could be any gender.' describes the real world accurately",
            'updated_hypothesis': "{base_hypothesis}, except {not_clauses}{imply_clauses}",
            'stop_prompt'    : "I cannot think of any more counterexamples",

            'systems_prompt' : "A counterexample illustrates where the hypothesis is wrong. It is either a combination of values that in not possible in the real world but is possible in my hypothesis (negative counterexample) OR a combination of values that are possible in the real world but are not possible in my hypothesis (positive counterexample). Your response should be one counterexample in the format 'A person born in <continent>, <time period>, who is a <occupation> CAN be a <gender>' if the counterexample is a positive counterexample and 'A person born in <continent>, <time period>, who is a <occupation> CANNOT be a <gender>' if the counterexample is a negative counterexample. Each variable must have a value following these restrictions:\n{category_restrictions}.\nIf you think that my hypothesis accurately and completely describes the real world then only reply with '{stop_prompt}'.\n",
            
            'summary_prompt' : [{'role': 'user', 'content':"<Assistant response>{response}</Assistant response>\n\nSummarize the response from the assistant. Reply with the counterexample from the assistant in the format 'A person born in <continent>, <time period>, who is a <occupation> CAN be a <gender>' if the counterexample is a positive counterexample and 'A person born in <continent>, <time period>, who is a <occupation> CANNOT be a <gender>' if the counterexample is a negative counterexample. Use these values: {category_restrictions}. Only reply with the counterexample in the sentence, ignore other examples. If there is no counterexample in the assistants reply, then repond with {stop_prompt}.\n"}],
            'summary_format': [{'role': 'user', 'content': "<Assistant>{response}</Assistant>\n<Summary>{summary_response}</Summary>\n\nThe summary of the assistant reply did not contain 'can' or 'cannot'. Summarize the reply from the assistant to one counterexample in the format 'A person born in <continent>, <time period>, who is a <occupation> can or cannot be a <gender>'."}],
            'summary_hallucinate': [{'role': 'user', 'content': "<Assistant>{response}</Assistant>\n<Summary>{summary_response}</Summary>\n\nThe summary of the assistant reply did contained multiple values for one or more categories. Summarize the reply from the assistant to one counterexample in the format 'A person born in <continent>, <time period>, who is a <occupation> can or cannot be a <gender>'."}],
            'values_identify': [{'role': 'user', 'content': "The response should be in the format: 'A person born in <continent>, <time period>, who is a <occupation> CAN/CANNOT be a <gender>'. You have to look that the response and the parsed response and identify if they use the same values. The reply was: <reply>{response}</reply>. Does the counterexample in the reply use these values? \n{category_parsed}\nIf the same values were used reply with only 'yes', if not then reply with only 'no'. Do not provide any additional information."}],
            
            'parsing': {
                'MQ_sentence': 'Has {gender}{continent}{time_period}{occupation} been possible in the real world?',
                'sentence': '{gender} {continent} {time_period} {occupation}',
                'antecedent': 'if a {gender} {continent}{time_period}{occupation}, ',
                'consequent': 'then the {gender}{continent}{time_period}{occupation}',
                'gender': {'default': "A person", 'given':"A {value}", 'mq':"a {value}"},
                'time_period': {'default': "", 'given':"born {value}", 'mq':" born {value}"},
                'occupation': {'default': "", 'given':"that is a {value}", 'mq':" and was a {value}"},
                'continent': {'default': "", 'given':"born in {value}", 'mq':" born in {value}"},
                
            
            },
        },

        'MQ': [{"role": "user", "content": "{MQ_example} Reply with 'It is possible' if yes and 'it is NOT possible' if no. Do not specify which values are different. \n"}],
        'EQ': [{"role": "user", "content": "Current hypothesis: {hypo}\n{systems_prompt}"}],

        'error_threshold': 10,
        'verbose': False
    },
    {

        'features': {
            'gender'     : ['woman', 'man'],
            'time period': ['before the year 1800', 'between the years 1800 and 1900', 'between the years 1900 and 1950', 'between the years 1950 and 1970', 'after the year 1970'], 
            'occupation' : ['fashion designer', 'nurse', 'dancer', 'priest', 'footballer', 'banker', 'singer', 'lawyer', 'mathematician', 'diplomat'],
        },

        'string_templates':{
            'feature_string': "<time period>, <occupation>, and <gender>",
            'reply_format'   : "A person that is born in <time period> and is a <occupation> and is a <gender> is possible",
            'base_hypothesis': "'A person born in any time period, who is any occupation, could be any gender.' describes the real world accurately",
            'updated_hypothesis': "{base_hypothesis}, except {not_clauses}{imply_clauses}",
            'stop_prompt'    : "I cannot think of any more counterexamples",

            'systems_prompt' : "A counterexample illustrates where the hypothesis is wrong. It is either a combination of values that in not possible in the real world but my hypothesis says is possible (negative counterexample) OR a combination of values that are possible in the real world but are not possible in my hypothesis (positive counterexample). Your response should be one counterexample in the format 'A person born <time period>, who is a <occupation> CAN be a <gender>' if the counterexample is a positive counterexample and 'A person born <time period>, who is a <occupation> CANNOT be a <gender>' if the counterexample is a negative counterexample. Each variable must have a value following these restrictions:\n{category_restrictions}.\nIf you think that my hypothesis accurately and completely describes the real world then only reply with '{stop_prompt}'.\n",
            
            # Summary prompts
            'summary_prompt' : [{'role': 'user', 'content':"<Assistant response>{response}</Assistant response>\n\nSummarize the response from the assistant. Reply with the counterexample from the assistant in the format 'A person born <time period>, who is a <occupation> CAN be a <gender>' if the counterexample is a positive counterexample and 'A person born <time period>, who is a <occupation> CANNOT be a <gender>' if the counterexample is a negative counterexample. Use these values: {category_restrictions}. Only reply with the counterexample in the sentence, ignore other examples. If there is no counterexample in the assistants reply, then repond with {stop_prompt}.\n"}],
            'summary_format': [{'role': 'user', 'content': "<Assistant>{response}</Assistant>\n<Summary>{summary_response}</Summary>\n\nThe summary of the assistant reply did not contain 'can' or 'cannot'. Summarize the reply from the assistant to one counterexample in the format 'A person born <time period>, who is a <occupation> can or cannot be a <gender>'."}],
            'summary_hallucinate': [{'role': 'user', 'content': "<Assistant>{response}</Assistant>\n<Summary>{summary_response}</Summary>\n\nThe summary of the assistant reply did contained multiple values for one or more categories. Summarize the reply from the assistant to one counterexample in the format 'A person born <time period>, who is a <occupation> can or cannot be a <gender>'."}],
            'values_identify': [{'role': 'user', 'content': "The response should be in the format: 'A person born <time period>, who is a <occupation> CAN/CANNOT be a <gender>'. Does the response and the parsed sentence use the same values for the categories? The response was: <response>{response}</response> and the parsed sentence is {parsed_sentence}.\nIf both responses have the same values for each category then reply with 'yes', if not then reply with 'no'. Do not provide any additional information."}],
            
            'parsing': {
                'MQ_sentence': 'Has {gender}{time_period}{occupation} been possible in the real world?',
                'sentence': '{gender} {time_period} {occupation}',
                'antecedent': 'if a {gender} {time_period}{occupation}, ',
                'consequent': 'then the {gender}{time_period}{occupation}',
                'gender': {'default': "A person", 'given':"A {value}", 'mq':"a {value}"},
                'time_period': {'default': "", 'given':"born {value}", 'mq':" born {value}"},
                'occupation': {'default': "", 'given':"that is a {value}", 'mq':" and was a {value}"},
            },
        },

        'MQ': [{"role": "user", "content": "{MQ_example} Reply with 'It is possible' if yes and 'it is NOT possible' if no. Do not specify which values are different. \n"}],
        'EQ': [{"role": "user", "content": "Current hypothesis: {hypo}\n{systems_prompt}"}],

        'error_threshold': 10,
        'verbose': False
    },
]

for config in configs:
    config['string_templates']['category_restrictions'] = get_category_restriction(config['features'])
    config['string_templates']['systems_prompt'] = config['string_templates']['systems_prompt'].format(category_restrictions=config['string_templates']['category_restrictions'], 
                                                               reply_format=config['string_templates']['reply_format'], 
                                                               stop_prompt=config['string_templates']['stop_prompt'])
    config['string_templates']['summary_prompt'][0]['content'] = config['string_templates']['summary_prompt'][0]['content'].format(category_restrictions=config['string_templates']['category_restrictions'],
                                                               stop_prompt=config['string_templates']['stop_prompt'],
                                                               response="{response}")
    config['string_templates']['updated_hypothesis'] = config['string_templates']['updated_hypothesis'].format(base_hypothesis=config['string_templates']['base_hypothesis'], not_clauses="{not_clauses}", imply_clauses="{imply_clauses}")
