#!/usr/bin/env python3

import numpy as np
from pathlib import Path
from typing import Tuple, List

def load_vector_file(filepath: Path) -> np.ndarray:
    if not filepath.exists():
        return np.array([])
    try:
        vectors = np.load(filepath)
        return vectors
    except Exception:
        return np.array([])

def cosine_similarity(x: np.ndarray, y: np.ndarray) -> float:
    norm_x = np.linalg.norm(x)
    norm_y = np.linalg.norm(y)
    if norm_x > 0 and norm_y > 0:
        return np.dot(x, y) / (norm_x * norm_y)
    return 0.0

def compute_cosine_batch(X: np.ndarray, Y: np.ndarray) -> List[float]:
    cosines = []
    for i in range(len(X)):
        cos_sim = cosine_similarity(X[i], Y[i])
        cosines.append(cos_sim)
    return cosines

def train_test_split(X: np.ndarray, Y: np.ndarray, 
                    train_ratio: float = 0.8, 
                    seed: int = 42) -> Tuple[np.ndarray, np.ndarray, np.ndarray, np.ndarray]:
    np.random.seed(seed)
    n_samples = len(X)
    n_train = int(train_ratio * n_samples)
    
    indices = np.arange(n_samples)
    np.random.shuffle(indices)
    
    train_indices = indices[:n_train]
    test_indices = indices[n_train:]
    
    X_train = X[train_indices]
    Y_train = Y[train_indices]
    X_test = X[test_indices]
    Y_test = Y[test_indices]
    
    return X_train, Y_train, X_test, Y_test

def ensure_dir(path: Path):
    path.mkdir(parents=True, exist_ok=True)