#!/bin/bash
set -x


export HF_HUB_OFFLINE=1 && export WANDB_MODE=offline && export HF_DATASETS_OFFLINE=1

if [[ "$1" = "--use_lora" ]]; then
    switch_lora=false
    use_lora=true
    other_params=( --use_lora --lora_dropout 0.1)
elif [[ "$1" = "--switch_lora" ]]; then
    switch_lora=false
    use_lora=true
    other_params=( --switch_lora --adam_warm_step 5 --zero_switch_state --zero_switch_step_state)
else
    switch_lora=false
    use_lora=false
fi

shift 1

# Default values for parameters
task_name="stsb"
use_llama=False
# generate random int seed
seed=$(od -vAn -N4 -tu4 < /dev/urandom)
max_seq_length=512
num_train_epochs=30
batch_size=16
lora_rank=8
load_lora=True

# Parse command line arguments
while [[ $# -gt 0 ]]; do
    case "$1" in
        --model_name_or_path)
            model_name_or_path="$2"
            shift 2
            ;;
        --task_name)
            task_name="$2"
            shift 2
            ;;
        --use_llama)
            use_llama="$2"
            shift 2
            ;;
        --seed)
            seed="$2"
            shift 2
            ;;
        --output_dir)
            output_dir="$2"
            shift 2
            ;;
        --max_seq_length)
            max_seq_length="$2"
            shift 2
            ;;
        --num_train_epochs)
            num_train_epochs="$2"
            shift 2
            ;;
        --batch_size)
            batch_size="$2"
            shift 2
            ;;
        --lora_rank)
            lora_rank="$2"
            shift 2
            ;;
        --load_lora)
            load_lora="$2"
            shift 2
            ;;
        --lr)
            lr="$2"
            shift 2
            ;;
        *)
            echo "Unknown parameter: $1"
            exit 1
            ;;
    esac
done

# if output_dir is not set, set it to ./glue_output/$task_name
if [ -z "$output_dir" ]; then
    output_dir="./glue_output/$task_name"
fi

# if model_name_or_path is not set, exit
if [ -z "$model_name_or_path" ]; then
    echo "model_name_or_path is not set"
    exit 1
fi

# if lr is not set, exit
if [ -z "$lr" ]; then
    echo "lr is not set"
    exit 1
fi

# Print the parameter values
echo "model_name_or_path: $model_name_or_path"
echo "task_name: $task_name"
echo "seed: $seed"
echo "output_dir: $output_dir"
echo "max_seq_length: $max_seq_length"
echo "num_train_epochs: $num_train_epochs"
echo "batch_size: $batch_size"
echo "lora_rank: $lora_rank"
echo "load_lora: $load_lora"
echo "lr: $lr"


# Loop through each task
echo "Running task: $task_name"
python run_glue.py \
--model_name_or_path "$model_name_or_path" \
--task_name "$task_name" \
--use_llama "$use_llama" \
--do_train \
--do_eval \
--evaluation_strategy epoch \
--save_total_limit 2 \
--save_safetensors False \
--save_steps 1000000 \
--overwrite_output_dir \
--fp16 \
--cache_dir ./glue \
--seed "$seed" \
--output_dir "$output_dir" \
--max_seq_length "$max_seq_length" \
--num_train_epochs "$num_train_epochs" \
--per_device_train_batch_size "$batch_size" \
--lora_rank "$lora_rank" \
--load_lora "$load_lora" \
--learning_rate "$lr" \
--scheduler linear \
"$@" "${other_params[@]}"

