# pip install playwright datasets
# python -m playwright install

from datasets import Dataset
from playwright.sync_api import sync_playwright, TimeoutError as PlaywrightTimeoutError
import sys
import time
import argparse



def _wait_for_user_confirmation():
    ans = input(
        "\nLog in to qoj.ac in the opened Chromium window.\n"
        "Once logged in, type 'y' here and press Enter to continue: "
    ).strip().lower()
    if ans != "y":
        print("Aborted by user.")
        sys.exit(0)


def _ensure_submit_tab_and_fill(page, code_text: str):
    # Click the "Submit" tab
    page.locator('a.nav-link[href="#tab-submit-answer"]').first.click()

    # Fill code in the textarea
    textarea = page.locator("#input-answer_answer_editor")
    textarea.wait_for(state="visible", timeout=20000)
    textarea.fill("")
    textarea.fill(code_text)

    # Submit
    submit_button = page.locator("#button-submit-answer")
    submit_button.wait_for(state="visible", timeout=20000)
    submit_button.click()

    # Wait for redirect
    old_url = page.url
    try:
        page.wait_for_url(lambda url: url != old_url, timeout=45000)
    except PlaywrightTimeoutError:
        page.wait_for_load_state("networkidle", timeout=30000)


def _visit_problem_and_submit(page, link: str, code_text: str):
    if not link:
        print("  [skip] no link")
        return
    if not code_text:
        print("  [skip] no code")
        return

    print(f"Visiting: {link}")
    page.goto(link, wait_until="load")
    time.sleep(0.5)

    try:
        _ensure_submit_tab_and_fill(page, code_text)
        print("  ✓ submitted")
    except Exception as e:
        print(f"  [warn] submission failed: {e}")


def submit_dataset_codes(dataset: Dataset):
    if not {"link", "code"} <= set(dataset.column_names):
        raise ValueError("Dataset must have 'link' and 'code' columns")

    with sync_playwright() as p:
        browser = p.chromium.launch(headless=False)
        page = browser.new_page()
        page.goto("https://qoj.ac", wait_until="load")

        print("Opened qoj.ac. Please log in manually.")
        _wait_for_user_confirmation()

        for idx, row in enumerate(dataset):
            print(f"\nRow {idx+1}/{len(dataset)}")
            _visit_problem_and_submit(page, row["link"], row["code"])

        print("\nAll rows processed. Keeping browser open.")
        input("Press Enter here to close the browser when done...")
        browser.close()


if __name__ == "__main__":
    parser = argparse.ArgumentParser(description="Submit solutions from a dataset to qoj.ac")
    parser.add_argument(
        "--dataset",
        type=str,
        required=True,
        help="Path to dataset (Parquet file)",
    )
    args = parser.parse_args()

    dataset = Dataset.from_parquet(args.dataset)
    submit_dataset_codes(dataset)