from reportlab.pdfgen import canvas
from reportlab.lib.units import inch

fontsize = 12

def draw_aligned_text(c, text, y, x1, x2, x3, terminals):
    """
    Draw text with aligned sections for non-terminals, the ::=, and definitions.
    """
    if text[0]:  # Non-terminal or left-hand side
        c.setFont("Helvetica-Bold", fontsize)  # Font size for non-terminals
        c.drawString(x1, y, text[0])  # Non-terminal

    # The ::= symbol for the first part (only if there is a non-terminal)
    if text[0]:
        c.setFont("Helvetica", fontsize)
        c.drawString(x2, y, "::=")

    # The right-hand side (grammar definition)
    c.setFont("Courier", fontsize)  # Default font for definitions
    definition = text[1].split(" ")

    # Go through the definition and apply italic to terminals
    x_pos = x3
    for part in definition:
        # Check if the part contains a terminal (even if it's part of a composite term like var(Expr+))
        is_terminal = any(term in part for term in terminals)
        if is_terminal:
            c.setFont("Courier-Oblique", fontsize)  # Italic for terminals
        else:
            c.setFont("Courier", fontsize)  # Regular font for non-terminals
        c.drawString(x_pos, y, part)
        x_pos += c.stringWidth(part + " ", "Courier", fontsize)  # Add space after each part

def generate_dsl_grammar_pdf(output_path):
    # Define the content of the grammar
    terminals = ["TaskName", "RegionName", "var", "int"]

    grammar = [
        ("Program", "Statement+"),
        ("Statement",       "TaskMap | DataMap | DataLayout |"),
        ("",                "TaskModifier | FuncDef |"),
        ("",                "IndexTaskMap TaskName var |"),
        ("",                "SingleTaskMap TaskName var"),
        ("",                ""),
        ("TaskMap",         "Task TaskName Proc+"),
        ("DataMap",         "Region TaskName RegionName"),
        ("",                "Proc Memory+"),
        ("Proc",            "CPU | GPU | OMP | FPGA"),
        ("Memory",          "SYSMEM | FBMEM | ZCMEM |"),
        ("",                "SOCKMEM | RDMEM"),
        ("",                ""),
        ("DataLayout",      "Layout TaskName RegionName"),
        ("",                "Proc Constraint+"),
        ("Constraint",      "SOA | AOS | C_order | F_order |"),
        ("",                "Align == int"),
        ("",                ""),
        ("TaskModifier",    "GarbageCollect TaskName RegionName |"),
        ("",                "Backpressure TaskName int"),
        ("",                ""),
        ("FuncDef",         "def var(var+): FuncStmt+"),
        ("FuncStmt",        "var = Expr | return Expr"),
        ("",                ""),
        ("Expr",            "var | var(Expr+) | Machine(Proc) |"),
        ("",                "Expr.Expr | Expr Op Expr |"),
        ("",                "(Expr) | Expr[Expr] | *Expr"),
        ("",                "Expr ? Expr : Expr | Primitive"),
        ("Primitive",       "merge | split | slice | decompose")
    ]

    # Estimate the width and height required for the content
    line_height = 27 # Tighter row height
    content_width = 380  # Adjusted content width to be compact
    content_height = len(grammar) * line_height + 20
    print("grammar", content_height, content_width)

    # Create a canvas with a custom size matching the content size
    c = canvas.Canvas(output_path, pagesize=(content_width, content_height))

    # Positions for the aligned text
    left_margin = 5  # No margin
    x2 = left_margin + 80  # Position for the ::= symbol
    x3 = x2 + 20  # Position for the definitions

    # Coordinates for text placement
    y_position = content_height - 20  # Start near the top of the page

    # Loop through the grammar rules and print them with aligned ::= symbols
    for rule in grammar:
        draw_aligned_text(c, rule, y_position, left_margin, x2, x3, terminals)  # Apply formatting for all lines
        y_position -= line_height

    # Save the PDF without any extra white space
    c.showPage()
    c.save()

# Usage
output_pdf = "grammar.pdf"
generate_dsl_grammar_pdf(output_pdf)
