"""
Testing compilation success rate of DSL vs. C++
"""
import os
from os.path import join as pjoin
import json
import subprocess
import os
import dspy
from dspy.evaluate import Evaluate

from llm_mapper.coder import Coder
from llm_mapper.prompt_loader import dsl_documentation, dsl_demo_programs, cpp_demo_programs, cpp_documentation

# DSL doc uses 5,995 tokens

def get_current_file_directory():
    return os.path.dirname(os.path.abspath(__file__))


def get_parent_directory():
    return os.path.abspath(os.path.join(os.path.dirname(__file__), os.pardir))

def write_to_text_file(data, filename):
    with open(filename, "w") as f:
        f.write(data)


def load_conversion_history(exampleidx, trialidxmax):
    res_str = ""
    for trialidx in range(0, trialidxmax):
        gen_fn     = f"results/cpp_generated_solution_{exampleidx}_trial{trialidx}.txt"
        if not os.path.exists(gen_fn):
            break
        compile_fn = f"results/cpp_compilation_{exampleidx}_trial{trialidx}.txt"
        res_str += f"Conversion history for trial {trialidx}: \n"
        with open(gen_fn, 'r') as file:
            res_str += file.read()
        res_str += "\n The corresponding compilation error is: \n"
        with open(compile_fn, 'r') as file:
            res_str += file.read()
    return res_str

def load_dataset(dataset_type, trialidx=0, filepath=pjoin(get_parent_directory(), "dataset/exp1_compilation_succ/data.json5")):
    assert dataset_type in ["dsl", "cpp"]
    ds = sorted(json.load(open(filepath)), key=lambda x: x['id'])
    if dataset_type == "dsl":
        documentation = dsl_documentation
        demo_programs = dsl_demo_programs
    elif dataset_type == "cpp":
        documentation = cpp_documentation
        demo_programs = cpp_demo_programs
    else:
        raise ValueError("Unsupported dataset type")

    instruction = "Please generate the code according to the mapping strategy."\
                +  "Do not include the anything else (e.g., Code Documentation / Conv History / etc)."\
                + "Do not use mapping features that are not specified in the mapping strategy."\
                + "Please DIRECTLY generate the code directly WITHOUT wrapping the code with ``` or ```cpp. or ```plaintext "\
                + " because I will directly copy and paste your raw response as the code for compilation!"
    
    examples = [dspy.Example({
        "code_documentation": documentation,
        "code_examples": demo_programs,
        "strategy": r["strategy"],
        "starter_code": r["starter_code_dsl"] if dataset_type == "dsl" else r["starter_code_cpp"],
        "conv_history": load_conversion_history(r['id'], trialidx),
        "instruction": instruction,
        "solution": r['solution']}).with_inputs(
        "code_documentation", "code_examples",
        "strategy", "starter_code", "conv_history", "instruction") for r in ds]
    return examples

def generate_code(examples, lang):
    llm = dspy.OpenAI(model="gpt-4o-2024-08-06", max_tokens=16383)  # this is the longest context we can have
    dspy.settings.configure(lm=llm)
    coder = Coder()
    evaluate = Evaluate(devset=examples, num_threads=1, metric=lambda x, y: 1, return_outputs=True)
    score, results = evaluate(coder)
    generated_solutions = [('# ' + r[0].strategy + '\n' if lang == "dsl" else '// ' + r[0].strategy + '\n') +
                           # ('' if lang == "dsl" else '// ' + r[0].conv_history + '\n') + # for debugging
                           r[1].code
                           for r in results]
    return generated_solutions

def generate_dsl_programs():
    lang = "dsl"
    examples = load_dataset(lang)
    for i in range(0, 10):
        # avoid shared state between examples so each test case only contain one example
        example = [examples[i]]
        generated_solution = generate_code(example, lang)[0]
        write_to_text_file(generated_solution, f"results/dsl_generated_solution_{i}.txt")

def generate_cpp_programs(max_trial=10):
    lang = "cpp"
    compile_results = []  # To store the results for each example
    for example_idx in range(0, 10): # iterate over the examples
        last_successful_trial = None
        for trialidx in range(max_trial):
            examples = load_dataset(lang, trialidx)
            # avoid shared state between examples so each test case only contain one example
            example = [examples[example_idx]]
            generated_solution = generate_code(example, lang)[0]
            write_to_text_file(generated_solution, f"results/cpp_generated_solution_{example_idx}_trial{trialidx}.txt")
            
            if cpp_try_compile(example_idx, trialidx):  # Compilation passes
                last_successful_trial = trialidx  # Update the last successful trial index
                break  # Stop trying further trials once compilation passes
        
        # Report whether compilation was successful and the last trial index that worked
        if last_successful_trial is not None:
            compile_results.append((example_idx, last_successful_trial))
        else:
            compile_results.append((example_idx, f"No successful compilation after {max_trial} trials."))
    
    # Print the results for each example
    for example_idx, result in compile_results:
        if isinstance(result, int):
            print(f"Example {example_idx}: Compilation succeeded at trial index {result}.")
        else:
            print(f"Example {example_idx}: {result}.")

legion_runtime_dir='../../../legion/runtime'

def process_file(filename):
    # Open the file in read mode to read its contents
    with open(filename, 'r') as file:
        # Read all the lines from the file
        lines = file.readlines()

    # Open the same file in write mode to overwrite it with modified lines
    with open(filename, 'w') as file:
        # Iterate over each line from the file
        for line in lines:
            # Check if the line starts with '```'
            if line.strip().startswith('```'):
                # Comment out the line using C++ style comment (//)
                file.write(f'// {line}')
            else:
                # Write the line as is if it doesn't start with '```'
                file.write(line)

def cpp_try_compile(exampleidx, trialidx):
    gen_fn     = f"results/cpp_generated_solution_{exampleidx}_trial{trialidx}.txt"
    process_file(gen_fn)
    compile_fn = f"results/cpp_compilation_{exampleidx}_trial{trialidx}.txt"
    
    # Copy the generated solution to the target directory
    target_path = "../../app/circuit/src_test/circuit_mapper.cc"
    subprocess.run(["cp", gen_fn, target_path], check=True)
    
    # Change directory and run the compile script, capturing the output
    compile_dir = "../../app/circuit"
    compile_output = f"../../llm_mapper/experiments/" + compile_fn
    compile_command = f"./compile2.sh src_test/ |& tee {compile_output}"
    
    # Set the environment variable
    env_vars = os.environ.copy()  # Copy the current environment variables
    env_vars["LG_RT_DIR"] = legion_runtime_dir

    # Run the compilation command
    subprocess.run(compile_command, shell=True, executable='/bin/bash', cwd=compile_dir, env=env_vars, check=True)
    
    # Check the compilation result for errors
    error_found = False
    lines_to_keep = []
    
    with open(compile_fn, 'r') as file:
        for line in file:
            lines_to_keep.append(line)
            if "Error" in line or "error" in line:
                error_found = True
            if "language/src/std/launcher.rg" in line:
                lines_to_keep = lines_to_keep[:-1]
                break

    if error_found:
        print(f"Compilation fails for example {exampleidx}, trial {trialidx}.")
        
        # Trim the file content if "Error" was found
        with open(compile_fn, 'w') as file:
            file.writelines(lines_to_keep)  # Write the trimmed lines back to the file
        return False
    else:
        print(f"Compilation passes for example {exampleidx}, trial {trialidx}.")
        return True


if __name__ == '__main__':
    generate_cpp_programs()
    generate_dsl_programs()

'''
Example 0: Compilation succeeded at trial index 0.
Example 1: No successful compilation after 10 trials..
Example 2: No successful compilation after 10 trials..
Example 3: Compilation succeeded at trial index 0.
Example 4: Compilation succeeded at trial index 4.
Example 5: Compilation succeeded at trial index 1.
Example 6: Compilation succeeded at trial index 0.
Example 7: Compilation succeeded at trial index 0.
Example 8: Compilation succeeded at trial index 1.
Example 9: Compilation succeeded at trial index 1.
'''