import autogen
import opto.trace as trace
from trace_mapper.trace_template import DSLMapperGenerator
import json
import os
import subprocess
import random

def create_dsl_mapper(app_info, iter):
    app = app_info['application']
    # Initialize the DSLMapperGenerator object
    mapper = DSLMapperGenerator(tasks=app_info['tasks'], regions=app_info['regions'], 
                                index_tasks=app_info['index_tasks'], single_tasks=app_info['single_tasks'], 
                                index_task_specification=app_info['index_task_specification'])

    # Generate the mapping program
    program = mapper.generate_mapper()

    # Extract the program data (string representation)
    program_data = program.data

    # Save the program to a mapping file
    mapping_file = f"result/{app}/r_mapping{iter}"
    with open(mapping_file, 'w') as f:
        f.write(program_data)


def copy_mapping_and_run(app_info, iter):
    app = app_info['application']

    # Determine the target path based on the application name
    if app in ["cannon", "pumma", "summa", "solomonik", "johnson", "cosma"]:
        target_path = f"../../app/gemm/{app}/bin_trace"
    else:
        assert app in ["circuit", "stencil", "pennant"]
        target_path = f"../../app/{app}/bin_trace"

    mapping_file = f"result/{app}/r_mapping{iter}"

    # Copy the mapping file to the target path
    subprocess.run(["cp", mapping_file, target_path], check=True)

    # Construct the run command
    run_command = f"python3 run.py sapling --conf {app_info['config']} --mapping {iter} --repeat -1"
    subprocess.run(run_command, shell=True, executable='/bin/bash', cwd=target_path, check=True)

    # Copy the log file
    log_file = target_path + f"/log_conf{app_info['config']}_repeatr_mapping{iter}.log"
    subprocess.run(["cp", log_file, f"result/{app}"], check=True)

def run_for_application(application_name, config_name, num_iterations=10):
    # Load application-specific info from a JSON file
    with open("app_config.json", "r") as f:
        json_dict = json.load(f)

    # Extract the specific application information
    app_info = {
        'application': application_name,
        'config': json_dict[application_name][config_name],  # Use the config_name passed to the function
        'tasks': json_dict[application_name]['tasks'],
        'regions': json_dict[application_name]['regions'],
        'index_tasks': json_dict[application_name]['index_tasks'],
        'single_tasks': json_dict[application_name]['single_tasks'],
        'index_task_specification': json_dict[application_name].get('index_task_specification', "")
    }

    # Iteratively generate and execute mapping for the application
    for iter in range(num_iterations):
        create_dsl_mapper(app_info, iter)
        copy_mapping_and_run(app_info, iter)

if __name__ == '__main__':
    random.seed(42)
    run_for_application("circuit", "config0")
    run_for_application("stencil", "config0")
    run_for_application("pennant", "config0")
    run_for_application("cannon", "config0")
    run_for_application("pumma", "config0")
    run_for_application("summa", "config0")
    run_for_application("solomonik", "config0")
    run_for_application("johnson", "config0")
    run_for_application("cosma", "config0")