import subprocess
import re

expert_feedback = {
    "stride does not match": "The layout constraints are not satisfied. Please try to adjust the layout constraints or move tasks to different processor types (to satisfy a different layout constraint).",
    "Execution time": "Now the mapping is valid. Please try to reduce the execution time (e.g., moving more tasks to GPU, moving more regions to GPU FBMEM, or avoid using some InstanceLimit statements).",
    "srun: got SIGCONT": "The program takes too much time to run, and it is killed by the timeout. Please try to reduce the elapsed time by moving more tasks to GPU (generate fewer Task statements to let them fallback to GPU) or avoid using some InstanceLimit statements.",
    "SingleTaskMap's function undefined": "You will need to define the function used in SingleTaskMap first before applying it to some task. Please re-read the docstring for single_task_map_decision.",
    "IndexTaskMap's function undefined": "You will need to define the function used in IndexTaskMap first before applying it to some task. Please do include the statement code_statements.append(chosen_function_definition) in index_task_map_decision. Also, do not modify how function_name is extracted: function_name = chosen_function_definition.split('(')[0].strip().split()[-1] so that the name of the function definintion and usage will match.",
    "report_profiling": "Maybe do NOT generate InstanceLimit statements for this application.",
    "syntax error": "Typically the syntax happens in how you define functions. Reminder about syntax: generally code comments start with # and each statement should end with ; Never use // for anything. Also, in the function body, please never use a variable that is not defined before. E.g., please first define mgpu=Machine(GPU); before using it in the function. Please carefully inspect all variables used in the function are first defined. No if-else statements are allowed in the generated code. Lastly, function defintion should look like this: def func_name(Task task) { function_body here } and note that the Task type must be there. Inside function body used by IndexTaskMap, please use task.ipoint",
    "On entry to DGEMM": "The layout constraint is wrong. Please adjust the layout constraint.",
    "achieved GFLOPS": "This suggests that the Task, Region, Layout statements are good. Please take a look at the generated mapping and fix those generated code so that it's not randomized decision any more. Now you want to use more or fewer or randomly sample different IndexTaskMap or SingleTaskMap statements to maximize the throughput.",
    "mgpu not found": "Please include the definition of mgpu in the generated code, i.e., mgpu = Machine(GPU);",
    "but machine model is for": "When using the IndexTaskMap or SingleTaskMap, please make sure that the corresponding type of machine model (GPU/OMP/CPU) is correctly used in the return statement of the mapping function.",
    "unexpected '}', expecting ',' or ']'": "First, please do not do complex arithmetic in the function body of the same_point function. same_point should always return m_2d[*task.parent.processor(m_2d)]; Second, do NOT use //, please use / for division instead. E.g.. // 4 should be replaced with / 4. Also, never use // for comments, comments start with # instead.",
    "syntax error, unexpected ':', expecting '{'": "Please define functions like this: def func_name(Task task) { function_body here } and there is no : in the function definition.",
    "unexpected ':'": "There should be no `:` in the generated program. Please use {} instead of `:`",
    "syntax error, unexpected T_Identifier, expecting ')'": "Please define functions like this: def func_name(Task task) { function_body here } and note that the Task type must be there.",
    "currently SingleTask does not support get_point()": "Please do not do complex arithmetic in the function body of the same_point function. same_point should return m_2d[*task.parent.processor(m_2d)];",
    "(Z_COPY_MEM) for processor": "Please change the placement of regions. E.g., please more regions onto GPU FBMEM.",
    "(GPU_FB_MEM) for processor": "One reason could be that you put too much regions on GPU, please avoid writing too much Region statements that place data onto FBMEM. Another approach could be to use more or fewer or randomly sample different the IndexTaskMap or SingleTaskMap statements.",
    "Slice processor index out of bound": "For the functions used by IndexTaskMap statements, please note that the first index of mgpu should end with % mgpu.size[0], and the second element should end with %mgpu.size[1]. E.g, return mgpu[.... % mgpu.size[0], ... % mgpu.size[1]]",
    "get_node_proc(const std::vector<int>&)": "For the functions used by IndexTaskMap statements, please note that the first index of mgpu should end with % mgpu.size[0], and the second element should end with %mgpu.size[1]",
    "min not found": "Please do not use the min. The min not a keyword in the DSL. Please use a < b ? a : b instead of min(a, b).",
    "does not support get_parent()": "In the function used by IndexTaskMap, please do not use task.parent.processor.",
}

non_directional_feedback = {
    "stride does not match": "The memory layout of the data does not match the application's expected stride.",
    "srun: got SIGCONT": "The program takes too much time to run, and it is killed by the timeout.",
    "report_profiling": "The InstanceLimit statements cause the issue in the application.",
    "On entry to DGEMM": "The memory layout is not expected by the application.",
    "Slice processor index out of bound": "The error is caused by the IndexTaskMap statements.",
    "get_node_proc(const std::vector<int>&)": "The error is caused by the IndexTaskMap statements",
    "does not support get_parent()": "The error is caused by the IndexTaskMap statements",
}

def analyze_syntax_error(mapping_content, log_content):
    # Check if "Error at line" exists in log_content
    if "Error at line" in log_content:
        # Extract the line number using a regular expression
        match = re.search(r"Error at line (\d+)", log_content)
        if match:
            line_number = int(match.group(1))  # Extract the line number
            # Split the mapping_content into lines
            mapping_lines = mapping_content.splitlines()
            # Check if the line_number is within the bounds of the mapping file
            if 0 < line_number <= len(mapping_lines):
                # Return the corresponding line
                return f"The line that causes the syntax error in the generated file is {mapping_lines[line_number - 1]} To fix this, you need to modify either this line or its related / nearby lines."
    return ""

class TestMapper:
    def __init__(self, program, app_info, iter, repeat_idx, mode=0):
        self.program = program
        self.iter = iter
        self.repeat_idx = repeat_idx
        self.application_name = app_info['application']
        self.config = app_info['config']
        self.tasks = app_info['tasks']
        self.regions = app_info['regions']
        self.mapping_file = f"result/{self.application_name}/{self.repeat_idx}_mapping{self.iter}"
        self.mode = mode
        if self.mode == 1: # OPRO
            self.mapping_file = f"result/{self.application_name}/o{self.repeat_idx}_mapping{self.iter}"
        elif self.mode == 2: # only performance feedback
            self.mapping_file = f"result/{self.application_name}/p{self.repeat_idx}_mapping{self.iter}"
        elif self.mode == 3: # only non-directional feedback
            self.mapping_file = f"result/{self.application_name}/n{self.repeat_idx}_mapping{self.iter}"
        else:
            assert self.mode == 0
        
        # Save the program to a mapping file
        with open(self.mapping_file, 'w') as f:
            f.write(self.program)

    def gen_feedback(self):
        if self.application_name in ["cannon", "pumma", "summa", "solomonik", "johnson", "cosma"]:
            target_path = f"../../app/gemm/{self.application_name}/bin_trace"
        else:
            assert self.application_name in ["circuit", "stencil", "pennant"]
            target_path = f"../../app/{self.application_name}/bin_trace"
        subprocess.run(["cp", self.mapping_file, target_path], check=True)

        run_command = f"python3 run.py sapling --conf {self.config} --mapping {self.iter} --repeat {self.repeat_idx}"
        if self.mode != 0:
            run_command += f" --mode {self.mode}"
        subprocess.run(run_command, shell=True, executable='/bin/bash', cwd=target_path, check=True)

        if self.mode == 0:
            repeat_str = f'{self.repeat_idx}'
        elif self.mode == 1:
            repeat_str = f'o{self.repeat_idx}'
        elif self.mode == 2:
            repeat_str = f'p{self.repeat_idx}'
        elif self.mode == 3:
            repeat_str = f'n{self.repeat_idx}'
        else:
            raise ValueError(f"Invalid mode {self.mode}")

        log_file = target_path + f"/log_conf{self.config}_repeat{repeat_str}_mapping{self.iter}.log"
        subprocess.run(["cp", log_file, f"result/{self.application_name}"], check=True)

        with open(log_file, 'r') as f:
            log_content = f.read()
        
        with open(self.mapping_file, 'r') as f:
            mapping_content = f.read()
        
        expert_designed_feedback = "\n".join(
            [expert_feedback[key] for key in expert_feedback if key in log_content]
            + [analyze_syntax_error(mapping_content, log_content)]
        )
        if self.mode == 2: # only performance feedback
            expert_designed_feedback = "Please pay attention to performance numbers in the execution log and try to improve it. If there is an error, please fix it first."
        elif self.mode == 3: # only non-directional feedback
            expert_designed_feedback = "\n".join(
            [non_directional_feedback[key] for key in non_directional_feedback if key in log_content]
        )
        deliminator = "\n-------------------------------------\n"
        feedback = "The execution log file (neglect most of the content if there is error while expert feedback exists, please pay attention to the Execution time and the achieved GFLOPS representing throughput if exists):\n" + log_content + deliminator \
                    + "The generated mapping file:\n" + mapping_content + deliminator \
                    + "Expert feedback for writing the next mapper (if exists, please pay most attention to it):\n" + expert_designed_feedback
        print(feedback)
        print("=====================================")
        return feedback
