#!/usr/bin/env python3

import argparse
import subprocess
import os
import glob

def execute_command(command, env):
    command_str = " ".join(command)
    print(f"Executing command: {command_str}", flush=True)
    
    result = subprocess.run(command_str, executable='/bin/bash', shell=True, capture_output=True, env=env)
    
    print(result.stdout.decode(), flush=True)
    print(result.stderr.decode(), flush=True)

def get_header(supercomputer, nodes, timeout):
    if supercomputer == "lassen":
        return ["timeout", f"{timeout}", "jsrun", "-b", "none", "-c", "ALL_CPUS", "-g", "ALL_GPUS", "-r", "1", "-n", f"{nodes}"]
    else:
        return ["timeout", f"{timeout}", "srun", "-n", f"{nodes}", "-N", f"{nodes}", "-c", "40", "-G", f"{nodes*4}", "--nodelist=$(hostname | cut -d'.' -f1)"]

def setup_config(supercomputer, config_index):
    # Example configurations based on supercomputer and config index
    config_map = {
        "lassen": [
            {"npp": 4, "wpp": 8, "p": 2, "pps": 3},
            {"npp": 6, "wpp": 10, "p": 3, "pps": 4}
        ],
        "sapling": [
            {"nodes": 1, "npp": 1000, "wpp": 5000, "p": 40, "pps": 10},
            {"nodes": 1, "npp": 1000, "wpp": 1000, "p": 20, "pps": 5}
        ]
    }
    
    # Retrieve the configuration for the supercomputer and config index
    configs = config_map[supercomputer]
    if config_index < 0 or config_index >= len(configs):
        raise ValueError(f"Invalid config index {config_index} for supercomputer {supercomputer}")
    
    return configs[config_index]

def main():
    parser = argparse.ArgumentParser(description="Run circuit with specified parameters.")
    parser.add_argument("supercomputer", choices=["lassen", "sapling"], help="Supercomputer name")
    parser.add_argument("--conf", type=int, required=True, help="Configuration index")
    parser.add_argument("--noperf", action='store_true', help="Turn off performance profiling")
    parser.add_argument("--mapping", type=int, required=True, help="Specify mapping file number")
    parser.add_argument("--repeat", type=int, required=True, help="Specify repeat number")
    parser.add_argument("--mode", type=int, default=0, help="Specify mode number")
    parser.add_argument("--timeout", default=20, type=int, help="Specify time limit in seconds")

    args = parser.parse_args()

    config = setup_config(args.supercomputer, args.conf)

    # Now you have the dynamic configuration
    nodes = config["nodes"]
    npp = config['npp'] # Number of nodes per piece
    wpp = config['wpp'] # Number of wires per piece
    p = config['p'] # Number of pieces per (machine) node
    pps = config['pps'] # Piece per superpiece

    mapping_file = f"{'r' if args.repeat == -1 else args.repeat}_mapping{args.mapping}"
    if args.mapping == -1: # expert mapping
        mapping_file = "dmapping"

    if args.mode == 1: # OPRO mode
        mapping_file = f"o{args.repeat}_mapping{args.mapping}"
    elif args.mode == 2: # only performance feedback
        mapping_file = f"p{args.repeat}_mapping{args.mapping}"
    elif args.mode == 3: # non-directional feedback
        mapping_file = f"n{args.repeat}_mapping{args.mapping}"
    else:
        assert args.mode == 0
    # if repeat == -1, random mapping

    base_command = [
        "circuit",
        f"-npp {npp}",
        f"-wpp {wpp}",
        "-l 10",
        f"-p {p * nodes}",
        f"-pps {pps}",
        "-prune 30",
        "-ll:util 2",
        "-ll:bgwork 2",
        "-hl:sched 1024",
        "-ll:gpu 4",
        "-ll:cpu 4",
        "-ll:csize 150000",
        "-ll:fsize 15000",
        "-ll:zsize 2048",
        "-ll:rsize 512",
        "-ll:gsize 0",
        "-level runtime=5",
        f"-mapping {mapping_file}",
        "-tm:use_semantic_name",
    ]

    if not args.noperf:
        base_command.extend([
            f"-lg:prof {nodes}",
            f"-lg:prof_logfile prof_circuit_conf{args.conf}_repeat{mapping_file}_%.gz",
        ])
    
    base_command.extend([
        f"|& tee log_conf{args.conf}_repeat{mapping_file}.log"
    ])

    command = get_header(args.supercomputer, nodes, args.timeout) + base_command

    env = os.environ.copy()
    env["LD_LIBRARY_PATH"] = "."
    execute_command(command, env)

if __name__ == "__main__":
    main()
