#!/bin/bash

# Define app groups
all_apps=("circuit" "stencil" "pennant" "cannon" "pumma" "summa" "solomonik" "johnson" "cosma")
gemm_apps=("cannon" "pumma" "summa" "solomonik" "johnson" "cosma")
sci_apps=("circuit" "stencil" "pennant")

# Function to clean files for sci app directories with optional repeat number
clean_sci_app() {
    dir=$1
    repeat_number=$2
    
    if [ -d "$dir/bin_trace" ]; then
        # If a repeat number is specified, clean only files matching that number
        if [ -n "$repeat_number" ]; then
            rm $dir/bin_trace/${repeat_number}_mapping*
            rm $dir/bin_trace/log_conf*_repeat${repeat_number}_mapping*.log
        else
            # Clean all files with digits in the repeat number slot
            rm $dir/bin_trace/*[0-9]_mapping*
            rm $dir/bin_trace/log_conf*_repeat[0-9]*_mapping*.log
        fi
        rm $dir/bin_trace/*.gz
        echo "Cleaned files in $dir/bin_trace directory."
    else
        echo "Directory $dir/bin_trace does not exist."
    fi
}

# Function to clean files for gemm app directories with optional repeat number
clean_gemm_app() {
    dir=$1
    repeat_number=$2
    
    if [ -d "gemm/$dir/bin_trace" ]; then
        # If a repeat number is specified, clean only files matching that number
        if [ -n "$repeat_number" ]; then
            rm gemm/$dir/bin_trace/${repeat_number}_mapping*
            rm gemm/$dir/bin_trace/log_conf*_repeat${repeat_number}_mapping*.log
        else
            # Clean all files with digits in the repeat number slot
            rm gemm/$dir/bin_trace/*[0-9]_mapping*
            rm gemm/$dir/bin_trace/log_conf*_repeat[0-9]*_mapping*.log
        fi
        rm gemm/$dir/bin_trace/*.gz
        echo "Cleaned files in gemm/$dir/bin_trace directory."
    else
        echo "Directory gemm/$dir/bin_trace does not exist."
    fi
}

# Check if an argument is provided
if [ -z "$1" ]; then
    echo "Usage: ./clean.sh <directory|group> [repeat_number]"
    echo "Available groups: all, gemm, sci"
    exit 1
fi

# Assign the optional repeat number argument
repeat_number=$2

# Check which argument is provided and act accordingly
case $1 in
    "all")
        for app in "${all_apps[@]}"; do
            if [[ " ${gemm_apps[@]} " =~ " $app " ]]; then
                clean_gemm_app "$app" "$repeat_number"
            else
                clean_sci_app "$app" "$repeat_number"
            fi
        done
        ;;
    "gemm")
        for app in "${gemm_apps[@]}"; do
            clean_gemm_app "$app" "$repeat_number"
        done
        ;;
    "sci")
        for app in "${sci_apps[@]}"; do
            clean_sci_app "$app" "$repeat_number"
        done
        ;;
    *)
        # If not a group, assume it's an individual directory
        if [[ " ${gemm_apps[@]} " =~ " $1 " ]]; then
            clean_gemm_app "$1" "$repeat_number"
        else
            clean_sci_app "$1" "$repeat_number"
        fi
        ;;
esac
