#!/usr/bin/env python3

import argparse
import subprocess
import os

def execute_command(command, env):
    command_str = " ".join(command)
    print(f"Executing command: {command_str}", flush=True)
    
    result = subprocess.run(command_str, executable='/bin/bash', shell=True, capture_output=True, env=env)
    
    print(result.stdout.decode(), flush=True)
    print(result.stderr.decode(), flush=True)

def get_header(supercomputer, nodes, timeout):
    if supercomputer == "lassen":
        return ["timeout", f"{timeout}", "jsrun", "-b", "none", "-c", "ALL_CPUS", "-g", "ALL_GPUS", "-r", "1", "-n", f"{nodes}"]
    else:
        return [
            "timeout", f"{timeout}", 
            "srun", "-n", "4",    # 4 separate tasks (1 per runtime)
            "-N", f"{nodes}",      # Still 1 node
            "-c", "10",            # 10 CPUs per task
            "--gres=gpu:1",        # 1 GPU per task
            "--cpu_bind=cores",    # Bind to CPU cores
            # "-G", "4",             # 1 GPU per task
            "--nodelist=$(hostname | cut -d'.' -f1)"  # Binding to the node
        ]

def setup_config(supercomputer, config_index):
    # Updated configurations with gx, gy, gz, size, and nodes included
    config_map = {
        "lassen": [
            {"gx": 2, "gy": 1, "gz": 1, "size": 512, "nodes": 16},
            {"gx": 4, "gy": 2, "gz": 2, "size": 1024, "nodes": 64}
        ],
        "sapling": [
            {"gx": 2, "gy": 1, "gz": 1, "size": 8000, "nodes": 1},
            {"gx": 2, "gy": 1, "gz": 1, "size": 20000, "nodes": 1}
        ]
    }
    
    # Retrieve the configuration for the supercomputer and config index
    configs = config_map[supercomputer]
    if config_index < 0 or config_index >= len(configs):
        raise ValueError(f"Invalid config index {config_index} for supercomputer {supercomputer}")
    
    return configs[config_index]

def main():
    parser = argparse.ArgumentParser(description="Run Cosma benchmark with specified parameters.")
    parser.add_argument("supercomputer", choices=["lassen", "sapling"], help="Supercomputer name")
    parser.add_argument("--conf", type=int, required=True, help="Configuration index")
    parser.add_argument("--noperf", action='store_true', help="Turn off performance profiling")
    parser.add_argument("--mapping", type=int, required=True, help="Specify mapping file number")
    parser.add_argument("--repeat", type=int, required=True, help="Specify repeat number")
    parser.add_argument("--mode", type=int, default=0, help="Specify mode number")
    parser.add_argument("--timeout", default=20, type=int, help="Specify time limit in seconds")

    args = parser.parse_args()

    config = setup_config(args.supercomputer, args.conf)

    # Dynamic configuration from config_map
    gx = config["gx"]
    gy = config["gy"]
    gz = config["gz"]
    size = config["size"]
    nodes = config["nodes"]

    mapping_file = f"{'r' if args.repeat == -1 else args.repeat}_mapping{args.mapping}" if args.mapping != -1 else "dmapping"

    if args.mode == 1: # OPRO mode
        mapping_file = f"o{args.repeat}_mapping{args.mapping}"
    elif args.mode == 2: # only performance feedback
        mapping_file = f"p{args.repeat}_mapping{args.mapping}"
    elif args.mode == 3: # non-directional feedback
        mapping_file = f"n{args.repeat}_mapping{args.mapping}"
    else:
        assert args.mode == 0
    # if repeat == -1, random mapping

    base_command = [
        "main",
        f"-n {size}",
        f"-gx {gx}",
        f"-gy {gy}",
        f"-gz {gz}",
        "-dm:exact_region",
        "-tm:untrack_valid_regions",
        "-ll:ocpu", "1",
        "-ll:othr", "1",
        "-ll:csize", "15000",
        "-ll:util", "1",
        "-dm:replicate", "1",
        "-ll:gpu", "1",
        "-ll:fsize", "3500",
        "-ll:bgwork", "2",
        "-ll:bgnumapin", "1",
        f"-mapping {mapping_file}"
    ]

    if not args.noperf:
        base_command.extend([
            f"-lg:prof {nodes*4}",
            f"-lg:prof_logfile prof_cosma_conf{args.conf}_repeat{mapping_file}_%.gz"
        ])

    base_command.extend([f"|& tee log_conf{args.conf}_repeat{mapping_file}.log"])

    command = get_header(args.supercomputer, nodes, args.timeout) + base_command

    env = os.environ.copy()
    env["LD_LIBRARY_PATH"] = "."
    execute_command(command, env)

if __name__ == "__main__":
    main()
