#!/usr/bin/env python3

import argparse
import subprocess
import os

def execute_command(command, env):
    command_str = " ".join(command)
    print(f"Executing command: {command_str}", flush=True)
    
    result = subprocess.run(command_str, executable='/bin/bash', shell=True, capture_output=True, env=env)
    
    print(result.stdout.decode(), flush=True)
    print(result.stderr.decode(), flush=True)

def get_header(supercomputer, nodes, timeout):
    if supercomputer == "lassen":
        return ["jsrun", "-b", "none", "-c", "ALL_CPUS", "-g", "ALL_GPUS", "-r", "1", "-n", f"{nodes}"]
    else:
        return ["timeout", f"{timeout}", "srun", "-n", f"{nodes}", "-N", f"{nodes}", "-c", "40", "-G", f"{nodes*4}", "--nodelist=$(hostname | cut -d'.' -f1)"]

def setup_config(supercomputer, config_index):
    # Updated configurations based on supercomputer and config index, with nodes included
    config_map = {
        "lassen": [
            {"input_file": "../input/sedovbig4x30/sedovbig.pnt", "nodes": 1},
            {"input_file": "../input/other_case/other_case.pnt", "nodes": 1}
        ],
        "sapling": [
            {"input_file": "../input/sedovbig4x30/sedovbig.pnt", "nodes": 1},
            {"input_file": "../input/medium_case/medium_case.pnt", "nodes": 1}
        ]
    }
    
    # Retrieve the configuration for the supercomputer and config index
    configs = config_map[supercomputer]
    if config_index < 0 or config_index >= len(configs):
        raise ValueError(f"Invalid config index {config_index} for supercomputer {supercomputer}")
    
    return configs[config_index]

def main():
    parser = argparse.ArgumentParser(description="Run pennant with specified parameters.")
    parser.add_argument("supercomputer", choices=["lassen", "sapling"], help="Supercomputer name")
    parser.add_argument("--conf", type=int, required=True, help="Configuration index")
    parser.add_argument("--noperf", action='store_true', help="Turn off performance profiling")
    parser.add_argument("--mapping", type=int, required=True, help="Specify mapping file number")
    parser.add_argument("--repeat", type=int, required=True, help="Specify repeat number")
    parser.add_argument("--mode", type=int, default=0, help="Specify mode number")
    parser.add_argument("--timeout", default=20, type=int, help="Specify time limit in seconds")

    args = parser.parse_args()

    config = setup_config(args.supercomputer, args.conf)
    input_file = config['input_file']
    nodes = config['nodes']

    numgpus = 4  # Set the number of GPUs to 4
    npieces = nodes * numgpus

    # Determine the mapping file based on the --mapping argument
    if args.mapping == -1:
        mapping_file = "dmapping"
    else:
        mapping_file = f"{'r' if args.repeat == -1 else args.repeat}_mapping{args.mapping}"

    if args.mode == 1: # OPRO mode
        mapping_file = f"o{args.repeat}_mapping{args.mapping}"
    elif args.mode == 2: # only performance feedback
        mapping_file = f"p{args.repeat}_mapping{args.mapping}"
    elif args.mode == 3: # non-directional feedback
        mapping_file = f"n{args.repeat}_mapping{args.mapping}"
    else:
        assert args.mode == 0
    # if repeat == -1, random mapping

    base_command = [
        "pennant",
        input_file,
        f"-npieces {npieces}",
        "-numpcx 1",
        f"-numpcy {npieces}",
        "-seq_init 0",
        "-par_init 1",
        "-prune 30",
        "-hl:sched 1024",
        f"-ll:gpu {numgpus}",
        "-ll:util 2",
        "-ll:bgwork 2",
        "-ll:csize 150000",
        "-ll:fsize 15000",
        "-ll:zsize 5000",
        "-ll:rsize 512",
        "-ll:gsize 0",
        f"-mapping {mapping_file}",
        "-tm:use_semantic_name",
    ]

    if not args.noperf:
        base_command.extend([
            f"-lg:prof {nodes}",
            f"-lg:prof_logfile prof_pennant_conf{args.conf}_repeat{mapping_file}_%.gz",
            f"-level runtime=5"
        ])
    
    base_command.extend([
        f"|& tee log_conf{args.conf}_repeat{mapping_file}.log"
    ])

    command = get_header(args.supercomputer, nodes, args.timeout) + base_command

    env = os.environ.copy()
    env["LD_LIBRARY_PATH"] = "."
    execute_command(command, env)

if __name__ == "__main__":
    main()
