import itertools
import json
import sys
import threading
import time
from multiprocessing import Pool
from pathlib import Path

import numpy as np
from tqdm import tqdm


def get_predictions(probabilities_id):
    predictions_path = Path("predictions") / probabilities_id
    if not predictions_path.exists():
        raise FileNotFoundError(f"Predictions file {predictions_path} does not exist.")

    all_softmax = np.load(predictions_path / "model_predictions.npy")
    all_labels = np.load(predictions_path / "labels.npy")
    all_embeddings = np.load(predictions_path / "embeddings.npy")
    with open(predictions_path / "metadata.json", "r") as f:
        metadata = json.load(f)
    dataset = metadata["dataset"]

    return all_softmax, all_labels, all_embeddings, dataset


def dict_to_list(leaf_nodes):
    return [leaf_nodes[key] for key in leaf_nodes.keys()]


def profile_and_animate(func):
    def wrapper(*args, **kwargs):
        start = time.time()
        running = True

        def blink_message():
            for c in itertools.cycle(["|", "/", "-", "\\"]):
                if not running:
                    break
                sys.stdout.write(
                    f"\rComputing non-conformity scores. This process may take a few minutes, please wait {c}"
                )
                sys.stdout.flush()
                time.sleep(0.2)
            sys.stdout.write("\rComputation completed!                                \n")

        t = threading.Thread(target=blink_message)
        t.start()

        result = func(*args, **kwargs)

        running = False
        t.join()

        end = time.time()
        print(f"Time elapsed: {end - start:.2f} seconds")
        return result

    return wrapper


def compute_parallel_process(function, iterable, paralel=True):
    if not paralel:
        predictions_set = []
        for i in tqdm(iterable):
            output = function(i)
            predictions_set.append(output)
    else:
        with Pool() as pool:
            predictions_set = pool.map(function, iterable)

    return predictions_set


class TreeNode:
    def __init__(self, value=0, children=None):
        self.value = value
        self.children = children if children is not None else []
