# Copyright (c) 2023-2024 DeepSeek.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

import os
import time
import json
from pathlib import Path
import random

import torch
import hydra
import logging
import numpy as np
from PIL import Image
from tqdm import trange
from rich.theme import Theme
from rich.console import Console
from rich.logging import RichHandler
from omegaconf import DictConfig, OmegaConf
from torchvision.utils import make_grid
import torchvision.transforms as T
from torchvision.transforms.functional import InterpolationMode
from transformers import AutoModelForCausalLM
from transformers import AutoProcessor, AutoModelForCausalLM
from huggingface_hub import login

from openai import OpenAI
from io import BytesIO
import base64
from typing import List, Optional, Tuple

from utils import set_seed, load_metadata
import decode
from tokenize_embed import *

from generator.base_generator import T2IResult, T2IRequest, BaseGenerator
from generator.janus_generator import JanusGenerator
from generator.llamagen_generator import LlamaGenGenerator

Generator_Dict = {
    "Janus-Pro-1B": JanusGenerator,
    "Janus-Pro-7B": JanusGenerator,
    "LlamaGen": LlamaGenGenerator
}

Tokenizer_Dict = {
    "Janus-Pro-1B": tokenize_text_janus,
    "Janus-Pro-7B": tokenize_text_janus,
    "LlamaGen": tokenize_text_llamagen,
}
Tokenizer_3_way_Dict = {
    "Janus-Pro-1B": tokenize_text_janus_3_way,
    "Janus-Pro-7B": tokenize_text_janus_3_way,
    "LlamaGen": tokenize_text_llamagen_3_way
}

FORMAT = "%(message)s"

IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)

GPT_SELECT_FIRST_QUARTERS_PROMPT = """You are given a single image consisting of 4 contiguous horizontal quarters (from top to bottom: quarter 1, quarter 2, quarter 3, quarter 4).
Each quarter shows the top quarter (upper 1/4 crop) of a different full image generated from the same text prompt. The lower three-quarters of each full image are not shown in this composite. Since only the top part is visible, some quarters may show only background without any objects, while in other cases objects may appear only partially, with the rest continuing into the unseen lower part of the image.

The text prompt is: "{}".

For each of the 4 quarters, answer strictly with either "possible" or "impossible" (lowercase, no punctuation).
Output must contain exactly 4 words, separated by commas, in order: quarter 1, quarter 2, quarter 3, and quarter 4.
Example format: possible, impossible, possible, impossible

Focus on the required attributes (e.g., color, shape, counts, spatial relations) of objects in the prompt.

Say "impossible" for a quarter only if it is certain that the prompt cannot be satisfied:
- the visible part already makes it clear that the prompt cannot be satisfied (e.g., too many objects are already drawn, or or an object has the wrong color or an incorrect shape), OR
- even if the lower three-quarters of that full image (not shown) were completed naturally, the final image would still not match the required attributes.

If there is any reasonable way the prompt could still be satisfied, say "possible".
"""

GPT_SELECT_SECOND_QUARTERS_PROMPT = """You are given a single image consisting of 2 contiguous halves (from top to bottom: half 1, half 2). 
Each half shows the top half of a different full image generated from the same text prompt. The bottom halves of those full images are not shown. 
Some objects may appear only partially (for example, the top half of an object is visible, and the bottom half would appear if the image is completed).

The text prompt is: "{}".

For each of the 2 halves, answer strictly with either "possible" or "impossible" (lowercase, no punctuation). 
Output must contain exactly 2 words, separated by a comma, in order: half 1, half 2. 
Example format: possible,impossible

Focus on the required attributes (e.g., color, shape, counts, spatial relations) of objects in the prompt.

Say "impossible" for a half only if it is certain that the prompt cannot be satisfied:
- the visible part already makes it clear that the prompt cannot be satisfied (e.g., too many objects are already drawn,or an object has the wrong color or an incorrect shape), OR
- even if the hidden bottom half of that full image were completed naturally, the final image would still not match the required attributes.

Otherwise, say "possible".
"""

GPT_SELECT_THIRD_QUARTERS_PROMT = """You are given a partially drawn image: the upper three quarters are drawn, and the bottom one quarter is padding. 
Some objects may appear only partially (for example, the top three quarters of an object are visible, and the bottom part would appear if the image is completed).
Answer strictly with "possible" or "impossible" (lowercase, no punctuation).

The text prompt is: "{}".

Focus on the required number of objects in the prompt.

Say "impossible" if:
- the visible part already makes it clear that the prompt cannot be satisfied (for example, too many objects are already drawn, or the objects are arranged such that the required number could not appear naturally), OR
- even if the bottom quarter is completed in a natural and consistent way (including completing partially drawn objects), the final image would still not match the required number of objects.

Otherwise, say "possible".
"""

REFINE_FIRST_PROMPT = """Rewrite the prompt "{}" considering the given partially generated images, so that it fully describes the final image layout with the correct total number of objects and accurately satisfies the required attributes (e.g., shape, color) in the original prompt, helping the model complete the remaining part.

RULES
- Keep the object type and total count exactly the same as in the original prompt.
- Do NOT add, remove, or change objects. Never alter the number.
- Do NOT introduce new attributes not present in the original prompt.
- Strictly preserve the original prompt at the beginning; only append a simple clause if it is directly useful (e.g., "<X> on the top and <Y> on the bottom").
- If the visible quarter already contains all required objects but only in incomplete form, leave the prompt unchanged; however, if the empty lower area could make the model add extras, append a minimal placement clause that locks the existing set (e.g., "centered in a single horizontal row").

FALLBACK
- If any rule would be violated, unfeasible, or uncertain, return the original prompt unchanged.

OUTPUT
- Output exactly one sentence: either the refined prompt or the unchanged original.
- Begin with the original text prompt; when refining, append the clause after a comma.

EXAMPLES
Original: "A photo of eight bears"; Visible: three bears on the top →
Output: "A photo of eight bears, three on the top and five on the bottom."

Original: "A photo of one chicken"; Visible: only the upper half of the same chicken →
Output: "A photo of one chicken" (unchanged; continuation only)

Original: "A green bench and a red car"; Visible: the upper part of the green bench →
Output: "A green bench and a red car, the green bench on the top and the red car on the bottom"

Original: "A photo of five plates"; Visible: the upper halves of the same five plates already spanning the frame →
Output: "A photo of five plates, centered in a single horizontal row."
"""

REFINE_SECOND_PROMPT = refine_prompt = """Rewrite the prompt "{}" considering the given partially generated images, so that it fully describes the final image layout with the correct total number of objects and accurately satisfies the required attributes (e.g., shape, color) in the original prompt, helping the model complete the remaining part.

RULES
- Keep the object type and total count exactly the same as in the original prompt.
- Do NOT add, remove, or change objects. Never alter the number.
- Do NOT introduce new attributes not present in the original prompt.
- Strictly preserve the original prompt at the beginning; only append a simple clause if it is directly useful (e.g., "<X> on the top and <Y> on the bottom").
- If the visible half already contains all required objects but only in incomplete form, leave the prompt unchanged; however, if the empty lower area could make the model add extras, append a minimal placement clause that locks the existing set (e.g., "centered in a single horizontal row").

FALLBACK
- If any rule would be violated, unfeasible, or uncertain, return the original prompt unchanged.

OUTPUT
- Output exactly one sentence: either the refined prompt or the unchanged original.
- Begin with the original text prompt; when refining, append the clause after a comma.

EXAMPLES
Original: "A photo of eight bears"; Visible: three bears on the top →
Output: "A photo of eight bears, three on the top and five on the bottom."

Original: "A photo of one chicken"; Visible: only the upper half of the same chicken →
Output: "A photo of one chicken" (unchanged; continuation only)

Original: "A green bench and a red car"; Visible: the upper part of the green bench →
Output: "A green bench and a red car, the green bench on the top and the red car on the bottom"

Original: "A photo of five plates"; Visible: the upper halves of the same five plates already spanning the frame →
Output: "A photo of five plates, centered in a single horizontal row."
"""

REFINE_THIRD_PROMPT = None


def verify_image(prompt, image_file, model, processor):
    orm_prompt = 'This image is generated by a prompt: {}. Does this image accurately represent the prompt? Please answer yes or no without explanation.'
    messages = [
        {
            "role": "system",
            "content": [{"type": "text", "text": "You are a helpful assistant."}]
        },
        {
            "role": "user",
            "content": [
                {"type": "image", "image": image_file},
                {"type": "text",  "text": orm_prompt.format(prompt)}
            ]
        }
    ]
    inputs = processor.apply_chat_template(
        [messages],
        add_generation_prompt=True,
        tokenize=True,
        return_dict=True,
        return_tensors="pt"
    ).to(model.device, dtype=torch.bfloat16)
    input_len = inputs["input_ids"].shape[-1]

    succeed = False
    max_retries = 1
    retry_count = 0

    yes_token_id = processor.tokenizer.convert_tokens_to_ids("yes")
    yes_token_id_2 = processor.tokenizer.convert_tokens_to_ids("Yes")

    while not succeed and retry_count < max_retries:
        retry_count += 1
        # Generate answer
        with torch.no_grad():
            cont = model.generate(
                **inputs,
                do_sample=True,
                temperature=1.,
                max_new_tokens=20,
                return_dict_in_generate=True,
                output_scores=True,
            )
        sequences = cont.sequences
        cur_reponse = processor.tokenizer.convert_ids_to_tokens(sequences[0][input_len:])[0].lower()
        print(orm_prompt.format(prompt))
        print(cur_reponse)
        #print(processor.tokenizer.convert_ids_to_tokens(sequences[0][input_len:]))
        if cur_reponse not in ['yes', 'no']:    break
        else:   succeed = True
        scores = torch.cat([score.unsqueeze(1) for score in cont.scores], dim=1)
        scores = torch.nn.functional.softmax(scores, dim=-1)
        first_token_prob = scores[0, 0]
        yes_prob = first_token_prob[yes_token_id].item()
        yes_prob += first_token_prob[yes_token_id_2].item()
    if not succeed:
        print("Failed to generate a valid 'yes' or 'no' answer after maximum retries.")
        return False, 50.
    return ('yes' in cur_reponse), yes_prob

def add_horizontal_lines(images: np.ndarray,
                         num_rows: int = 4,
                         line_thickness: int = 1) -> np.ndarray:

    if images.ndim == 3:        # (H, W, C)
        images = images[np.newaxis, ...]  # (1, H, W, C)

    h = images.shape[1]
    line_positions = [(i * h) // num_rows for i in range(1, num_rows)]

    for img in images:
        for y in line_positions:
            img[y-1:y + 2, :, :] = 0  

    return images


def pil_to_pixel_values(pil_img, input_size=448, max_num=12):
    pil_img = pil_img.convert("RGB")
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(pil_img, img_size=input_size, use_thumbnail=True, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values


def encode_pil_image(pil_image):
    buffered = BytesIO()
    pil_image.save(buffered, format="JPEG")  
    return base64.b64encode(buffered.getvalue()).decode("utf-8")

def propagate_selection_flexible(
    gen_tokens: torch.Tensor,
    verify_texts: List[str],
    batch_size: int,
    BoN: int,
    grouped_by_4: bool,
    images: np.ndarray,
    cot_list: Optional[List[str]],
    prompts_per_group: int = 4,   # <-- 추가: 기본값 4
) -> Tuple[torch.Tensor, Optional[np.ndarray], Optional[List[str]], int, int]:
    assert BoN % 4 == 0, f"BoN({BoN}) must be a multiple of 4."
    groups_per_prompt = BoN // 4
    if prompts_per_group == 4:
        expected = batch_size * groups_per_prompt
    elif prompts_per_group == 2:
        expected = batch_size * groups_per_prompt * 2
    else:
        raise ValueError("prompts_per_group must be 2 or 4")
    
    assert len(verify_texts) == expected, \
        f"verify_texts len mismatch: expect {expected}, got {len(verify_texts)}"


    gen_tokens = gen_tokens.clone()
    images = images.copy()
    if cot_list is not None:
        cot_list = cot_list.copy()

    def parse_flags(txts: List[str]) -> List[bool]:
        flags_all = []
        for txt in txts:
            chunk = [v.strip().lower() for v in txt.split(",")]
            flags = [x == "possible" for x in chunk]
            flags_all.extend(flags)
        # ensure 4 flags
        if len(flags_all) < 4:
            print(f"[propagate_selection_flexible] Short verify_text (pad with 'False'): {txts}")
            flags_all += [False] * (4 - len(flags_all))
        elif len(flags_all) > 4:
            print(f"[propagate_selection_flexible] Long verify_text: {txts}")
            flags_all = flags_all[:4]
        return flags_all

    num_possible = 0
    num_all_impossible = 0
    
    for p in range(batch_size):
        group_labels = []
        possible_abs, impossible_abs = [], []

        for g in range(groups_per_prompt):
            if prompts_per_group == 4:
                txts = [verify_texts[p * groups_per_prompt + g]]
            else:  # prompts_per_group == 2
                base_idx = (p * groups_per_prompt + g) * 2
                txts = [verify_texts[base_idx], verify_texts[base_idx + 1]]

            flags = parse_flags(txts)
            group_labels.append(flags)

            base = p * BoN + g * 4
            for j, ok in enumerate(flags):
                idx = base + j
                (possible_abs if ok else impossible_abs).append(idx)

        if grouped_by_4:
            # propagate within each group of 4
            for g, flags in enumerate(group_labels):
                base = p * BoN + g * 4
                poss_local = [j for j, ok in enumerate(flags) if ok]
                imp_local  = [j for j, ok in enumerate(flags) if not ok]
                if len(poss_local) == 0:
                    num_all_impossible += 1
                    continue
                for k, j in enumerate(imp_local):
                    src_local = poss_local[k % len(poss_local)]
                    src, dst = base + src_local, base + j
                    gen_tokens[dst] = gen_tokens[src]
                    images[dst] = images[src]
                    if cot_list is not None:
                        cot_list[dst] = cot_list[src]
        else:
            # propagate across all groups for the prompt
            if len(possible_abs) == 0:
                num_all_impossible += 1
                continue
            for k, dst in enumerate(impossible_abs):
                src = possible_abs[k % len(possible_abs)]
                gen_tokens[dst] = gen_tokens[src]
                images[dst] = images[src]
                if cot_list is not None:
                    cot_list[dst] = cot_list[src]

        num_possible += len(possible_abs)

    return gen_tokens, images, cot_list, num_possible, num_all_impossible


@hydra.main(config_path="./configs", config_name="config_BoM")
def main(cfg: DictConfig):
    set_seed(cfg.seed)
    
    debug = getattr(cfg, "debug", False)
    
    model_params = cfg.model_params
    orm_id = cfg.orm_id
    model_name = cfg.model_params.model_name
    
    method_name = f"FIX_Ours__{model_name}__{cfg.prm_name}PRM__{cfg.orm_name}ORM__prompt_{cfg.refine.name}__basecfg{cfg.cfg_scale}_{cfg.three_way_cfg.name}__{cfg.benchmark.name}__group4{cfg.is_grouped_by_4}__N{cfg.BoN}_v2"
    folder_name = f"generated/{model_name}/{method_name}/samples"
    
    device = torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
    
    model_path = model_params.model.path
    dtype = getattr(torch, model_params.model.dtype)
    
    if "Janus" in model_name:
        from janus.models import MultiModalityCausalLM, VLChatProcessor
        
        vl_chat_processor: VLChatProcessor = VLChatProcessor.from_pretrained(model_path)
        tokenizer = vl_chat_processor.tokenizer
        
        vl_gpt: MultiModalityCausalLM = AutoModelForCausalLM.from_pretrained(
            model_path, trust_remote_code=True, device_map="auto"
        )
        vl_gpt = vl_gpt.to(dtype).eval()
        
        generator = Generator_Dict[model_name](vl_gpt, model_params.max_batch_size)
        embedding_model = vl_gpt

    elif "LlamaGen" in model_name:
        vl_chat_processor = None
        from llamagen.tokenizer.tokenizer_image.vq_model import VQ_models
        from llamagen.language.t5 import T5Embedder
        from llamagen.autoregressive.models.gpt import GPT_models
        
        vq_model = VQ_models["VQ-16"](
            codebook_size=16384,
            codebook_embed_dim=8)
        vq_model.to(device)
        vq_model.eval()
        checkpoint = torch.load(model_params.vq.path, map_location="cpu")
        vq_model.load_state_dict(checkpoint["model"])
        del checkpoint
        
        downsample_size = model_params.patch_size
        latent_size = model_params.img_size // downsample_size
        gpt_model = GPT_models["GPT-XL"](
            block_size=latent_size ** 2,
            cls_token_num=120,
            model_type="t2i",
        ).to(device=device, dtype=torch.bfloat16)
        
        checkpoint = torch.load(model_params.model.path, map_location="cpu")
        
        if "model" in checkpoint:  # ddp
            model_weight = checkpoint["model"]
        elif "module" in checkpoint: # deepspeed
            model_weight = checkpoint["module"]
        elif "state_dict" in checkpoint:
            model_weight = checkpoint["state_dict"]
        else:
            raise Exception("please check model weight")
            
        gpt_model.load_state_dict(model_weight, strict=False)
        gpt_model.eval()
        del checkpoint
        
        t5_model = T5Embedder(
            device=device, 
            local_cache=True, 
            cache_dir=model_params.t5.path, 
            dir_or_name=model_params.t5.type,
            model_max_length=model_params.t5.feature_max_len,
            torch_dtype=torch.bfloat16
        )
        
        generator = Generator_Dict[model_name](gpt_model, vq_model, latent_size, model_params.max_batch_size, device)
        embedding_model = (t5_model, gpt_model)
    
    if "Qwen2.5" in orm_id:
        from transformers import Qwen2_5_VLForConditionalGeneration
        from qwen_vl_utils import process_vision_info
        model = Qwen2_5_VLForConditionalGeneration.from_pretrained(
            orm_id, load_in_8bit=False, torch_dtype="auto", device_map='auto', trust_remote_code=True, 
        )
        processor = AutoProcessor.from_pretrained(orm_id)   
        
    # if "gemma-3" in orm_id:
    #     pass
    #     # from transformers import Gemma3ForConditionalGeneration
    #     # model = Gemma3ForConditionalGeneration.from_pretrained(
    #     #     orm_id, device_map='auto',trust_remote_code=True, max_memory={0: "32GB", 1: "32GB", 2: "32GB", 3: "32GB"}
    #     # ).eval()
    #     # processor = AutoProcessor.from_pretrained(orm_id)   
    # elif "InternVL3_5" in orm_id:
    #     # from transformers import AutoModel, AutoTokenizer
    #     # model = AutoModel.from_pretrained(
    #     #     orm_id,
    #     #     torch_dtype=torch.bfloat16,
    #     #     load_in_8bit=False,
    #     #     low_cpu_mem_usage=True,
    #     #     use_flash_attn=True,
    #     #     trust_remote_code=True,
    #     #     device_map="auto").eval()
    #     # internvl_tokenizer = AutoTokenizer.from_pretrained(orm_id, trust_remote_code=True, use_fast=False)
    #     # internvl_generation_config = dict(max_new_tokens=1024, do_sample=True)

    #     # from transformers import Qwen2_5_VLForConditionalGeneration
    #     # from qwen_vl_utils import process_vision_info
    #     # qwen_model = Qwen2_5_VLForConditionalGeneration.from_pretrained(
    #     #     "/home/Qwen2.5-VL-7B-Instruct", load_in_8bit=False, torch_dtype="auto", device_map='auto', trust_remote_code=True, 
    #     # )
    #     # processor = AutoProcessor.from_pretrained("/home/Qwen2.5-VL-7B-Instruct")   
    #     pass
    
    # elif "Llama-3.2" in orm_id:
    #     # from transformers import MllamaForConditionalGeneration
    #     # model = MllamaForConditionalGeneration.from_pretrained(
    #     #     orm_id, device_map='auto',trust_remote_code=True
    #     # )
    #     # processor = AutoProcessor.from_pretrained(orm_id)  
    #     pass 
    # elif "NVILA" in orm_id:
    #     # from transformers import AutoConfig, AutoModel
    #     # model = AutoModel.from_pretrained(orm_id, trust_remote_code=True, device_map="auto")
    #     pass
    
    print("- Load Completed - ")
    val_prompts, metadatas = load_metadata(cfg)
    categories = val_prompts.get("categories", None)
    
    batch_size = 2
    BoN = cfg.BoN
    
    if "geneval" in cfg.benchmark.name:
        assert cfg.benchmark.batch == 4
    else:
        assert cfg.benchmark.batch == 2
        
    out_gen_token_dir = Path(cfg.benchmark.outdirs) / f"{method_name}_gen_tokens"
    out_gen_token_dir.mkdir(parents=True, exist_ok=True)
    out_intermediate_dir = Path(cfg.benchmark.outdirs) / f"{method_name}_intermediate"
    out_intermediate_dir.mkdir(parents=True, exist_ok=True)
    
    print("Partition ID: ", cfg.partition_id)
    data_start_idx = (len(val_prompts['prompts']) // cfg.total_partitions) * (cfg.partition_id)
    data_end_idx = (len(val_prompts['prompts']) // cfg.total_partitions) * (cfg.partition_id+1)
    print("Start_idx: ", data_start_idx, "End_idx: ", data_end_idx)
    
    n_possible_first_row  = 0
    n_total_first_row = 0
    n_possible_second_row = 0
    n_total_second_row = 0
    n_all_impossible_first_row = 0
    n_all_impossible_second_row = 0
    
    for start_idx in trange(data_start_idx, data_end_idx, batch_size):
        set_seed(start_idx)
        
        if "geneval" in cfg.benchmark.name:
            prompts = val_prompts['prompts'][start_idx : start_idx + batch_size]
            names = val_prompts['name'][start_idx : start_idx + batch_size]
            save_path = [Path(cfg.benchmark.outdirs) / folder_name / name for name in names if not (Path(cfg.benchmark.outdirs) / folder_name / name).exists()]
            metas = metadatas[start_idx: start_idx + batch_size]

            for b_idx, (save, metadata) in enumerate(zip(save_path[::4], metas[::4])):
                os.makedirs(save.parent, exist_ok=True)
                with open(os.path.join(save.parent, "metadata.jsonl"), "w") as fp:
                    json.dump(metadata, fp)
            
        elif cfg.benchmark.name=="dpgbench":
            prompts = val_prompts['prompts'][start_idx: start_idx + batch_size]
            names = val_prompts['name'][start_idx: start_idx + batch_size]
            save_path = [Path(cfg.benchmark.outdirs) / folder_name / name for name in names if not (Path(cfg.benchmark.outdirs) / folder_name / name).exists()]
            
        elif "compbench" in cfg.benchmark.name:
            prompts = val_prompts['prompts'][start_idx : start_idx + batch_size]
            names = val_prompts['name'][start_idx : start_idx + batch_size]
            save_path = [Path(cfg.benchmark.outdirs) / folder_name / name for name in names if not (Path(cfg.benchmark.outdirs) / folder_name / name).exists()]
            # if not "non_spatial" in cfg.benchmark.name:
            #     prompts = ["A photo of " + prompt for prompt in prompts] # for a photo of
            # else:
            #     print("Not add a photo of")
        else:
            raise ValueError(f"benchmark name {cfg.benchmark.name} not supported.")
        
        if not len(save_path):
            print(f"already generated, skip prompt {start_idx}-{start_idx+batch_size}")
            continue
        
        if (out_intermediate_dir / f"batch{start_idx + b_idx:04d}").is_dir():
            print(f"already generated, skip prompt {start_idx}-{start_idx+batch_size}")
            continue
        
        print((out_intermediate_dir / f"batch{start_idx + b_idx:04d}").is_dir())
        
        BoN_scores = torch.zeros(batch_size, BoN, device=model.device)
        
        prompts_exp = [prompts[i // BoN] for i in range(batch_size * BoN)]
        
        input_embeds_prompt, attention_mask = Tokenizer_Dict[model_name](vl_chat_processor, embedding_model, prompts_exp, model_name, device)

        start_time = time.time()
        print('decoding start')
        
        ### Q1 Generate
        
        q1_req = T2IRequest(
            inputs_embeds=input_embeds_prompt,
            attention_mask=attention_mask,
            image_token_num_per_image=getattr(model_params, "image_token_num_per_image", None),
            img_size=getattr(model_params, "img_size", None),
            patch_size=getattr(model_params, "patch_size", None),
            temperature=getattr(model_params, "temperature", 1.0),
            top_k=getattr(model_params, "top_k", None),
            top_p=getattr(model_params, "top_p", None),
            pad=cfg.pad,
            cfg=cfg
        )
        
        q1_gen_tokens = generator.generate_t2i_first_quarter(q1_req)      # Best_of_N * batch_size tokens
        q1_image_array = generator.image_decode(q1_req, q1_gen_tokens)
        
        q1_img_size = model_params.img_size // 4
        q1_image_array[:, q1_img_size:, :, :] = 0
        
        q1_images = np.zeros((batch_size*BoN, model_params.img_size, model_params.img_size, 3), dtype=np.uint8)
        q1_images[:, :, :] = q1_image_array
        q1_images = add_horizontal_lines(q1_images, num_rows=4, line_thickness=10)
        
        aggregate_q1_prompts = prompts_exp[0::4]
        num_aggregate_q1_images = batch_size*BoN // 4
        aggregate_q1_images = np.zeros((num_aggregate_q1_images, model_params.img_size, model_params.img_size, 3), dtype=np.uint8)
        
        for prompt_idx in range(len(aggregate_q1_prompts)):
            quarters = q1_image_array[4*prompt_idx:4*(prompt_idx+1)]           
            for i in range(4):
                aggregate_q1_images[prompt_idx,q1_img_size*i:q1_img_size*(i+1),:,:] = quarters[i, :q1_img_size, :, :]
           
        aggregate_q1_images = add_horizontal_lines(aggregate_q1_images, num_rows=4, line_thickness=10)
             
        if debug:
            for i in range(num_aggregate_q1_images):
                fname = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage1_{i}.png"
                Path(fname).parent.mkdir(parents=True, exist_ok=True)
                
                Image.fromarray(aggregate_q1_images[i].astype("uint8")).save(fname)
            q1_gen_tokens_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage1.pt"
            torch.save(q1_gen_tokens.cpu(), q1_gen_tokens_file)
        ### Q1 Verify
           
        q1_verify_output = []
        for prompt_idx, prompt in enumerate(aggregate_q1_prompts):
            base64_image = encode_pil_image(Image.fromarray(aggregate_q1_images[prompt_idx]))
            
            if "gpt-4" in cfg.prm_name:
                response = client.responses.create(
                    model=cfg.prm_name,
                    input=[
                        {
                            "role": "user",
                            "content": [
                                { "type": "input_text", "text": GPT_SELECT_FIRST_QUARTERS_PROMPT.format(prompt)},
                                {
                                    "type": "input_image",
                                    "image_url": f"data:image/jpeg;base64,{base64_image}",
                                },
                            ],
                        }
                    ],
                    #  reasoning={
                    # "effort": "minimal"
                    # }
                )
                q1_verify_output.append(response.output_text)
            elif "gpt-5" in cfg.prm_name:
                response = client.responses.create(
                    model=cfg.prm_name,
                    input=[
                        {
                            "role": "user",
                            "content": [
                                { "type": "input_text", "text": GPT_SELECT_FIRST_QUARTERS_PROMPT.format(prompt)},
                                {
                                    "type": "input_image",
                                    "image_url": f"data:image/jpeg;base64,{base64_image}",
                                },
                            ],
                        }
                    ],
                    reasoning={
                    "effort": "minimal"
                    }
                )
                q1_verify_output.append(response.output_text)
            elif "gemini-2.5" in cfg.prm_name:
                # response = client_google.responses.create(
                #     model=cfg.prm_name,
                #     input=[
                #         {
                #             "role": "user",
                #             "content": [
                #                 { "type": "input_text", "text": GPT_SELECT_FIRST_QUARTERS_PROMPT.format(prompt)},
                #                 {
                #                     "type": "input_image",
                #                     "image_url": f"data:image/jpeg;base64,{base64_image}",
                #                 },
                #             ],
                #         }
                #     ],
                #     reasoning={
                #     "effort": "low"
                #     }
                # )
                # print(response.output_text)
                # raise RuntimeError
                response = client_google.chat.completions.create(
                    model=cfg.prm_name,
                    reasoning_effort="none",
                     messages=[
                        {
                            "role": "user",
                            "content": [
                                {
                                "type": "text",
                                "text": GPT_SELECT_FIRST_QUARTERS_PROMPT.format(prompt),
                                },
                                {
                                "type": "image_url",
                                "image_url": {
                                    "url":  f"data:image/jpeg;base64,{base64_image}"
                                },
                                },
                            ],
                        }
                    ],
                )

                q1_verify_output.append(response.choices[0].message.content)
            else:
                raise NotImplementedError
            
        q1_verify_output = [o.lower() for o in q1_verify_output]
        print(f"q1 verify output: {q1_verify_output}")
        print(f"Grouping by 4 image: {cfg.is_grouped_by_4}")
        
        if debug:
            q1_verify_output_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage1_verify_first_row.txt"
            
            with open(q1_verify_output_file, "w", encoding="utf-8") as f:
                f.write(str(q1_verify_output))
                
        
        q1_gen_tokens, q1_images, _, num_possible, num_all_impossible = propagate_selection_flexible(
            gen_tokens=q1_gen_tokens,
            verify_texts=[o.lower() for o in q1_verify_output],
            batch_size=batch_size,
            BoN=BoN,
            grouped_by_4=getattr(cfg, "is_grouped_by_4", False),
            images=q1_images,
            cot_list=None,
            prompts_per_group=4
        )
        
        n_possible_first_row += num_possible
        n_total_first_row += BoN * batch_size
        
        n_all_impossible_first_row += num_all_impossible
        
        if cfg.refine.skip_first_quarter:
            print("Skip refining at first quarter")
        else:
            q1_refine_output = []
            for idx, prompt in enumerate(prompts_exp):
                base64_image = encode_pil_image(Image.fromarray(q1_images[idx]))
                if "gpt-4" in cfg.prm_name:
                    response = client.responses.create(
                        model=cfg.prm_name,
                        input=[
                            {
                                "role": "user",
                                "content": [
                                    { "type": "input_text", "text": REFINE_FIRST_PROMPT.format(prompt)},
                                    {
                                        "type": "input_image",
                                        "image_url": f"data:image/jpeg;base64,{base64_image}",
                                    },
                                ],
                            }
                        ],
                    )
                    q1_refine_output.append(response.output_text)
                elif "gpt-5" in cfg.prm_name:
                    response = client.responses.create(
                        model=cfg.prm_name,
                        input=[
                            {
                                "role": "user",
                                "content": [
                                    { "type": "input_text", "text": REFINE_FIRST_PROMPT.format(prompt)},
                                    {
                                        "type": "input_image",
                                        "image_url": f"data:image/jpeg;base64,{base64_image}",
                                    },
                                ],
                            }
                        ],
                        reasoning={
                        "effort": "minimal"
                        }
                    )
                    q1_refine_output.append(response.output_text)
                elif "gemini-2.5" in cfg.prm_name:
                    # response = client.responses.create(
                    #     model=cfg.prm_name,
                    #     input=[
                    #             {
                    #             "role": "user",
                    #             "content": [
                    #                 { "type": "input_text", "text": REFINE_FIRST_PROMPT.format(prompt)},
                    #                 {
                    #                     "type": "input_image",
                    #                     "image_url": f"data:image/jpeg;base64,{base64_image}",
                    #                 },
                    #             ],
                    #         }
                    #     ],
                    #     reasoning={
                    #     "effort": "low"
                    #     }
                    # )
                    response = client_google.chat.completions.create(
                        model=cfg.prm_name,
                        reasoning_effort="none",
                        messages=[
                            {
                                "role": "user",
                                "content": [
                                    {
                                    "type": "text",
                                    "text": REFINE_FIRST_PROMPT.format(prompt),
                                    },
                                    {
                                    "type": "image_url",
                                    "image_url": {
                                        "url":  f"data:image/jpeg;base64,{base64_image}"
                                    },
                                    },
                                ],
                            }
                        ],
                    )
                    print([
                            {
                                "role": "user",
                                "content": [
                                    {
                                    "type": "text",
                                    "text": REFINE_FIRST_PROMPT.format(prompt),
                                    },
                                    {
                                    "type": "image_url",
                                    "image_url": {
                                        "url":  f"data:image/jpeg;base64,{base64_image}"
                                    },
                                    },
                                ],
                            }
                        ])
                    try:
                        q1_refine_output.append(response.choices[0].message.content)
                    except:
                        q1_refine_output.append(prompt)
                else:
                    raise NotImplementedError

            if cfg.save:
                q1_refine_output_file = out_gen_token_dir / f"batch{start_idx:04d}-{start_idx+batch_size-1}_prompt_{prompts[0]}_first_row.txt"
                
                with open(q1_refine_output_file, "w", encoding="utf-8") as f:
                    f.write(str(q1_refine_output))
                    
            if debug:
                q1_refine_output_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage1_refine_first_row.txt"
                
                with open(q1_refine_output_file, "w", encoding="utf-8") as f:
                    f.write(str(q1_refine_output))
                
            if getattr(cfg.three_way_cfg, "second_quarter", False):
                input_embeds_prompt, attention_mask = Tokenizer_3_way_Dict[model_name](vl_chat_processor, embedding_model, prompts_exp, q1_refine_output, model_name, device)
            else:
                input_embeds_prompt, attention_mask = Tokenizer_Dict[model_name](vl_chat_processor, embedding_model, q1_refine_output, model_name, device)
           
        ### Q2 Generate
     
        q2_req = T2IRequest(
            inputs_embeds=input_embeds_prompt,
            attention_mask=attention_mask,
            image_token_num_per_image=getattr(model_params, "image_token_num_per_image", None),
            img_size=getattr(model_params, "img_size", None),
            patch_size=getattr(model_params, "patch_size", None),
            temperature=getattr(model_params, "temperature", 1.0),
            top_k=getattr(model_params, "top_k", None),
            top_p=getattr(model_params, "top_p", None),
            pad=cfg.pad,
            cfg=cfg
        )
        
        q2_gen_tokens = generator.generate_t2i_second_quarter(q2_req, gen_tokens_q1=q1_gen_tokens)      # Best_of_N * batch_size tokens
        q2_image_array = generator.image_decode(q2_req, q2_gen_tokens)
        
        q2_img_size = model_params.img_size // 2
        q2_image_array[:, q2_img_size:, :, :] = 0
        
        q2_images = np.zeros((batch_size*BoN, model_params.img_size, model_params.img_size, 3), dtype=np.uint8)
        q2_images[:, :, :] = q2_image_array
        # q2_images = add_horizontal_lines(q2_images, num_rows=4, line_thickness=10)
        
        aggregate_q2_prompts = prompts_exp[0::2]
        num_aggregate_q2_images = batch_size*BoN // 2
        aggregate_q2_images = np.zeros((num_aggregate_q2_images, model_params.img_size, model_params.img_size, 3), dtype=np.uint8)
        
        for prompt_idx in range(len(aggregate_q2_prompts)):
            halfs = q2_image_array[2*prompt_idx:2*(prompt_idx+1)]           
            for i in range(2):
                aggregate_q2_images[prompt_idx,q2_img_size*i:q2_img_size*(i+1),:,:] = halfs[i, :q2_img_size, :, :]

        aggregate_q2_images = add_horizontal_lines(aggregate_q2_images, num_rows=2, line_thickness=10)
            
        if debug:
            for i in range(num_aggregate_q2_images):
                fname = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage2_{i}.png"
                Image.fromarray(aggregate_q2_images[i].astype("uint8")).save(fname)

            q2_gen_tokens_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage2.pt"
            torch.save(q2_gen_tokens.cpu(), q2_gen_tokens_file)
              
        ### Q2 Verify  
                
        q2_verify_output = []
        for prompt_idx, prompt in enumerate(aggregate_q2_prompts):
            base64_image = encode_pil_image(Image.fromarray(aggregate_q2_images[prompt_idx]))
            
            if "gpt-4" in cfg.prm_name:
                response = client.responses.create(
                    model=cfg.prm_name,
                    input=[
                        {
                            "role": "user",
                            "content": [
                                { "type": "input_text", "text": GPT_SELECT_SECOND_QUARTERS_PROMPT.format(prompt)},
                                {
                                    "type": "input_image",
                                    "image_url": f"data:image/jpeg;base64,{base64_image}",
                                },
                            ],
                        }
                    ],
                )
                q2_verify_output.append(response.output_text)
            elif "gpt-5" in cfg.prm_name:
                response = client.responses.create(
                    model=cfg.prm_name,
                    input=[
                        {
                            "role": "user",
                            "content": [
                                { "type": "input_text", "text": GPT_SELECT_SECOND_QUARTERS_PROMPT.format(prompt)},
                                {
                                    "type": "input_image",
                                    "image_url": f"data:image/jpeg;base64,{base64_image}",
                                },
                            ],
                        }
                    ],
                    reasoning={
                    "effort": "minimal"
                    }
                )
                q2_verify_output.append(response.output_text)
            elif "gemini-2.5" in cfg.prm_name:
                # response = client.responses.create(
                #     model=cfg.prm_name,
                #     input=[
                #         {
                #             "role": "user",
                #             "content": [
                #                 { "type": "input_text", "text": GPT_SELECT_SECOND_QUARTERS_PROMPT.format(prompt)},
                #                 {
                #                     "type": "input_image",
                #                     "image_url": f"data:image/jpeg;base64,{base64_image}",
                #                 },
                #             ],
                #         }
                #     ],
                #     reasoning={
                #     "effort": "low"
                #     }
                # )
                response = client_google.chat.completions.create(
                    model=cfg.prm_name,
                    reasoning_effort="none",
                    messages=[
                        {
                            "role": "user",
                            "content": [
                                {
                                "type": "text",
                                "text": GPT_SELECT_SECOND_QUARTERS_PROMPT.format(prompt),
                                },
                                {
                                "type": "image_url",
                                "image_url": {
                                    "url":  f"data:image/jpeg;base64,{base64_image}"
                                },
                                },
                            ],
                        }
                    ],
                )
                q2_verify_output.append(response.choices[0].message.content)
            else:
                raise NotImplementedError
            
        q2_verify_output = [o.lower() for o in q2_verify_output]
        print(f"q2 verify output: {q2_verify_output}")
        print(f"Grouping by 4 image: {cfg.is_grouped_by_4}")
        
        if debug:
            q2_verify_output_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage2_verify_second_row.txt"
            
            with open(q2_verify_output_file, "w", encoding="utf-8") as f:
                f.write(str(q2_verify_output))
        
        q2_gen_tokens, q2_images, _, num_possible, num_all_impossible = propagate_selection_flexible(
            gen_tokens=q2_gen_tokens,
            verify_texts=[o.lower() for o in q2_verify_output],
            batch_size=batch_size,
            BoN=BoN,
            grouped_by_4=getattr(cfg, "is_grouped_by_4", False),
            images=q2_images,
            cot_list=None,
            prompts_per_group=2
        )
        
        n_possible_second_row += num_possible
        n_total_second_row += BoN * batch_size
        
        n_all_impossible_second_row += num_all_impossible
        
        if cfg.refine.skip_second_quarter:
            print("Skip refining at second quarter")
        else:
            q2_refine_output = []
            for idx, prompt in enumerate(prompts_exp):
                base64_image = encode_pil_image(Image.fromarray(q2_images[idx]))
                if "gpt-4" in cfg.prm_name:
                    response = client.responses.create(
                        model=cfg.prm_name,
                        input=[
                            {
                                "role": "user",
                                "content": [
                                    { "type": "input_text", "text": REFINE_SECOND_PROMPT.format(prompt)},
                                    {
                                        "type": "input_image",
                                        "image_url": f"data:image/jpeg;base64,{base64_image}",
                                    },
                                ],
                            }
                        ],
                    )
                    q2_refine_output.append(response.output_text)
                elif "gpt-5" in cfg.prm_name:
                    response = client.responses.create(
                        model=cfg.prm_name,
                        input=[
                            {
                                "role": "user",
                                "content": [
                                    { "type": "input_text", "text": REFINE_SECOND_PROMPT.format(prompt)},
                                    {
                                        "type": "input_image",
                                        "image_url": f"data:image/jpeg;base64,{base64_image}",
                                    },
                                ],
                            }
                        ],
                        reasoning={
                        "effort": "minimal"
                        }
                    )
                    q2_refine_output.append(response.output_text)
                elif "gemini-2.5" in cfg.prm_name:
                    # response = client.responses.create(
                    #     model=cfg.prm_name,
                    #     input=[
                    #             {
                    #             "role": "user",
                    #             "content": [
                    #                 { "type": "input_text", "text": REFINE_SECOND_PROMPT.format(prompt)},
                    #                 {
                    #                     "type": "input_image",
                    #                     "image_url": f"data:image/jpeg;base64,{base64_image}",
                    #                 },
                    #             ],
                    #         }
                    #     ],
                    #     reasoning={
                    #     "effort": "low"
                    #     }
                    # )
                    response = client_google.chat.completions.create(
                        model=cfg.prm_name,
                        reasoning_effort="none",
                        messages=[
                            {
                                "role": "user",
                                "content": [
                                    {
                                    "type": "text",
                                    "text": REFINE_SECOND_PROMPT.format(prompt),
                                    },
                                    {
                                    "type": "image_url",
                                    "image_url": {
                                        "url":  f"data:image/jpeg;base64,{base64_image}"
                                    },
                                    },
                                ],
                            }
                        ],
                    )
                    try:
                        q2_refine_output.append(response.choices[0].message.content)
                    except:
                        q2_refine_output.append(prompt)
                else:
                    raise NotImplementedError

            print(q2_refine_output)
            
            if cfg.save:
                q2_refine_output_file = out_gen_token_dir / f"batch{start_idx:04d}-{start_idx+batch_size-1}_prompt_{prompts[0]}_second_row.txt"
                
                with open(q2_refine_output_file, "w", encoding="utf-8") as f:
                    f.write(str(q2_refine_output))
            
            if debug:
                q2_refine_output_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage2_refine_second_row.txt"
                
                with open(q2_refine_output_file, "w", encoding="utf-8") as f:
                    f.write(str(q2_refine_output))    
            
            if getattr(cfg.three_way_cfg, "third_quarter", False) or getattr(cfg.three_way_cfg, "second_half", False):
                input_embeds_prompt, attention_mask = Tokenizer_3_way_Dict[model_name](vl_chat_processor, embedding_model, prompts_exp, q2_refine_output, model_name, device)
            else:
                input_embeds_prompt, attention_mask = Tokenizer_Dict[model_name](vl_chat_processor, embedding_model, q2_refine_output, model_name, device)
                
        # If 3 quarters reject needed, Generate Q3
        if cfg.reject_3_quarters:
            raise NotImplementedError
        else:
            q4_req = T2IRequest(
                inputs_embeds=input_embeds_prompt,
                attention_mask=attention_mask,
                image_token_num_per_image=getattr(model_params, "image_token_num_per_image", None),
                img_size=getattr(model_params, "img_size", None),
                patch_size=getattr(model_params, "patch_size", None),
                temperature=getattr(model_params, "temperature", 1.0),
                top_k=getattr(model_params, "top_k", None),
                top_p=getattr(model_params, "top_p", None),
                pad=cfg.pad,
                cfg=cfg
            )
            
            q4_gen_tokens = generator.generate_t2i_second_half(q4_req, gen_tokens_half=q2_gen_tokens)      # Best_of_N * batch_size tokens
            q4_image_array = generator.image_decode(q4_req, token_ids=q4_gen_tokens)
            
            q4_images = np.zeros((batch_size*BoN, model_params.img_size, model_params.img_size, 3), dtype=np.uint8)
            q4_images[:, :, :] = q4_image_array
        
        if cfg.save:
            q4_gen_tokens_file = out_gen_token_dir / f"batch{start_idx:04d}-{start_idx+batch_size-1}_prompt_{prompts[0]}.pt"
            torch.save(q4_gen_tokens.cpu(), q4_gen_tokens_file)
            
        if debug:
            q4_gen_tokens_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_stage3.pt"
            torch.save(q4_gen_tokens.cpu(), q4_gen_tokens_file)


        images_bon = q4_images.reshape(
            batch_size, BoN, model_params.img_size, model_params.img_size, 3
        )
            
        for b_idx in range(batch_size):
            if cfg.save:
                out_intermediate_batch_dir = out_intermediate_dir / f"batch{start_idx + b_idx:04d}"
                out_intermediate_batch_dir.mkdir(parents=True, exist_ok=True)
            
            for score_idx in range(BoN):
                arr = images_bon[b_idx, score_idx]
                pil = Image.fromarray(arr.astype("uint8"))
                
                prompt_text = prompts[b_idx]
                _, score = verify_image(prompt_text, pil, model, processor)
                BoN_scores[b_idx, score_idx] = score

                if cfg.save:
                    out_intermediate_file = out_intermediate_batch_dir / f"cand_batch{b_idx}_sample{score_idx}.png"
                    pil.save(out_intermediate_file)
                    
                if debug:
                    debug_intermediate_file = f"./debug/{method_name}/{start_idx}-{start_idx+batch_size-1}/{prompts_exp[i]}_batch{b_idx}_sample{score_idx}.png"
                    pil.save(debug_intermediate_file)
                
        print(f"Num Possible, first: {n_possible_first_row}/{n_total_first_row}, second: {n_possible_second_row}/{n_total_second_row}")
        print(f"Num all impossible, first: {n_all_impossible_first_row}, second: {n_all_impossible_second_row}")
        print(f"---{method_name}---")
        
        select_idx = BoN_scores.argmax(dim=1)
                    
        end_time = time.time()
        print(f"Time taken: {end_time - start_time} seconds")
                                
        if cfg.benchmark.name=="dpgbench":
            per_prompt_images.extend([image for image in images])
            for img_idx in range(0, len(per_prompt_images), cfg.benchmark.batch):
                images = make_grid(per_prompt_images[img_idx: img_idx + cfg.benchmark.batch], nrow=2)
                images = images.astype('uint8')
                images = Image.fromarray(images)
                save_path[img_idx].parent.mkdir(parents=True, exist_ok=True)
                images.save(save_path[img_idx])
            per_prompt_images = []
        else:
            pil_images = []
            for b_idx in range(batch_size):
                k = int(select_idx[b_idx].item())
                arr = images_bon[b_idx, k]
                pil = Image.fromarray(arr.astype("uint8"))
                pil_images.append(pil)
                
            if cfg.save:
                for save_at, image in zip(save_path, pil_images):
                    save_at.parent.mkdir(parents=True, exist_ok=True)
                    image.save(save_at)
    
    if cfg.save:            
        cfg_path = Path(cfg.benchmark.outdirs) / f"generated/{model_name}/{method_name}/config/config_{cfg.partition_id}_over_{cfg.total_partitions}.json"
        
        cfg_path.parent.mkdir(parents=True, exist_ok=True)
        
        def next_available_path(p: Path) -> Path:
            if not p.exists():
                return p
            n = 1
            while True:
                candidate = p.with_name(f"{p.stem} ({n}){p.suffix}")
                if not candidate.exists():
                    return candidate
                n += 1
                
        cfg_path = next_available_path(cfg_path)
        
        save_config = {
            "cfg": OmegaConf.to_container(cfg, resolve=True),
            "n_possible_first_row": n_possible_first_row,
            "n_total_first_row": n_total_first_row,
            "n_possible_second_row": n_possible_second_row,
            "n_total_second_row": n_total_second_row,
            "n_all_impossible_first_row": n_all_impossible_first_row,
            "n_all_impossible_second_row": n_all_impossible_second_row,
        }
        
        with cfg_path.open("w", encoding="utf-8") as f:
            json.dump(save_config, f, ensure_ascii=False, indent=2)
    
if __name__=="__main__":
    main()