import matplotlib.pyplot as plt
import ast
import importlib.util
import inspect
import os
import csv
import torch.nn as nn
from fvcore.nn import FlopCountAnalysis

import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim

def get_network_width_depth_ratio(net):
    depth = 0
    width = []
    for name, module in net.named_modules():
        if isinstance(module, (nn.Conv1d, nn.Conv2d, nn.Conv3d,
                               nn.ConvTranspose1d, nn.ConvTranspose2d, nn.ConvTranspose3d,
                               nn.BatchNorm1d, nn.BatchNorm2d, nn.BatchNorm3d,
                               nn.InstanceNorm1d, nn.InstanceNorm2d, nn.InstanceNorm3d,
                               nn.LocalResponseNorm,
                               nn.Linear, nn.Bilinear,
                               nn.Dropout, nn.Dropout2d, nn.Dropout3d,
                               nn.Embedding, nn.EmbeddingBag,
                               nn.LSTM, nn.GRU, nn.RNN,
                               nn.PReLU, nn.ReLU, nn.ReLU6, nn.RReLU,
                               nn.SELU, nn.CELU, nn.ELU, nn.GELU, nn.SiLU,
                               nn.Sigmoid, nn.Tanh, nn.LogSigmoid, nn.Softplus, nn.Softshrink,
                               nn.Softsign, nn.Tanhshrink, nn.Threshold,
                               nn.AdaptiveAvgPool1d, nn.AdaptiveAvgPool2d, nn.AdaptiveAvgPool3d,
                               nn.AdaptiveLogSoftmaxWithLoss, nn.AdaptiveMaxPool1d, nn.AdaptiveMaxPool2d,
                               nn.AdaptiveMaxPool3d, nn.AvgPool1d, nn.AvgPool2d, nn.AvgPool3d,
                               nn.FractionalMaxPool2d, nn.MaxPool1d, nn.MaxPool2d, nn.MaxPool3d,
                               nn.LPPool1d, nn.LPPool2d, nn.LocalResponseNorm, nn.Softmax, nn.Softmin,
                               nn.LogSoftmax, nn.Threshold)):
            depth += 1
            #if hasattr(module, 'out_features'):
            if isinstance(module,nn.Conv2d):
                width.append(module.out_channels)
            elif hasattr(module, 'out_features'):
                width.append(module.out_features)
    ratio = max(width)/depth
    print(width)
    print(depth)
    return ratio



def count_parameters(model):
    """
    Counts the number of parameters in a PyTorch model.
    
    Args:
        model: PyTorch model.
    
    Returns:
        None.
    """
    total_params = 0
    for name, param in model.named_parameters():
        num_params = param.numel()
        total_params += num_params
    return total_params

class Net():
    """Dummy class. Will be replaced by evolved networks."""
    pass

def get_max_curiosity(species_list, n):
    species_list_sorted = sorted(species_list, key=lambda x: x.curiosity, reverse=True)
    return species_list_sorted[:n]

def get_max_fitness(species_list, n):
    species_list_sorted = sorted(species_list, key=lambda x: x.fitness, reverse=False)
    return species_list_sorted[:n]


def imshow(img):
    img = img / 2 + 0.5     # unnormalize
    npimg = img.numpy()
    plt.imshow(np.transpose(npimg, (1, 2, 0)))
    plt.show()

def read_python_file(filepath):
    remove_after_return(filepath)
    with open(filepath, "r") as f:
        code = f.read()
    return code

def extract_code(generation, prompt, file_path):
    code_parts = generation.split(prompt)

    with open(file_path, "w") as f:
        f.write(code_parts[1])


def get_net_name(filename):
    with open(filename, 'r') as file:
        source = file.read()

    class_nodes = [node for node in ast.parse(source).body if isinstance(node, ast.ClassDef)]
    if len(class_nodes) > 0:
        return class_nodes[0].name
    else:
        return None

def extract_code_section(code_string: str, prompt, file_path):
    # Split the input string into code sections based on commented lines
    current_section = []
    code_section = " "
    #print(f"code after mutation:\n {code_string}")
    for line in code_string.splitlines():
        if line.strip().startswith('"""'):
            # Start a new code section
            if current_section:
                code_section = "\n".join(current_section)
                current_section = []
        else:
            # Add the line to the current code section
            current_section.append(line)

    with open(file_path, "w") as f:
        f.write(f"import torch\nimport torch.nn as nn\nimport torch.nn.functional as F\n{code_string}")



def get_class(filename):
    module = os.path.basename(filename)
    spec = importlib.util.spec_from_file_location(module[:-3], filename)
    module = importlib.util.module_from_spec(spec)
    spec.loader.exec_module(module)
    
    for name, obj in inspect.getmembers(module):
        if inspect.isclass(obj):
            return obj
        
    return None


def csv_writer(results, output_file):

    with open(output_file, 'a', newline='') as csvfile:
        writer = csv.writer(csvfile)
        writer.writerow(results)

def count_unique_components(net):
    # Initialize a set to store the types of all components
    component_types = set()
    
    # Traverse the network to extract the type of each component
    for module in net.modules():
        module_type = type(module).__name__
        if module_type not in ["Sequential", "ModuleList", "ModuleDict", "Tensor"]:
            component_types.add(module_type)
    
    # Count the number of unique component types
    num_unique_components = len(component_types)
    
    return num_unique_components

def remove_after_return(file_path):
    # Read in the file
    with open(file_path, "r") as f:
        lines = f.readlines()

    # Find the index of the first "return" statement
    for i, line in enumerate(lines):
        if "return" in line:
            break

    # Remove everything after the "return" statement
    new_lines = lines[:i+1]

    # Write the updated code to the file
    with open(file_path, "w") as f:
        f.writelines(new_lines)

class Net11(nn.Module):
    def __init__(self):
        super(Net11, self).__init__()

        self.conv1 = nn.Conv2d(3, 256, kernel_size=3, padding=1)
        self.conv2 = nn.Conv2d(256, 256, kernel_size=3, padding=1)
        self.conv3 = nn.Conv2d(256, 512, kernel_size=3, padding=1)
        self.conv4 = nn.Conv2d(512, 512, kernel_size=3, padding=1)
        self.conv5 = nn.Conv2d(512, 1024, kernel_size=3, padding=1)
        self.conv6 = nn.Conv2d(1024, 1024, kernel_size=3, padding=1)
        self.conv7 = nn.Conv2d(1024, 2048, kernel_size=3, padding=1)
        self.conv8 = nn.Conv2d(2048, 2048, kernel_size=3, padding=1)
        
        self.bn1 = nn.BatchNorm2d(256)
        self.bn2 = nn.BatchNorm2d(512)
        self.bn3 = nn.BatchNorm2d(1024)
        self.bn4 = nn.BatchNorm2d(2048)

        self.pool = nn.MaxPool2d(2, 2)
        self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))

        self.fc = nn.Linear(2048, 10)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.pool(F.relu(self.bn1(self.conv2(x))))
        x = F.relu(self.bn2(self.conv3(x)))
        x = self.pool(F.relu(self.bn2(self.conv4(x))))
        x = F.relu(self.bn3(self.conv5(x)))
        x = self.pool(F.relu(self.bn3(self.conv6(x))))
        x = F.relu(self.bn4(self.conv7(x)))
        x = self.pool(F.relu(self.bn4(self.conv8(x))))
        
        x = self.avg_pool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x


class Net22(nn.Module):
    def __init__(self):
        super(Net22, self).__init__()

        self.conv1 = nn.Conv2d(3, 1024, kernel_size=3, padding=1)
        self.conv2 = nn.Conv2d(1024, 1024, kernel_size=3, padding=1)
        self.conv3 = nn.Conv2d(1024, 2048, kernel_size=3, padding=1)
        self.bn1 = nn.BatchNorm2d(1024)
        self.bn2 = nn.BatchNorm2d(2048)

        self.pool = nn.MaxPool2d(2, 2)
        self.avg_pool = nn.AdaptiveAvgPool2d((1, 1))

        self.fc = nn.Linear(2048, 10)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.pool(F.relu(self.bn1(self.conv2(x))))
        x = self.pool(F.relu(self.bn2(self.conv3(x))))

        x = self.avg_pool(x)
        x = torch.flatten(x, 1)  # flatten all dimensions except the batch dimension
        x = self.fc(x)

        return x

class Net33(nn.Module):
    def __init__(self):
        super(Net33, self).__init__()
        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, padding=1)
        self.bn1 = nn.BatchNorm2d(64)
        self.conv2 = nn.Conv2d(64, 64, kernel_size=3, padding=1)
        self.bn2 = nn.BatchNorm2d(64)
        self.conv3 = nn.Conv2d(64, 128, kernel_size=3, padding=1)
        self.bn3 = nn.BatchNorm2d(128)
        self.conv4 = nn.Conv2d(128, 128, kernel_size=3, padding=1)
        self.bn4 = nn.BatchNorm2d(128)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.25)

        self.lstm_input_size = 128 * 16 * 16
        self.lstm_hidden_size = 256
        self.lstm_num_layers = 2

        self.lstm = nn.LSTM(input_size=self.lstm_input_size,
                            hidden_size=self.lstm_hidden_size,
                            num_layers=self.lstm_num_layers,
                            batch_first=True)

        self.fc1 = nn.Linear(self.lstm_hidden_size, 512)
        self.fc2 = nn.Linear(512, 10)

    def forward(self, x):
        x = F.relu(self.bn1(self.conv1(x)))
        x = self.pool(F.relu(self.bn2(self.conv2(x))))
        x = F.relu(self.bn3(self.conv3(x)))
        x = F.relu(self.bn4(self.conv4(x)))
        x = self.dropout(x)

        x = x.view(x.size(0), 1, -1)  # Reshape tensor for LSTM input
        x, _ = self.lstm(x)  # LSTM output
        x = x[:, -1, :]  # Last LSTM output

        x = F.relu(self.fc1(x))
        x = self.fc2(x)
        return x
    
class CustomCNN(nn.Module):
    def __init__(self):
        super(CustomCNN, self).__init__()

        self.conv1 = nn.Conv2d(3, 64, kernel_size=3, padding=1)
        self.bn1 = nn.BatchNorm2d(64)
        self.conv2 = nn.Conv2d(64, 64, kernel_size=3, padding=1)
        self.bn2 = nn.BatchNorm2d(64)
        self.dropout1 = nn.Dropout2d(0.5)
        
        self.conv3 = nn.Conv2d(64, 128, kernel_size=3, padding=1)
        self.bn3 = nn.BatchNorm2d(128)
        self.conv4 = nn.Conv2d(128, 128, kernel_size=3, padding=1)
        self.bn4 = nn.BatchNorm2d(128)
        self.dropout2 = nn.Dropout2d(0.5)
        
        self.conv5 = nn.Conv2d(128, 256, kernel_size=3, padding=1)
        self.bn5 = nn.BatchNorm2d(256)
        self.conv6 = nn.Conv2d(256, 256, kernel_size=3, padding=1)
        self.bn6 = nn.BatchNorm2d(256)
        self.dropout3 = nn.Dropout2d(0.5)
        
        self.fc1 = nn.Linear(256*4*4, 10) # CIFAR-10 has 10 classes

    def forward(self, x):
        out1 = F.relu(self.bn1(self.conv1(x)))
        out2 = self.dropout1(F.relu(self.bn2(self.conv2(out1))))
        out2 += out1 # skip connection
        out2 = F.max_pool2d(out2, 2)
        
        out3 = F.relu(self.bn3(self.conv3(out2)))
        out4 = self.dropout2(F.relu(self.bn4(self.conv4(out3))))
        out4 += out3 # skip connection
        out4 = F.max_pool2d(out4, 2)
        
        out5 = F.relu(self.bn5(self.conv5(out4)))
        out6 = self.dropout3(F.relu(self.bn6(self.conv6(out5))))
        out6 += out5 # skip connection
        out6 = F.max_pool2d(out6, 2)
        
        out6 = out6.view(out6.size(0), -1) # flatten the tensor
        out7 = self.fc1(out6)

        return out7


dummy_inputs = torch.zeros((1, 3, 32, 32))
"""print("net1")

net = Net11()
print(get_network_width_depth_ratio(net))
flps = FlopCountAnalysis(net, dummy_inputs)
print(flps.total())
print(count_parameters(net))

print("net2")
net2 = Net22()
flps = FlopCountAnalysis(net2, dummy_inputs)
print(get_network_width_depth_ratio(net2))
print(flps.total())
print(count_parameters(net2))

print("net3")
net3 = Net33()
flps = FlopCountAnalysis(net3, dummy_inputs)
print(get_network_width_depth_ratio(net3))
print(flps.total())
print(count_parameters(net3))

print("net4")
net4 = CustomCNN()
flps = FlopCountAnalysis(net4, dummy_inputs)
print(get_network_width_depth_ratio(net4))
print(flps.total())
print(count_parameters(net3))
"""

### FOR GIF

#Add a layer to improve the above network.
class Net(nn.Module):
    def __init__(self):
        super().__init__()
        self.conv1 = nn.Conv2d(3, 1, 1)
        self.fc1 = nn.Linear(1024, 10)

    def forward(self, x):
        x = F.relu(self.conv1(x))
        x = torch.flatten(x, 1)
        x = F.relu(self.fc1(x))
        return x
# Improve the above network
class Neta(nn.Module):
    def __init__(self):
        super(Neta, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.pool = nn.MaxPool2d(2, 2)
        self.fc1 = nn.Linear(32 * 8 * 8, 512)
        self.fc2 = nn.Linear(512, 10)

    def forward(self, x):
        x = self.pool(F.relu(self.conv1(x)))
        x = self.pool(F.relu(self.conv2(x)))
        x = x.view(-1, 32 * 8 * 8)
        x = F.relu(self.fc1(x))
        x = self.fc2(x)
        return x
#Add fully connected layer to improve the above network    
class Netb(nn.Module):
    def __init__(self):
        super(Netb, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(64 * 4 * 4, 512)
        self.fc2 = nn.Linear(512, 256)
        self.fc3 = nn.Linear(256, 10)

    def forward(self, x):
        x = self.pool(F.relu(self.bn1(self.conv1(x))))
        x = self.pool(F.relu(self.bn2(self.conv2(x))))
        x = self.pool(F.relu(self.bn3(self.conv3(x))))
        x = x.view(-1, 64 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = F.relu(self.fc2(x))
        x = self.dropout(x)
        x = self.fc3(x)
        return x

class Netc(nn.Module):
    def __init__(self):
        super(Netc, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(64 * 4 * 4, 512)
        self.fc2 = nn.Linear(512, 256)
        self.fc3 = nn.Linear(256, 128)
        self.fc4 = nn.Linear(128, 10)

    def forward(self, x):
        x = self.pool(F.relu(self.bn1(self.conv1(x))))
        x = self.pool(F.relu(self.bn2(self.conv2(x))))
        x = self.pool(F.relu(self.bn3(self.conv3(x))))
        x = x.view(-1, 64 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = F.relu(self.fc2(x))
        x = self.dropout(x)
        x = F.relu(self.fc3(x))
        x = self.dropout(x)
        x = self.fc4(x)
        return x
#Delete a layer to improve the above network  
class Netd(nn.Module):
    def __init__(self):
        super(Netd, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(64 * 4 * 4, 512)
        self.fc2 = nn.Linear(512, 10)

    def forward(self, x):
        x = self.pool(F.relu(self.bn1(self.conv1(x))))
        x = self.pool(F.relu(self.bn2(self.conv2(x))))
        x = self.pool(F.relu(self.bn3(self.conv3(x))))
        x = x.view(-1, 64 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = self.fc2(x)
        return x
#Add residual connection to improve the above network   
class Nete(nn.Module):
    def __init__(self):
        super(Nete, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(64 * 4 * 4, 512)
        self.fc2 = nn.Linear(512, 10)
        self.residual_conv = nn.Conv2d(16, 64, 1, stride=4, padding=0)  # 1x1 convolution for matching dimensions

    def forward(self, x):
        x1 = self.pool(F.relu(self.bn1(self.conv1(x))))
        x2 = self.pool(F.relu(self.bn2(self.conv2(x1))))
        x3 = self.pool(F.relu(self.bn3(self.conv3(x2))))
        x_residual = self.residual_conv(x1)  # Apply 1x1 convolution to match dimensions
        x3 += x_residual  # Add the residual connection
        x = x3.view(-1, 64 * 4 * 4)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = self.fc2(x)
        return x
#Add convolutional layer to improve the above network
class Netf(nn.Module):
    def __init__(self):
        super(Netf, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.conv4 = nn.Conv2d(64, 128, 3, padding=1)
        self.bn4 = nn.BatchNorm2d(128)
        self.pool = nn.MaxPool2d(2, 2)
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(128 * 2 * 2, 512)
        self.fc2 = nn.Linear(512, 10)
        self.residual_conv = nn.Conv2d(16, 128, 1, stride=8, padding=0)  # 1x1 convolution for matching dimensions

    def forward(self, x):
        x1 = self.pool(F.relu(self.bn1(self.conv1(x))))
        x2 = self.pool(F.relu(self.bn2(self.conv2(x1))))
        x3 = self.pool(F.relu(self.bn3(self.conv3(x2))))
        x4 = self.pool(F.relu(self.bn4(self.conv4(x3))))
        x_residual = self.residual_conv(x1)  # Apply 1x1 convolution to match dimensions
        x4 += x_residual  # Add the residual connection
        x = x4.view(-1, 128 * 2 * 2)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = self.fc2(x)
        return x
#Improve the above network   
class Netg(nn.Module):
    def __init__(self):
        super(Netg, self).__init__()
        self.conv1 = nn.Conv2d(3, 16, 3, padding=1)
        self.bn1 = nn.BatchNorm2d(16)
        self.conv2 = nn.Conv2d(16, 32, 3, padding=1)
        self.bn2 = nn.BatchNorm2d(32)
        self.conv3 = nn.Conv2d(32, 64, 3, padding=1)
        self.bn3 = nn.BatchNorm2d(64)
        self.conv4 = nn.Conv2d(64, 128, 3, padding=1)
        self.bn4 = nn.BatchNorm2d(128)
        self.pool = nn.MaxPool2d(2, 2)
        self.adaptive_pool = nn.AdaptiveAvgPool2d((1, 1))
        self.dropout = nn.Dropout(0.5)
        self.fc1 = nn.Linear(128, 512)
        self.fc2 = nn.Linear(512, 10)
        self.residual_conv1 = nn.Conv2d(16, 64, 1, stride=4, padding=0)  # 1x1 convolution for matching dimensions
        self.residual_conv2 = nn.Conv2d(64, 128, 1, stride=2, padding=0)  # 1x1 convolution for matching dimensions

    def forward(self, x):
        x1 = self.pool(F.relu(self.bn1(self.conv1(x))))
        x2 = self.pool(F.relu(self.bn2(self.conv2(x1))))
        x_residual1 = self.residual_conv1(x1)  # Apply 1x1 convolution to match dimensions
        x3 = self.pool(F.relu(self.bn3(self.conv3(x2))))
        x3 += x_residual1  # Add the residual connection
        x4 = self.pool(F.relu(self.bn4(self.conv4(x3))))
        x_residual2 = self.residual_conv2(x3)  # Apply 1x1 convolution to match dimensions
        x4 += x_residual2  # Add the residual connection
        x = self.adaptive_pool(x4)
        x = x.view(-1, 128)
        x = F.relu(self.fc1(x))
        x = self.dropout(x)
        x = self.fc2(x)
        return x
    
net = Netg()
output = net(dummy_inputs)
print(output)
print(output.shape == (1, 10))





