import torch
import torch.nn as nn
import torch.nn.functional as F


class Generator(nn.Module):
    def __init__(self, classes=100, channels=3):
        super().__init__()
        # Filters [1024, 512, 256]
        # Input_dim = 100
        # Output_dim = C (number of channels)
        self.main_module = nn.Sequential(
            # Z latent vector 100
            nn.ConvTranspose2d(in_channels=200, out_channels=1024, kernel_size=4, stride=1, padding=0),
            nn.BatchNorm2d(num_features=1024),
            nn.ReLU(True),

            # State (1024x4x4)
            nn.ConvTranspose2d(in_channels=1024, out_channels=512, kernel_size=4, stride=2, padding=1),
            nn.BatchNorm2d(num_features=512),
            nn.ReLU(True),

            # State (512x8x8)
            nn.ConvTranspose2d(in_channels=512, out_channels=256, kernel_size=4, stride=2, padding=1),
            nn.BatchNorm2d(num_features=256),
            nn.ReLU(True),

            # State (256x16x16)
            nn.ConvTranspose2d(in_channels=256, out_channels=channels, kernel_size=4, stride=2, padding=1))
        # output of main module --> Image (Cx32x32)

        self.output = nn.Tanh()
        self.fc_y = nn.Linear(classes, 1024)
        self.fc_x = nn.Linear(100, 1024)
        self.fc = nn.Linear(2048, 200)

        self.weights_init()

    def forward(self, x, y):
        y_ = F.relu(self.fc_y(y))
        x = self.fc_x(x)
        x = torch.cat([x, y_], 1)
        x = self.fc(x)
        B, C = x.size()
        x = x.view(B, C, 1, 1)
        x = self.main_module(x)
        x = self.output(x)
        return x

    def weights_init(self):
        '''
        input: an initialized model
        output: reinitialized convolutional, convolutional-transpose, and batch normalization layers
        '''
        for m in self.main_module:
            if isinstance(m, nn.Conv2d):
                nn.init.normal_(m.weight.data, 0.0, 0.02)
            elif isinstance(m, nn.BatchNorm3d):
                nn.init.normal_(m.weight.data, 1.0, 0.02)
                nn.init.constant_(m.bias.data, 0)


class Discriminator(nn.Module):
    def __init__(self, classes=100, channels=3):
        super().__init__()
        # Filters [256, 512, 1024]
        # Input_dim = channels (Cx64x64)
        # Output_dim = 1
        self.main_module = nn.Sequential(
            # Image (Cx32x32)
            nn.Conv2d(in_channels=channels, out_channels=256, kernel_size=4, stride=2, padding=1),
            nn.LeakyReLU(0.2, inplace=True),

            # State (256x16x16)
            nn.Conv2d(in_channels=256, out_channels=512, kernel_size=4, stride=2, padding=1),
            nn.BatchNorm2d(512),
            nn.LeakyReLU(0.2, inplace=True),

            # State (512x8x8)
            nn.Conv2d(in_channels=512, out_channels=1024, kernel_size=4, stride=2, padding=1),
            nn.BatchNorm2d(1024),
            nn.LeakyReLU(0.2, inplace=True))
        # outptut of main module --> State (1024x4x4)

        self.final_conv = nn.Conv2d(in_channels=1024, out_channels=1, kernel_size=4, stride=1, padding=0)

        self.output = nn.Sequential(
            nn.Linear(2, 1),
            # Output 1
            nn.Sigmoid())

        self.fc_y = nn.Linear(classes, 1)
        self.weights_init()

    def forward(self, x, y):
        x = self.main_module(x)
        x = self.final_conv(x)
        y = F.relu(self.fc_y(y))
        x = torch.cat([x.view(x.size(0), 1), y], 1)
        return self.output(x)

    def weights_init(self):
        '''
        input: an initialized model
        output: reinitialized convolutional, convolutional-transpose, and batch normalization layers
        '''
        for m in self.main_module:
            if isinstance(m, nn.Conv2d):
                nn.init.normal_(m.weight.data, 0.0, 0.02)
            elif isinstance(m, nn.BatchNorm3d):
                nn.init.normal_(m.weight.data, 1.0, 0.02)
                nn.init.constant_(m.bias.data, 0)
        nn.init.normal_(self.final_conv.weight.data, 0.0, 0.02)

    def feature_extraction(self, x):
        # Use discriminator for feature extraction then flatten to vector of 16384 features
        x = self.main_module(x)
        return x.view(-1, 1024 * 4 * 4)


if __name__ == '__main__':
    x = torch.randn(1, 100)
    y = torch.ones(1, 100)
    g = Generator()
    d = Discriminator()
    o = g(x, y)
    print(o.size())

    import torchvision.transforms as transforms
    from PIL import Image
    import matplotlib.pyplot as plt

    generated_image = transforms.ToPILImage()(o.squeeze(0))
    plt.imshow(generated_image)
    plt.axis('off')
    plt.show()

    print(d(o, y).size())

