import argparse
import distutils.util
import os
import random
import shutil

import math
import numpy as np
import sys
import yaml
from PIL import Image

import torch
import torch.nn as nn
import torch.optim as optim
import torch.nn.functional as F
import torch.utils.data as data
import torchvision.transforms as transforms
import torchvision.datasets as datasets
import torch.optim as optim
from torch.optim.lr_scheduler import LRScheduler

# config_file = './../../env.yml'
config_file = './env.yml'
with open(config_file, 'r') as stream:
    yamlfile = yaml.safe_load(stream)
    root_dir = yamlfile['root_dir']
    src_dir = yamlfile['src_dir']

sys.path.append(src_dir)
sys.path.append(os.path.join(src_dir, 'attack'))
sys.path.append(os.path.join(src_dir, 'models'))
from attack.dsq_attack import system_attack
from utils import mkdir_p, AverageMeter, accuracy, print_acc_conf, TrainRecorder
from tinyimagenet_utils import transform_train, transform_train_aug, transform_test, TINdata, DistillTINdata, WarmUpLR, \
    ModelwNorm, transform_train_aug_atm
from tinyimagenet.models.freeze_util import dfs_freeze_fea_rewind_cls, dfs_freeze_ps_rewind_pr
from tinyimagenet.models.model_selector import get_network

device = torch.device("cuda" if torch.cuda.is_available() else "cpu")


class AllInOne(data.Dataset):
    def __init__(self, data, labels, non_data, non_labels, transform, transform_test, k=1):
        #
        self.labels = labels
        self.data = data
        self.non_labels = non_labels
        self.non_data = non_data
        self.transform = transform
        self.transform_test = transform_test
        self.r = np.arange(len(self.labels))
        self.non_r = np.arange(len(self.non_labels))
        self.k = k

    def __getitem__(self, index):
        label = self.labels[index]
        img = Image.fromarray((self.data[index].transpose(1, 2, 0).astype(np.uint8)))
        img = self.transform(img)

        tr_index = np.random.choice(self.r, size=self.k)
        tr_label = self.labels[tr_index]#.reshape((1, self.k))
        temp = []
        for ind in tr_index:
            tr_img = Image.fromarray((self.data[ind].transpose(1, 2, 0).astype(np.uint8)))
            temp += [self.transform_test(tr_img)]
        tr_img = torch.stack(temp)

        non_index = np.random.choice(self.non_r, size=self.k)
        non_label = self.non_labels[non_index]#.reshape((1, self.k))
        temp = []
        for ind in non_index:
            non_img = Image.fromarray((self.non_data[ind].transpose(1, 2, 0).astype(np.uint8)))
            temp += [self.transform_test(non_img)]
        non_img = torch.stack(temp)

        return img, label, tr_img, tr_label, non_img, non_label

    def __len__(self):
        return len(self.labels)


class Cifardata(data.Dataset):
    def __init__(self, data, labels, transform):
        self.data = data
        self.transform = transform
        self.labels = labels

    def __getitem__(self, index):
        img = Image.fromarray((self.data[index].transpose(1, 2, 0).astype(np.uint8)))
        label = self.labels[index]
        img = self.transform(img)
        return img, label

    def __len__(self):
        return len(self.labels)

    def update(self):
        r = np.arange(len(self.labels))
        np.random.shuffle(r)
        self.labels = self.labels[r]
        self.data = self.data[r]


class InferenceAttack_HZ(nn.Module):
    def __init__(self, num_classes):
        self.num_classes = num_classes
        super(InferenceAttack_HZ, self).__init__()
        self.features = nn.Sequential(
            nn.Linear(num_classes, 1024),
            nn.ReLU(),
            nn.Linear(1024, 512),
            nn.ReLU(),
            nn.Linear(512, 64),
            nn.ReLU(),
        )
        self.labels = nn.Sequential(
            nn.Linear(num_classes, 128),
            nn.ReLU(),
            nn.Linear(128, 64),
            nn.ReLU(),
        )
        self.combine = nn.Sequential(
            nn.Linear(64 * 2, 512),
            nn.ReLU(),
            nn.Linear(512, 256),
            nn.ReLU(),
            nn.Linear(256, 128),
            nn.ReLU(),
            nn.Linear(128, 64),
            nn.ReLU(),
            nn.Linear(64, 1),
        )
        for key in self.state_dict():
            if key.split('.')[-1] == 'weight':
                nn.init.normal(self.state_dict()[key], std=0.01)
            elif key.split('.')[-1] == 'bias':
                self.state_dict()[key][...] = 0
        self.output = nn.Sigmoid()

    def forward(self, x1, l):
        out_x1 = self.features(x1)
        out_l = self.labels(l)
        is_member = self.combine(torch.cat((out_x1, out_l), 1))

        return self.output(is_member)


def train(trainloader, model, criterion, optimizer, epoch, warmup_scheduler, args):
    # switch to train mode
    model.train()

    losses = AverageMeter()
    top1 = AverageMeter()

    for batch_ind, (inputs, targets) in enumerate(trainloader):

        if epoch <= args.warmup:
            warmup_scheduler.step()

        inputs = inputs.to(device, torch.float)
        targets = targets.to(device, torch.long)

        outputs = model(inputs)
        loss = criterion(outputs, targets)

        # compute gradient and do SGD step        
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        prec1, _ = accuracy(outputs.data, targets.data, topk=(1, 5))
        losses.update(loss.item(), inputs.size()[0])
        top1.update(prec1.item() / 100.0, inputs.size()[0])

    return (losses.avg, top1.avg)


def train_privatly(
        trainloader, model, inference_model, criterion, optimizer, inference_criterion, inference_optimizer, epoch,
        warmup_scheduler, args,
        alpha=1, k=1
):
    losses = AverageMeter()
    top1 = AverageMeter()

    for batch_ind, (inputs, targets, inputs_tag, targets_tag, inputs_attack, targets_attack) in enumerate(trainloader):

        if epoch <= args.warmup:
            warmup_scheduler.step()

        inference_model.train()
        model.eval()
        # train inf model
        for ind in range(k):
            # print(targets_tag.size(), targets_attack.size())
            targets_tag_cur = targets_tag[:, ind].view(-1).to(device, torch.long)
            targets_attack_cur = targets_attack[:, ind].view(-1).to(device, torch.long)
            inputs_tag_cur = inputs_tag[:, ind, :, :, :].to(device, torch.float)
            inputs_attack_cur = inputs_attack[:, ind, :, :, :].to(device, torch.float)

            outputs = model(inputs_tag_cur)
            outputs_non = model(inputs_attack_cur)

            comb_inputs = torch.cat((outputs, outputs_non))
            comb_targets = torch.cat((targets_tag_cur, targets_attack_cur)).view([-1, 1]).to(device, torch.float)

            one_hot_tr = torch.zeros(comb_inputs.size()[0], comb_inputs.size()[1]).to(device, torch.float)
            target_one_hot = one_hot_tr.scatter_(1, comb_targets.to(device, torch.long).data, 1)

            attack_output = inference_model(comb_inputs, target_one_hot).view([-1])

            att_labels = torch.zeros((inputs.size()[0] + inputs_attack.size()[0]))
            att_labels[:inputs.size()[0]] = 1.0
            att_labels[inputs.size()[0]:] = 0.0
            is_member_labels = att_labels.to(device, torch.float)

            loss_attack = inference_criterion(attack_output, is_member_labels)
            inference_optimizer.zero_grad()
            loss_attack.backward()
            inference_optimizer.step()


        model.train()
        inference_model.eval()
        # train classification model
        inputs = inputs.to(device, torch.float)
        targets = targets.to(device, torch.long)
        # infer
        outputs = model(inputs)
        one_hot_tr = torch.zeros(outputs.size()[0], outputs.size()[1]).to(device, torch.float)
        target_one_hot = one_hot_tr.scatter_(1, targets.to(device, torch.long).view([-1, 1]).data, 1)

        inference_output = inference_model(outputs, target_one_hot).view([-1])
        loss = criterion(outputs, targets) + ((alpha) * (torch.mean((inference_output)) - 0.5))
        # compute gradient and do SGD step
        optimizer.zero_grad()
        loss.backward()
        optimizer.step()

        prec1, _ = accuracy(outputs.data, targets.data, topk=(1, 5))

        losses.update(loss.item(), inputs.size()[0])
        top1.update(prec1.item() / 100.0, inputs.size()[0])

    return (losses.avg, top1.avg)


def test(testloader, model, criterion, batch_size):
    model.eval()

    losses = AverageMeter()
    top1 = AverageMeter()

    for batch_ind, (inputs, targets) in enumerate(testloader):
        inputs = inputs.to(device, torch.float)
        targets = targets.to(device, torch.long)

        outputs = model(inputs)
        prec1, _ = accuracy(outputs.data, targets.data, topk=(1, 5))
        top1.update(prec1.item() / 100.0, inputs.size()[0])

        loss = criterion(outputs, targets)
        losses.update(loss.item(), inputs.size()[0])

    return (losses.avg, top1.avg)


def save_checkpoint(state, is_best, acc, checkpoint):
    if not os.path.isdir(checkpoint):
        mkdir_p(checkpoint)
    filepath = os.path.join(checkpoint, 'model_last.pth.tar')
    torch.save(state, filepath)
    if is_best:
        shutil.copyfile(filepath, os.path.join(checkpoint, 'model_best.pth.tar'))


def get_learning_rate(optimizer):
    lr = []
    for param_group in optimizer.param_groups:
        lr += [param_group['lr']]
    return lr


def get_opt_and_lrsch(args, model, num_epoch, num_iter, warmup):
    if args.model in ['hivit_tiny', 'hivit_small', 'hivit_base']:
        # from timm.scheduler.cosine_lr import CosineLRScheduler
        optimizer = optim.AdamW(model.parameters(), lr=0.001, weight_decay=0.05)
        train_scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(
            optimizer, num_epoch, eta_min=0.0  # , last_epoch=classifier_epochs
        )
    elif args.model in ['atm_xt','atmf_xt','atmf_t','atmf_b']:
        optimizer = optim.AdamW(model.parameters(), lr=1e-3, weight_decay=0.05)
        train_scheduler = torch.optim.lr_scheduler.CosineAnnealingLR(
            optimizer, num_epoch, eta_min=0.0  # , last_epoch=classifier_epochs
        )
    else:
        milestones = {
            'mobilenetv3_small_100': [30, 50, 70],
            'mobilenetv3_small_50':  [30, 50, 70],
            'resnet18':              [30, 50, 70],
            'resnet34':              [30, 50, 70],
            'resnet50':              [30, 50, 70],
            'resnet152':             [30, 50, 70],
            'vgg11_bn':              [30, 50, 70],
            'vgg19_bn':              [30, 50, 70],
        }
        optimizer = optim.SGD(model.parameters(), lr=0.1, momentum=0.9, weight_decay=5e-4)
        train_scheduler = optim.lr_scheduler.MultiStepLR(
            optimizer, milestones=milestones[args.model], gamma=0.2
        )  # learning rate decay
    # warm scheduler
    warmup_scheduler = WarmUpLR(optimizer, math.ceil(num_iter / args.batch_step) * warmup)
    return optimizer, train_scheduler, warmup_scheduler


def main():
    parser = argparse.ArgumentParser(description='Setting for CIFAR100 by Adversarial Regularization')
    parser.add_argument('--cuda', type=int, default=0)
    parser.add_argument('--model', type=str, default='mobilenetv3_small_50')
    parser.add_argument('--classifier_epochs', type=int, default=200, help='classifier epochs')
    parser.add_argument('--attack_epochs', type=int, default=150, help='attack epochs in NN attack')
    parser.add_argument('--print_epoch', type=int, default=5,
                        help='print model training stats per print_epoch_splitai during splitai training')
    parser.add_argument('--batch_size', type=int, default=256, help='batch size')
    parser.add_argument('--batch_step', type=int, default=256, help='batch accumulation steps')
    parser.add_argument('--warmup', type=int, default=1, help='warm up epochs')
    parser.add_argument('--num_worker', type=int, default=2, help='number workers')
    parser.add_argument('--num_class', type=int, default=200, help='num class')
    parser.add_argument('--data_aug', type=distutils.util.strtobool, default=True, help='turn on data augmentation')
    parser.add_argument('--run_idx', type=int, default=100, help='idx running')
    parser.add_argument('--save_path', default='save_checkpoints/', type=str, help='folder to save the checkpoints')
    parser.add_argument('--load_path', default='save_checkpoints/', type=str, help='folder to load the checkpoints')
    # defence conf
    parser.add_argument('--alpha', type=float, default=2.0, help='para for Adversarial Regularization')
    parser.add_argument('--k', type=int, default=5, help='k steps for Adversarial Regularization')

    args = parser.parse_args()
    print(dict(args._get_kwargs()))

    global device
    cuda_id = args.cuda
    device = torch.device(f"cuda:{str(cuda_id)}" if torch.cuda.is_available() else "cpu")

    attack_epochs = args.attack_epochs
    batch_size = args.batch_size
    num_class = args.num_class
    classifier_epochs = args.classifier_epochs
    print_epoch = args.print_epoch
    warmup = args.warmup
    num_worker = args.num_worker
    alpha = args.alpha
    k = args.k

    DATASET_PATH = os.path.join(root_dir, 'tinyimagenet', 'data')
    checkpoint_path = os.path.join(args.save_path, 'tinyimagenet', args.model, f'advreg',
                                   'aug' if args.data_aug else 'no_aug', f'{str(int(args.alpha * 100))}_{k}', str(args.run_idx))
    load_checkpoint_path = os.path.join(args.load_path, 'tinyimagenet', args.model, 'undefend',
                                        'aug' if args.data_aug else 'no_aug', str(args.run_idx))
    print(checkpoint_path)

    train_data_tr_attack = np.load(os.path.join(DATASET_PATH, 'partition', 'tr_data.npy'))
    train_label_tr_attack = np.load(os.path.join(DATASET_PATH, 'partition', 'tr_label.npy'))
    train_data_te_attack = np.load(os.path.join(DATASET_PATH, 'partition', 'te_data.npy'))
    train_label_te_attack = np.load(os.path.join(DATASET_PATH, 'partition', 'te_label.npy'))
    train_data = np.load(os.path.join(DATASET_PATH, 'partition', 'train_data.npy'))
    train_label = np.load(os.path.join(DATASET_PATH, 'partition', 'train_label.npy'))
    test_data = np.load(os.path.join(DATASET_PATH, 'partition', 'test_data.npy'))
    test_label = np.load(os.path.join(DATASET_PATH, 'partition', 'test_label.npy'))
    ref_data = np.load(os.path.join(DATASET_PATH, 'partition', 'ref_data.npy'))
    ref_label = np.load(os.path.join(DATASET_PATH, 'partition', 'ref_label.npy'))
    all_test_data = np.load(os.path.join(DATASET_PATH, 'partition', 'all_test_data.npy'))
    all_test_label = np.load(os.path.join(DATASET_PATH, 'partition', 'all_test_label.npy'))

    # print first 20 labels for each subset, for checking with other experiments
    print(train_label_tr_attack[:20])
    print(train_label_te_attack[:20])
    print(test_label[:20])
    print(ref_label[:20])

    r = np.arange(len(train_data))
    np.random.shuffle(r)

    # if data augmented
    if args.data_aug:
        trainset = TINdata(train_data, train_label, transform_train_aug)
    else:
        trainset = TINdata(train_data, train_label, transform_train)
    # load dataset
    traintestset = TINdata(train_data, train_label, transform_test)

    testset = TINdata(test_data, test_label, transform_test)
    refset = TINdata(ref_data, ref_label, transform_test)
    alltestset = TINdata(all_test_data, all_test_label, transform_test)
    trset = TINdata(train_data_tr_attack, train_label_tr_attack, transform_test)
    teset = TINdata(train_data_te_attack, train_label_te_attack, transform_test)

    trainloader = torch.utils.data.DataLoader(trainset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    traintestloader = torch.utils.data.DataLoader(traintestset, batch_size=batch_size, shuffle=False,
                                                  num_workers=num_worker)
    testloader = torch.utils.data.DataLoader(testset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    refloader = torch.utils.data.DataLoader(refset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    alltestloader = torch.utils.data.DataLoader(alltestset, batch_size=batch_size, shuffle=False,
                                                num_workers=num_worker)
    trloader = torch.utils.data.DataLoader(trset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    teloader = torch.utils.data.DataLoader(teset, batch_size=batch_size, shuffle=False, num_workers=num_worker)

    best_acc = 0.00
    model_1 = get_network(arch=args.model, num_classes=num_class)
    model = model_1

    resume = load_checkpoint_path + '/model_last.pth.tar'
    print('==> Resuming from checkpoint' + resume)
    assert os.path.isfile(resume), 'Error: no checkpoint directory found!'
    checkpoint = torch.load(resume, map_location='cpu')
    model.load_state_dict(checkpoint['state_dict'])

    criterion = (nn.CrossEntropyLoss()).to(device, torch.float)
    model = model.to(device, torch.float)
    print("training sets: {:d}".format(len(train_data)))

    iter_per_epoch = len(trainloader)
    # optimizer = optim.SGD(model.parameters(), lr=0.1, momentum=0.9, weight_decay=5e-4)
    # train_scheduler = optim.lr_scheduler.MultiStepLR(optimizer, milestones= [60, 120, 160], gamma=0.2) #learning rate decay
    # warmup_scheduler = WarmUpLR(optimizer,  iter_per_epoch * args.warmup)
    optimizer, train_scheduler, warmup_scheduler = get_opt_and_lrsch(
        args, model, classifier_epochs, iter_per_epoch, warmup
    )

    attackset = AllInOne(
        train_data, train_label, all_test_data, all_test_label,
        transform=transform_train_aug if args.data_aug else transform_train, transform_test=transform_test, k=k
    )
    trainloader = torch.utils.data.DataLoader(trainset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    attackloader = torch.utils.data.DataLoader(attackset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
    print("attack set: {:d}".format(len(attackset)))

    dfs_freeze_ps_rewind_pr(model, False, args)

    trer = TrainRecorder()
    best_epoch = 0

    attack_model0 = InferenceAttack_HZ(num_class).to(device, torch.float)
    attack_criterion0 = nn.MSELoss().to(device, torch.float)
    attack_optimizer0 = optim.Adam(attack_model0.parameters(), lr=0.0001)

    for epoch in range(1, classifier_epochs + 1):

        # trainset.update()
        # attackset.update()
        # trainloader = torch.utils.data.DataLoader(trainset, batch_size=batch_size, shuffle=False, num_workers=num_worker)
        # attackloader = torch.utils.data.DataLoader(attackset, batch_size=batch_size, shuffle=False, num_workers=num_worker)

        if epoch > warmup:
            train_scheduler.step(epoch)

        if epoch <= 4:
            training_loss, training_acc, = train(trainloader, model, criterion, optimizer, epoch, warmup_scheduler, args)
        else:
            training_loss, training_acc, = train_privatly(
                attackloader, model, attack_model0,
                criterion, optimizer, attack_criterion0, attack_optimizer0,
                epoch, warmup_scheduler, args, alpha=alpha, k=k
            )

        # training_loss, training_acc = test(trainloader, model, criterion, batch_size)
        train_loss, train_acc = test(traintestloader, model, criterion, batch_size)
        test_loss, test_acc = test(testloader, model, criterion, batch_size)
        # record
        trer.update(train_loss, train_acc, test_loss, test_acc, training_loss, training_acc)

        # save model
        is_best = test_acc > best_acc
        best_acc = max(test_acc, best_acc)
        if is_best:
            best_epoch = epoch
            save_checkpoint({
                'epoch': epoch,
                'state_dict': model.state_dict(),
                'acc': test_acc,
                'best_acc': best_acc,
                'optimizer': optimizer.state_dict(),
            }, is_best, best_acc, checkpoint=checkpoint_path)

        # if (epoch)%print_epoch ==0:
        lr = get_learning_rate(optimizer)
        print('Epoch: [{:d} | {:d}]: learning rate:{:.4f}. acc: test: {:.4f}. loss: test: {:.4f}'.format(epoch,
                                                                                                         classifier_epochs,
                                                                                                         lr[0],
                                                                                                         test_acc,
                                                                                                         test_loss))
        sys.stdout.flush()

    # save the record
    trer.save(checkpoint_path, 'train_record.csv')
    # save the last
    save_checkpoint({
        'epoch': epoch,
        'state_dict': model.state_dict(),
        'acc': test_acc,
        'best_acc': best_acc,
        'optimizer': optimizer.state_dict(),
    }, False, best_acc, checkpoint=checkpoint_path)
    print("Final saved epoch {:d} with best acc {:.4f}".format(best_epoch, best_acc))


if __name__ == '__main__':
    main()
