''' VGG in PyTorch
[1] Simonyan, Karen, and Andrew Zisserman.
    Very deep convolutional networks for large-scale image recognition
    ICLR
    Modified from https://github.com/chengyangfu/pytorch-vgg-cifar10/blob/master/vgg.py
'''
import math
import sys

import torch
import torch.nn as nn

fc_in = 512


class ConvBlock(nn.Module):
    def __init__(self, num_layers, c_in, c_out, stride):
        super().__init__()
        self.feas = nn.ModuleList([
            nn.Sequential(*[
                nn.Conv2d(c_in if i == 0 else c_out, c_out, kernel_size=3, padding=1, stride=stride if i==0 else 1),
                nn.BatchNorm2d(c_out),
                nn.ReLU(inplace=True)
            ]) for i in range(num_layers)
        ])

        self.shortcut = nn.Sequential(
            nn.Conv2d(c_in, c_out, kernel_size=stride, padding=0),
            nn.BatchNorm2d(c_out),
            nn.ReLU(inplace=True)
        )

    def forward(self, x):
        s = self.shortcut(x)
        for layer in self.feas:
            x = layer(x)
        return x + s


class VGG(nn.Module):
    '''
    VGG model
    '''

    def __init__(self, features, num_classes=100):
        super(VGG, self).__init__()
        self.features = features
        self.classifier = nn.Sequential(
            nn.Dropout(),
            nn.Linear(fc_in, 512),
            nn.ReLU(True),
            nn.Dropout(),
            nn.Linear(512, 512),
        )
        self.final = nn.Sequential(
            nn.ReLU(True),
            nn.Linear(512, num_classes),
        )
        # Initialize weights
        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                n = m.kernel_size[0] * m.kernel_size[1] * m.out_channels
                m.weight.data.normal_(0, math.sqrt(2. / n))
                m.bias.data.zero_()

    def forward(self, x):
        x = self.features(x)
        x = x.view(x.size(0), -1)
        x = self.classifier(x)
        # fea = x
        x = self.final(x)
        return x  # , fea

    def multi_forward(self, x):
        feas = []
        for layer in self.features:
            x = layer(x)
            if isinstance(layer, nn.MaxPool2d):
                feas.append(x)
                # print(x.size())
        x = x.view(x.size(0), -1)
        x = self.classifier(x)
        x = self.final(x)
        return x, tuple(feas)

    def prepool_forward(self, x):
        feas = []
        for layer in self.features:
            if isinstance(layer, nn.MaxPool2d):
                feas.append(x)
            x = layer(x)
        x = x.view(x.size(0), -1)
        x = self.classifier(x)
        x = self.final(x)
        return x, tuple(feas)

    def last_forward(self, x):
        feas = []
        cnt = 0
        for layer in self.features:
            x = layer(x)
            if isinstance(layer, nn.MaxPool2d):
                cnt += 1
            if isinstance(layer, nn.Conv2d) and cnt >= 3:
                feas.append(x)
                # print(x.size())
        x = x.view(x.size(0), -1)
        x = self.classifier(x)
        x = self.final(x)
        return x, tuple(feas)


def make_layers(cfg, batch_norm=False):
    layers = []
    in_channels = 3
    for v in cfg:
        if v == 'M':
            layers += [nn.MaxPool2d(kernel_size=2, stride=2)]
        else:
            if len(layers) == 0:
                c_out, num_layers = v
                for i in range(num_layers):
                    layers += [
                        nn.Conv2d(in_channels if i==0 else c_out, c_out, kernel_size=3, stride=2 if i==0 else 1, padding=3 if i==0 else 1),
                        nn.BatchNorm2d(c_out),
                        nn.ReLU(inplace=True)
                    ]
            else:
                c_out, num_layers = v
                layers += [ConvBlock(num_layers, in_channels, c_out, 1)]
            in_channels = c_out
    global fc_in
    fc_in = in_channels
    return nn.Sequential(*layers)


cfg = {
    'A': [(64, 1), 'M', (128, 1), 'M', (256, 1), 'M', (512, 1), 'M', (512, 2), 'M'],
    'B': [(64, 2), 'M', (128, 2), 'M', (256, 2), 'M', (512, 2), 'M', (512, 2), 'M'],
    'D': [(64, 2), 'M', (128, 2), 'M', (256, 3), 'M', (512, 3), 'M', (512, 3), 'M'],
    'E': [(64, 2), 'M', (128, 2), 'M', (256, 4), 'M', (512, 4), 'M', (512, 4), 'M'],
}


def vgg11(num_classes):
    """VGG 11-layer model (configuration "A")"""
    return VGG(make_layers(cfg['A']), num_classes)


def vgg11_bn(num_classes):
    """VGG 11-layer model (configuration "A") with batch normalization"""
    return VGG(make_layers(cfg['A'], batch_norm=True), num_classes)


def vgg13(num_classes):
    """VGG 13-layer model (configuration "B")"""
    return VGG(make_layers(cfg['B']), num_classes)


def vgg13_bn(num_classes):
    """VGG 13-layer model (configuration "B") with batch normalization"""
    return VGG(make_layers(cfg['B'], batch_norm=True), num_classes)


def vgg16(num_classes):
    """VGG 16-layer model (configuration "D")"""
    return VGG(make_layers(cfg['D']), num_classes)


def vgg16_bn(num_classes):
    """VGG 16-layer model (configuration "D") with batch normalization"""
    return VGG(make_layers(cfg['D'], batch_norm=True), num_classes)


def vgg19(num_classes):
    """VGG 19-layer model (configuration "E")"""
    return VGG(make_layers(cfg['E']), num_classes)


def vgg19_bn(num_classes):
    """VGG 19-layer model (configuration 'E') with batch normalization"""
    return VGG(make_layers(cfg['E'], batch_norm=True), num_classes)


def vgg(num_classes, arch='vgg16'):
    if arch == 'svgg11_bn':
        return vgg11_bn(num_classes)
    elif arch == 'svgg13_bn':
        return vgg13_bn(num_classes)
    elif arch == 'svgg16_bn':
        return vgg16_bn(num_classes)
    elif arch == 'svgg19_bn':
        return vgg19_bn(num_classes)
    else:
        print_not_support(arch)


def print_not_support(arch):
    """ return unsupported info
    """
    print('the network {} you have entered is not supported yet'.format(arch))
    sys.exit()


if __name__ == '__main__':
    net = vgg(num_classes=200, arch='vgg11_bn')
    print(net)
    i = torch.zeros(64, 3, 64, 64)
    o = net(i)
    print(o.size())
    o, (fea1, fea2, fea3, fea4, fea5) = net.multi_forward(i)
    print(o.size(), fea1.size(), fea2.size(), fea3.size(), fea4.size())
    #o, (fea1, fea2, fea3, fea4) = net.last_forward(i)
    #print(o.size(), fea1.size(), fea2.size())
