from __future__ import print_function
import os
import argparse
import wandb
import pandas as pd
import torch
import torch.nn as nn
import torch.nn.functional as F
import torchvision
import torch.optim as optim
from torchvision import datasets, transforms

from models.wideresnet import *
from models.resnet import *
from trades import trades_loss
import os
os.environ['CUDA_VISIBLE_DEVICES'] = '5cd ..' 
device = torch.device("cuda")

parser = argparse.ArgumentParser(description='PyTorch CIFAR TRADES Adversarial Training')
parser.add_argument('--batch-size', type=int, default=128, metavar='N',
                    help='input batch size for training (default: 128)')
parser.add_argument('--test-batch-size', type=int, default=128, metavar='N',
                    help='input batch size for testing (default: 128)')
parser.add_argument('--epochs', type=int, default=100, metavar='N',
                    help='number of epochs to train')
parser.add_argument('--weight-decay', '--wd', default=2e-4,
                    type=float, metavar='W')
parser.add_argument('--lr', type=float, default=0.1, metavar='LR',
                    help='learning rate')
parser.add_argument('--momentum', type=float, default=0.9, metavar='M',
                    help='SGD momentum')
parser.add_argument('--no-cuda', action='store_true', default=False,
                    help='disables CUDA training')
parser.add_argument('--epsilon', default=0.031,
                    help='perturbation')
parser.add_argument('--num-steps', default=10,
                    help='perturb number of steps')
parser.add_argument('--step-size', default=0.007,
                    help='perturb step size')
parser.add_argument('--beta', default=6.0, type=float,
                    help='regularization, i.e., 1/lambda in TRADES')
parser.add_argument('--seed', type=int, default=1, metavar='S',
                    help='random seed (default: 1)')
parser.add_argument('--log-interval', type=int, default=100, metavar='N',
                    help='how many batches to wait before logging training status')
parser.add_argument('--model-dir', default='./model-cifar-wideResNet',
                    help='directory of model for saving checkpoint')
parser.add_argument('--save-freq', '-s', default=20, type=int, metavar='N',
                    help='save frequency')
parser.add_argument('--layer_snr_weight_type', default="exp", type=str,
                    help='type of layer snr weight')
parser.add_argument('--snr_type', default="n2sv", type=str,
                help='type of snr loss')
parser.add_argument('--base', default=1.5, type=float,
                        help='base number of ')
parser.add_argument('--use_snr', action='store_true',
                    help='use snr loss or not')
parser.add_argument('--use_adv', action='store_true',
                    help='use adv loss or not')
parser.add_argument('--snr_layers', default=None, nargs='+', type=str,
                        help='the layers need to compute snr')
parser.add_argument('--snr_weight', default=10, type=float,
                    help='weight of snr loss')
parser.add_argument('--name', default="", type=str,
                        help='weight of snr loss')
parser.add_argument('--resume', default=None, type=str,
                        help='weight of snr loss')
parser.add_argument('--opt_resume', default=None, type=str,
                        help='weight of snr loss')

args = parser.parse_args()

# settings
model_dir = args.model_dir
if not os.path.exists(model_dir):
    os.makedirs(model_dir)
use_cuda = not args.no_cuda and torch.cuda.is_available()
torch.manual_seed(args.seed)
device = torch.device("cuda" if use_cuda else "cpu")
kwargs = {'num_workers': 1, 'pin_memory': True} if use_cuda else {}

# setup data loader
transform_train = transforms.Compose([
    transforms.RandomCrop(32, padding=4),
    transforms.RandomHorizontalFlip(),
    transforms.ToTensor(),
])
transform_test = transforms.Compose([
    transforms.ToTensor(),
])
trainset = torchvision.datasets.CIFAR10(root='../data', train=True, download=True, transform=transform_train)
train_loader = torch.utils.data.DataLoader(trainset, batch_size=args.batch_size, shuffle=True, **kwargs)
testset = torchvision.datasets.CIFAR10(root='../data', train=False, download=True, transform=transform_test)
test_loader = torch.utils.data.DataLoader(testset, batch_size=args.test_batch_size, shuffle=False, **kwargs)


def train(args, model, device, train_loader, optimizer, epoch):
    model.train()
    metrics = pd.DataFrame()
    for batch_idx, (data, target) in enumerate(train_loader):
        data, target = data.to(device), target.to(device)

        optimizer.zero_grad()

        # calculate robust loss
        loss, batch_metrics = trades_loss(model=model,
                           x_natural=data,
                           y=target,
                           optimizer=optimizer,
                           step_size=args.step_size,
                           epsilon=args.epsilon,
                           perturb_steps=args.num_steps,
                           beta=args.beta,
                           args=args)
        loss.backward()
        optimizer.step()

        metrics = metrics.append(pd.DataFrame(batch_metrics, index=[0]), ignore_index=True)
        # print progress
        if batch_idx % args.log_interval == 0:
            print('Train Epoch: {} [{}/{} ({:.0f}%)]\tLoss: {:.6f}'.format(
                epoch, batch_idx * len(data), len(train_loader.dataset),
                       100. * batch_idx / len(train_loader), loss.item()))    
    return dict(metrics.mean())



def eval_train(model, device, train_loader):
    model.eval()
    train_loss = 0
    correct = 0
    with torch.no_grad():
        for data, target in train_loader:
            data, target = data.to(device), target.to(device)
            output = model(data)
            train_loss += F.cross_entropy(output, target, size_average=False).item()
            pred = output.max(1, keepdim=True)[1]
            correct += pred.eq(target.view_as(pred)).sum().item()
    train_loss /= len(train_loader.dataset)
    print('Training: Average loss: {:.4f}, Accuracy: {}/{} ({:.0f}%)'.format(
        train_loss, correct, len(train_loader.dataset),
        100. * correct / len(train_loader.dataset)))
    training_accuracy = correct / len(train_loader.dataset)
    return train_loss, training_accuracy


def eval_test(model, device, test_loader):
    model.eval()
    test_loss = 0
    correct = 0
    with torch.no_grad():
        for data, target in test_loader:
            data, target = data.to(device), target.to(device)
            output = model(data)
            test_loss += F.cross_entropy(output, target, size_average=False).item()
            pred = output.max(1, keepdim=True)[1]
            correct += pred.eq(target.view_as(pred)).sum().item()
    test_loss /= len(test_loader.dataset)
    print('Test: Average loss: {:.4f}, Accuracy: {}/{} ({:.0f}%)'.format(
        test_loss, correct, len(test_loader.dataset),
        100. * correct / len(test_loader.dataset)))
    test_accuracy = correct / len(test_loader.dataset)
    return test_loss, test_accuracy


def adjust_learning_rate(optimizer, epoch):
    """decrease the learning rate"""
    lr = args.lr
    if epoch >= 75:
        lr = args.lr * 0.1
    if epoch >= 90:
        lr = args.lr * 0.01
    if epoch >= 100:
        lr = args.lr * 0.001  
    for param_group in optimizer.param_groups:
        param_group['lr'] = lr


def main():
    # init model, ResNet18() can be also used here for training
    model = WideResNet().to(device)
    wandb.init(project="unsupervised-robust-learning", name=args.name)
    #model = ResNet18().to(device)
    wandb.watch(model)
    optimizer = optim.SGD(model.parameters(), lr=args.lr, momentum=args.momentum, weight_decay=args.weight_decay)
    if args.resume: 
        start_epoch = 81
        checkpoint = torch.load(args.resume, map_location=torch.device('cpu'))
        model.load_state_dict(checkpoint)
        opt_checkpoint = torch.load(args.opt_resume, map_location=torch.device('cpu'))
        optimizer.load_state_dict(opt_checkpoint)
    else: 
        start_epoch = 1

    for epoch in range(start_epoch, args.epochs + 1):
        # adjust learning rate for SGD
        adjust_learning_rate(optimizer, epoch)

        # adversarial training
        res = train(args, model, device, train_loader, optimizer, epoch)

        # evaluation on natural examples
        print('================================================================')
        #train_loss, training_accuracy = eval_train(model, device, train_loader)
        test_loss, test_accuracy = eval_test(model, device, test_loader)
        print('================================================================')

        # save checkpoint
        if epoch % args.save_freq == 0:
            torch.save(model.state_dict(),
                       os.path.join(model_dir, 'model-wideres-epoch{}.pt'.format(epoch)))
            torch.save(optimizer.state_dict(),
                       os.path.join(model_dir, 'opt-wideres-checkpoint_epoch{}.tar'.format(epoch)))
            
        if args.use_snr:
            wandb.log({  
                        "epoch": epoch,
                        "lr": optimizer.param_groups[0]['lr'],
                        "train_clean_acc" : res['clean_acc'],
                        "test_clean_acc": test_accuracy,
                        "train_adversarial_acc" : res['adversarial_acc'],
                        "loss":res['loss'],
                        "test_loss":test_loss,
                        "loss_natural":res['loss_natural'],
                        "loss_robust":res['loss_robust'],
                        "snr_loss":res['snr_loss'],
                        'layer35_snr':res['layer1_snr'],
                        'layer36_snr':res['layer2_snr'],
                        'layer37_snr':res['layer3_snr'],
                        'layer38_snr':res['layer4_snr'],
                        'layer39_snr':res['layer5_snr'],  
                        'layer40_snr':res['layer6_snr'],
                        'layer41_snr':res['layer7_snr'],
                        'layer42_snr':res['layer8_snr'],
                        'layer43_snr':res['layer9_snr'],
                        'layer44_snr':res['layer10_snr'],
                        'layer45_snr':res['layer11_snr'],
                        'layer46_snr':res['layer12_snr']
                        })
        else:
            wandb.log({
                        "epoch": epoch,
                        "lr": optimizer.param_groups[0]['lr'],
                        "train_clean_acc" : res['clean_acc'],
                        "test_clean_acc": test_accuracy,
                        "train_adversarial_acc" : res['adversarial_acc'],
                        "loss":res['loss'],
                        "test_loss":test_loss,
                        "loss_natural":res['loss_natural'],
                        "loss_robust":res['loss_robust']
                        }) 


if __name__ == '__main__':
    main()
