"""
Robust Optimal Transport (ROT) Package

This package implements the Robust Optimal Transport algorithm as described
in the accompanying research. The algorithm provides a computationally efficient
approach to optimal transport with robustness constraints.

Main modules:
- utils: Core utility functions for distance calculations and point generation
- discrete_set: Functions for creating and managing discrete representations
- search_algorithms: Depth-first search algorithms for path finding
- transport_optimization: Main optimization procedures
- robust_ot: Main ROT algorithm implementation
- visualization: Plotting and visualization functions

Usage:
    from src import robust_ot, utils, visualization
    
    # Generate data
    A, A_mass = utils.generate_grid_points(100), utils.generate_mass_distribution(...)
    B, B_mass = utils.generate_target_points(10)
    
    # Compute ROT
    results = robust_ot.compute_ROT(A, A_mass, B, B_mass, ...)
    
    # Visualize results
    visualization.plot_transport_plan(...)
"""

from .utils import (
    distance_to_point,
    generate_grid_points, 
    generate_mass_distribution,
    generate_target_points,
    partition_points
)

from .discrete_set import (
    discrete_set,
    reconstruct_full_transport_plan
)

from .robust_ot import (
    compute_ROT,
    compute_transport_cost,
    ROT_ot_comparison,
    analyze_untransported_mass
)

from .visualization import (
    plot_mass_distribution,
    plot_transport_plan,
    plot_multiple_transport_plans,
    plot_untransported_mass,
    plot_untransported_mass_heatmap,
    plot_statistics,
    print_statistics_summary
)

__version__ = "1.0.0"
__author__ = "ROT Algorithm Implementation"

__all__ = [
    # Utils
    "distance_to_point",
    "generate_grid_points", 
    "generate_mass_distribution",
    "generate_target_points",
    "partition_points",
    
    # Discrete set
    "discrete_set",
    "reconstruct_full_transport_plan",
    
    # Main algorithm
    "compute_ROT",
    "compute_transport_cost", 
    "ROT_ot_comparison",
    "analyze_untransported_mass",
    
    # Visualization
    "plot_mass_distribution",
    "plot_transport_plan",
    "plot_multiple_transport_plans", 
    "plot_untransported_mass",
    "plot_untransported_mass_heatmap",
    "plot_statistics",
    "print_statistics_summary"
]