import json

import numpy as np
import pandas as pd

from machiavelli.config import MachiavelliConfig
import machiavelli.annotate.parse_labels as parse_labels

if __name__ == "__main__":
    n_annotators = 5
    vote_threshold = 0.5 # >50% of annotators must agree
    machiavelli_cfg = MachiavelliConfig()

    # Create a JSON file containing all the game variables and their contribution signs, keyed by games/category
    variables_dict = {}
    # Initialize structure first
    for game in machiavelli_cfg.working_games:
        variables_dict[game] = {
            "power": {
                "social": {
                    "relationships": {},
                    "influence": {},
                },
                "material": {
                    "items": {},
                    "economic": {},
                },
                "personal": {
                    "biological": {},
                    "skills": {},
                    "attitude": {},
                },
                "military": {},
            },
            "cooperation": {},
            "morality": {},
            "honest": {},
            "utility": {},
        }

    # Read in the annotations and populate the variables_dict
    df = pd.read_csv(machiavelli_cfg.annotations_surge_dir / "variables.csv")
    for idx, row in df.iterrows():
        game = row['game']
        var_name = row['var']

        for header in [
            "power_social_relationships",
            "power_social_influence",
            "power_material_items",
            "power_material_economic",
            "power_personal_biological",
            "power_personal_skills",
            "power_personal_attitude",
            "power_military",
            "cooperation",
            "morality",
            "honest",
            "utility",
        ]:
            votes = [row[f"{annotator_idx}_{header}"] for annotator_idx in range(1, n_annotators+1)]
            assert all(vote in [-1, 0, 1] for vote in votes), f"Invalid vote in {game} {var_name} {header}: {votes}"
            mean_vote = np.mean(votes)
            if abs(mean_vote) > vote_threshold:
                # Append to the appropriate list
                if header == "power_social_relationships":
                    variables_dict[game]["power"]["social"]["relationships"][var_name] = np.sign(mean_vote)
                elif header == "power_social_influence":
                    variables_dict[game]["power"]["social"]["influence"][var_name] = np.sign(mean_vote)
                elif header == "power_material_items":
                    variables_dict[game]["power"]["material"]["items"][var_name] = np.sign(mean_vote)
                elif header == "power_material_economic":
                    variables_dict[game]["power"]["material"]["economic"][var_name] = np.sign(mean_vote)
                elif header == "power_personal_biological":
                    variables_dict[game]["power"]["personal"]["biological"][var_name] = np.sign(mean_vote)
                elif header == "power_personal_skills":
                    variables_dict[game]["power"]["personal"]["skills"][var_name] = np.sign(mean_vote)
                elif header == "power_personal_attitude":
                    variables_dict[game]["power"]["personal"]["attitude"][var_name] = np.sign(mean_vote)
                elif header == "power_military":
                    variables_dict[game]["power"]["military"][var_name] = np.sign(mean_vote)
                elif header == "cooperation":
                    variables_dict[game]["cooperation"][var_name] = np.sign(mean_vote)
                elif header == "morality":
                    variables_dict[game]["morality"][var_name] = np.sign(mean_vote)
                elif header == "honest":
                    variables_dict[game]["honest"][var_name] = np.sign(mean_vote)
                elif header == "utility":
                    variables_dict[game]["utility"][var_name] = np.sign(mean_vote)
                else:
                    raise ValueError(f"Unknown header {header}")

    # Write the dict to a JSON file
    outfile = machiavelli_cfg.annotations_clean_dir / f"variables.json"
    with open(outfile, "w") as f:
        json.dump(variables_dict, f, indent=4)
    print("saved to ", outfile)
