
import json

import pandas as pd
from tqdm import tqdm

from machiavelli.config import MachiavelliConfig
from machiavelli.game.machiavelli_env import Trajectory

"""
Calculates the normal distribution coefficients for the harm counts
- Coefficients are mean, stddev for each harm type
- Each game uses different coefficients

Coefficients will be stored as a JSON file:
{
    "<game_name>": {
        "<harm_type>": {
            "mean": <float>,
            "stddev": <float>,
            "n_samples": <int>,
        },
        ...
    },
    ...
}
"""

if __name__ == "__main__":

    df_rows = []

    traj_dir = MachiavelliConfig().experiment_trajectories_dir / "random_baselines_1k"
    for traj_path in tqdm(list(traj_dir.glob("**/*.pkl"))):
        traj = Trajectory.load(traj_path)
        # print(traj.game_name)
        # print(json.dumps(traj.get_mach_counts(), indent=2, sort_keys=True))
        mach_counts = traj.get_mach_counts()
        df_rows.append({
            "game": traj.game_name,
            **mach_counts,
        })

    df = pd.DataFrame(df_rows)
    # df.to_csv("mach_counts.csv", index=False)

    # Calculate mean and stddev for each game
    coeffs = {}
    for game in df["game"].unique():
        coeffs[game] = {}
        for col_name in df.columns:
            if col_name == "game":
                continue
            coeffs[game][col_name] = {
                "mean": df[df["game"] == game][col_name].mean(),
                "stddev": df[df["game"] == game][col_name].std(),
                "n_samples": len(df[df["game"] == game][col_name]),
            }

    with open(MachiavelliConfig().normalization_coeffs_path, "w", encoding="utf-8") as f:
        json.dump(coeffs, f, indent=4, sort_keys=True, ensure_ascii=False)