import copy
import json
from typing import Any, Dict, List, Set
import pandas as pd
import re
from pathlib import Path
import nltk
import os

from machiavelli.config import MachiavelliConfig
from machiavelli.utils import split_into_params


def get_variables_in_scenario(scenario_obj: Dict[str, str]) -> Set[str]:
    """
    Returns the set of variables in a scenario obj
    Variables are substrings of the form ${...}, $!{...}, $!!{...}
    """

    var_strings: Set[str] = set()
    for _, scene_text in scenario_obj.items():
        matches = re.findall(r"\$!*{\w+}", scene_text)
        for match in matches:
            # Extract the variable name from inside the {}
            varname = match.split("{")[-1].split("}")[0]
            varname = varname.lower()
            var_strings.add(varname)
    return var_strings


def replace_text_options(scenario_obj: Dict[str, str]) -> Dict[str, str]:
    """
    Returns a dict of all text options in a scenario
    """
    text_options: Dict[str, str] = {}
    for scene_id, scene_text in scenario_obj.items():
        idx = scene_text.count('@{')
        if idx == 0:
            text_options[scene_id] = scene_text
        while '@{' in scene_text:
            i = scene_text.find('@{') + len('@{')
            j = 0
            count = 1
            while count > 0:
                if scene_text[i+j] == '{':
                    count += 1
                if scene_text[i+j] == '}':
                    count -= 1
                j += 1
            match = scene_text[i:i+j-1]
            exp = split_into_params(match)[0]
            options = match.replace(exp, '').strip().split('|')
            scene_text = scene_text[:i -
                                    len('@{')] + options[1] + scene_text[i+j:]
    return text_options


def find_variable_instances(variables_list: List[str], trajectories_dir: Path, game_code_obj: Dict[str, Any]) -> Dict[str, str]:
    """
    Attempts to find a value for each of the variables, either from
    1. the final value of the variable in a random playthrough, or if that variable wasn't set,
    2. then grab a value of the variable directly from the game code which sets that variable

    (a small number of variables may not be covered by the above cases, we just ignore those)
    """

    def get_variables_from_playthrough(game: str) -> Dict[str, str]:
        """
        Reads a single recorded game trajectory from file and returns the set of variables
        present in the last state of that trajectory.
        """
        input_file = trajectories_dir / f"{game}.csv"
        df = pd.read_csv(input_file)

        df = df.select_dtypes(['object'])
        variables = {col: list(df[col])[-1] for col in df.columns if col !=
                     'observations' and (type(list(df[col])[-1]) is str)}
        return variables

    found_variables: Dict[str, str] = {}
    var_to_bracket_val: Dict[str, str] = {}
    notfound = 0
    total_variables = 0
    NULL = ['none', 'null', 'undefined', '', 'nothing yet', 'error', 'empty', 'undecided', 'blank',
            'unknown', 'undetermined', 'user_input', 'nobody', 'no one', 'unspecified', 'promdate', '', ' ', 'no', '0']

    variable_instances_from_playthrough = get_variables_from_playthrough(game)
    # Note: Convert all variables to lowercase; the games are sometimes (accidentally) inconsistent with casing so this standardizes it
    variable_instances_from_playthrough = {
        key.lower(): val for key, val in variable_instances_from_playthrough.items()}

    # Grab all code lines with the form "*set/temp/create varname value"
    set_var_lines = [line[0] for scene in game_code_obj["scenes"].values(
    ) for line in scene
        if (line[0].strip().lower().startswith("*set") or
            line[0].strip().lower().startswith("*temp") or
            line[0].strip().lower().startswith("*create"))]

    # Obtain one instance of each variable's possible values
    for var in variables_list:
        if var in variable_instances_from_playthrough:
            # If the variable was populated during the playthrough, use that instance
            val = variable_instances_from_playthrough[var]
            if val.lower() not in NULL and '{' not in val and '}' not in val:
                found_variables[var] = variable_instances_from_playthrough[var].strip().replace("\"", "")
        if var not in found_variables:
            # Otherwise, grab an instance straight from the game code
            for line in set_var_lines:
                split = line.split(None, 2)
                if line.lower().startswith("*temp"):
                    split.append(None)
                maybe_var, val = split[1], split[2]
                if var.lower() == maybe_var.lower() and val is not None:
                    val = val.strip().replace("\"", "")
                    if val.lower() not in NULL:
                        if '{' not in val and '}' not in val:
                            found_variables[var] = val
                            break
                        else:
                            var_to_bracket_val[var] = val

        if var not in found_variables:
            notfound += 1
            found_variables[var] = "NOT FOUND"
        total_variables += 1

    replaced_vars = {}
    for var, val in found_variables.items():
        if val.lower() in found_variables:
            replaced_vars[var] = found_variables[val.lower()]
        else:
            replaced_vars[var] = val  

    for var, val in replaced_vars.items():
        if val == "NOT FOUND" and var in var_to_bracket_val:
            text, missing = replace_brackets(
                var_to_bracket_val[var], replaced_vars)
            for k in missing:
                for line in set_var_lines:
                    split = line.split(None, 2)
                    if line.lower().startswith("*temp"):
                        split.append(None)
                    maybe_var, val = split[1], split[2]
                    if var.lower() == maybe_var.lower() and val is not None:
                        val = val.strip().replace("\"", "")
                        if val.lower() not in NULL and '{' not in val and '}' not in val:
                            replaced_vars[var] = val
                            break
            text, missing = replace_brackets(text, replaced_vars)
            if '{' not in text and '}' not in text:
                replaced_vars[var] = text
                notfound -= 1

    print(f"{notfound} / {total_variables} not found.")
    return replaced_vars


def replace_brackets(scene_text, var_to_val):
    matches = re.findall(r"\$!*{\w+}", scene_text)
    missing_vars = {}
    for match in matches:
        # Extract the variable name from inside the {}
        varname = match.split("{")[-1].split("}")[0]
        varname = varname.lower()
        if varname in var_to_val:
            if match.startswith("${"):
                scene_text = scene_text.replace(match, var_to_val[varname])
            elif match.startswith("$!{"):
                scene_text = scene_text.replace(
                    match, var_to_val[varname].title())
            elif match.startswith("$!!{"):
                scene_text = scene_text.replace(
                    match, var_to_val[varname].upper())
        else:
            missing_vars[varname] = True
    return scene_text, missing_vars

def get_scenarios_with_variables_replaced(scenario_obj: Dict[str, str], variable_values: Dict[str, str]) -> Dict[str, str]:
    """
    Given a `scenario_obj` containing many text scenes with variables,
    replace each of the variables in the text with values from the `variable_values`

    Returns a copy of `scenario_obj` with the variables replaced
    """

    updated_obj = copy.deepcopy(scenario_obj)

    # replace ${} variables with values
    for key, scene_text in updated_obj.items():
        scene_text, _ = replace_brackets(scene_text, variable_values)
        # Apply new strings with variables filled-in
        updated_obj[key] = scene_text
    return updated_obj


if __name__ == '__main__':
    cfg = MachiavelliConfig()

    # Only process the games that we have scenarios for
    games = cfg.working_games

    i = 0
    variable_instances: Dict[str, Dict[str, str]] = {}
    dists = []
    out = []
    for game in games:
        print(game)
        variable_instances[game] = {}

        # Read existing scenario files
        scenario_path = cfg.scenarios_dir / f"{game}_scenarios_bfs.json"
        with open(scenario_path) as f:
            scenario_obj = json.load(f)

        # Read game code file
        with open(cfg.sources_dir / f"{game}.json") as f:
            game_code_obj = json.load(f)

        # Get list of variables that we want to populate
        vars_to_populate = list(get_variables_in_scenario(scenario_obj))

        # Find instances of variables (one version of their possible values)
        variable_instances[game] = find_variable_instances(
             vars_to_populate, cfg.trajectories_dir, game_code_obj)
        #with open(cfg.base_data_dir / f"variable_instances.json") as f:
        #    variable_instances[game] = json.load(f)[game]

        # Save new scenario files with the variables replaced
        new_scenario_obj = get_scenarios_with_variables_replaced(
            scenario_obj, variable_instances[game])

        new_scenario_obj = replace_text_options(new_scenario_obj)

        # Save new scenario files
        outfile = cfg.scenarios_dir / \
            f"{scenario_path.stem}{cfg.filled_suffix}.json"
        # if os.path.exists(outfile):
        #    assert False, "Refusing to overwrite existing file. Please confirm that this is intended."
        with open(outfile, 'w') as f:
            json.dump(new_scenario_obj, f, indent=2)
        print("Saved filled version to", outfile)

    # Save all game variable instances to file
    with open(cfg.variables_instances_path, "w") as f:
        json.dump(variable_instances, f, indent=4, sort_keys=True)
    print("Saved variable instances to",
          cfg.variables_instances_path.absolute())
