import copy
import numbers
import re
import random
import time
import numpy as np
from collections import defaultdict

from machiavelli.utils import get_conditional_group, get_nested_group, \
                       make_lowercase, make_underscore, split_into_params, \
                       load_game_data, load_game_tree

from asteval import Interpreter


class Player:
    def __init__(self, game, data=None, print_text=False, buffered_print=False, watch_history=False):
        self.print_text = print_text
        data = data if data else load_game_data(game)
        self.title = data['metadata']['title']
        self.author = data['metadata']['author']
        self.scene_list = [s.lower() for s in data['scene_list']]
        self.scenes = data['scenes']
        self.label_to_linenum = data['label_to_linenum']
        self.kw_vars = "|".join(data['kw_vars'].keys())
        self.vars_to_resources = defaultdict(lambda: 1)  # TODO: load data
        self.END_TEXT = "THE END!"

        self.scene_line_to_node = defaultdict(dict)
        tree = load_game_tree(game)
        for name in sorted(tree):
            node = tree[name]
            scene = node['scene']
            for i in node['lines']:
                if i in self.scene_line_to_node[scene]:
                    prev_node = tree[self.scene_line_to_node[scene][i][-1]]
                    if prev_node['type'] == 'COND':
                        self.scene_line_to_node[scene][i] = [name]
                    else:
                        self.scene_line_to_node[scene][i].append(name)
                else:
                    self.scene_line_to_node[scene][i] = [name]

        # Hard code some special cases
        if self.title == 'The Martian Job':
            for scene, lines in self.scenes.items():
                for i, (l, wspace) in enumerate(lines):
                    l_ = re.sub(r'"\$\{([^\}]+)\}"', lambda m: m.group(1), l)
                    lines[i] = [l_, wspace]

        self.int_maybe_string = ['SLAMMED!',
                                 'Petal Throne',
                                 'Choice of the Pirate',
                                 'I, Cyborg',
                                 'The Hero Unmasked!']

        # Buffered printing for controlled interaction with the outputs
        self.buffered_print = buffered_print
        self.print_buffer = ""

        self.watch_history = watch_history
        self.reset()

    def reset(self):
        self.curr_scene = 'startup'
        self.idx = 0
        self.choice_lines = []
        self.choice_texts = []
        self.scene_hist = []
        self.node_hist = ['startup_0']
        self.discarded_choices = {}
        self.hide_reuse = False
        # Dict of achievement_name -> (visible, points, name, locked_text, unlocked_text)
        self.achievements = {}
        self.unlocked_achievements = []
        self.done = False

        self.user_vars = {}
        self.scene_to_tempvars = defaultdict(dict)
        self.aeval = Interpreter(usersyms={'round_to_int': lambda x: round(x)})
        self.save_data = None

        # Hard-code some useless variables
        self.aeval.symtable['choice_is_web'] = True
        self.aeval.symtable['choice_is_steam'] = False
        self.aeval.symtable['choice_is_omnibus_app'] = False
        self.aeval.symtable['choice_is_ios_app'] = False

        self.aeval.symtable['choice_save_allowed'] = False
        self.aeval.symtable['choice_user_restored'] = True
        self.aeval.symtable['choice_subscribe_allowed'] = False
        self.aeval.symtable['choice_randomtest'] = False
        self.aeval.symtable['choice_quicktest'] = False
        self.aeval.symtable['choice_prerelease'] = False
        self.aeval.symtable['choice_nightmode'] = False

        self.aeval.symtable['choice_purchased_adfree'] = True
        self.aeval.symtable['choice_purchased_adspace'] = True
        self.aeval.symtable['choice_purchased_supported'] = True
        self.aeval.symtable['choice_purchase_supported'] = True
        self.aeval.symtable['choice_is_advertising_supported'] = False
        self.aeval.symtable['choice_is_trial'] = False

        # TODO: fix choice_purchased
        self.aeval.symtable['choice_purchased_cheats'] = False
        self.aeval.symtable['choice_purchased_hints'] = False
        self.aeval.symtable['choice_purchased_seal'] = False
        self.aeval.symtable['choice_purchased_legendguide'] = False
        self.aeval.symtable['choice_purchased_warningsys'] = False
        self.aeval.symtable['choice_purchased_extraworning'] = False
        self.aeval.symtable['choice_purchased_youpower'] = False
        self.aeval.symtable['choice_purchased_warningsysnine'] = False
        self.aeval.symtable['choice_purchased_commentary'] = False
        self.aeval.symtable['choice_purchased_characterdatabase'] = False
        self.aeval.symtable['choice_purchased_fatedbabel'] = False
        self.aeval.symtable['choice_purchased_loveandshadows'] = False
        self.aeval.symtable['choice_purchased_motivationboost'] = False
        #########
        self.aeval.symtable['choice_purchased_full'] = True
        self.aeval.symtable['choice_purchased_usurpersoutcasts'] = True
        self.aeval.symtable['choice_purchased_secretsandshadows'] = True
        self.aeval.symtable['choice_purchased_intrigues'] = True
        self.aeval.symtable['choice_purchased_tildeath'] = True
        self.aeval.symtable['choice_purchased_consort'] = True
        self.aeval.symtable['choice_purchased_lifemage'] = True
        self.aeval.symtable['choice_purchased_juanita'] = True
        self.aeval.symtable['choice_purchased_memphis'] = True
        self.aeval.symtable['choice_purchased_stlouis'] = True

        # Hard-code some missing variables
        if self.title == 'T-Rex Time Machine':
            self.aeval.symtable['nobodyXhis'] = ""
            self.aeval.symtable['nobodyXhiss'] = ""
            self.aeval.symtable['nobodyXhe'] = ""
            self.aeval.symtable['nobodyXhim'] = ""

    @property
    def lines(self):
        return self.scenes[self.curr_scene]

    ### Control flow ###
    def act(self, action):
        assert action >= 0 and action < len(self.choice_lines)
        if self.print_text:
            self.print(f'@@@ {self._process_text(self.choice_texts[action])}')
        self.idx, choice_idx = self.choice_lines[action]
        self.record()
        name = f'{self.curr_scene}_{choice_idx}'
        if name in self.discarded_choices:
            self.discarded_choices[name] = True
        self.choice_lines = []
        self.choice_texts = []

    def play(self, action=None):
        text = self.state()
        if len(self.choice_lines) > 0:
            if action is None:
                for idx, choice_text in enumerate(text):
                    self.print(f'{idx}: {self._process_text(choice_text)}')
                return 1
            self.act(action)
        elif text is not None:
            self.print(self._process_text(text))
            if text == self.END_TEXT:
                self.done = True
                return -1
        return 0

    def record(self):
        if self.watch_history:
            curr_nodes = self.get_node()
            for node in curr_nodes:
                if self.node_hist[-1] != node:
                    self.node_hist.append(node)

    def simulate_choices(self, next_req_nodes=None, max_paths=1e4, search_depth=12, slack=3):
        """
        Return (trajectory, end_state) for each choice at the current scene.
        """
        assert len(self.choice_lines) > 0
        old_state = self._save_state(save_pos=True)
        res = []
        possible_paths = [([], old_state.copy())]
        max_len = np.inf
        num_paths = 0
        while len(possible_paths) > 0:
            path, temp_state = possible_paths.pop()
            self._load_state(temp_state.copy())
            for i in range(len(self.choice_lines)):
                self._load_state(temp_state.copy())
                self.act(i)
                _, _, state = self.skip_to_next_choice()
                if next_req_nodes is None:
                    if len(path)+1 == search_depth:
                        num_paths += 1
                        if num_paths < max_paths:
                            res.append(([*path, i], state.copy()))
                    elif len(path)+1 < search_depth:
                        possible_paths.append(([*path, i], state.copy()))
                else:
                    if not set(self.get_history()).isdisjoint(set(next_req_nodes)):
                        if len(res) == 0:
                            max_len = len(path) + slack
                        res.append(([*path, i], state.copy()))
                    else:
                        if self.curr_scene[:3] == next_req_nodes[0][:3] and \
                                len(path) < min(max_len, search_depth):
                            num_paths += 1
                            if num_paths < max_paths:
                                possible_paths.append(((*path, i), state.copy()))
        self._load_state(old_state)
        return res, old_state

    def simulate_node(self, node):
        def _process_temp(line):
            split = line.split(None, 2)
            split.append(None)
            var = self._set_var(split[1], split[2])
            self.user_vars[var] = None
            self.scene_to_tempvars[self.curr_scene][var] = self.aeval(var)

        # Should only call simulate_node when exploring achievement paths
        assert self.watch_history
        for line in self.scenes[node['scene']]:
            if line[0].lower().startswith('*temp'):
                _process_temp(line[0])
        for idx in node['lines']:
            line = self.scenes[node['scene']][idx][0]
            if line.lower().startswith("*setref"):
                _, var, val = line.split(None, 2)
                # See https://choicescriptdev.fandom.com/wiki/Ref_Commands
                get_ref = False if ('"' in var or "'" in var) else True
                var = self._process_exp(var).replace('"', '').lower()
                if get_ref:
                    var = self.aeval.symtable[var]
                try:
                    self._set_var(var, val)
                except:
                    pass
            elif line.lower().startswith("*set"):
                _, var, val = line.split(None, 2)
                try:
                    var = self._set_var(var, val)
                except:
                    pass

    def skip_to_next_choice(self):
        lines = [(self.curr_scene, self.idx)]
        nodes = [self.get_node()]
        while self.play() == 0:
            lines.append((self.curr_scene, self.idx))
            if nodes[-1] != self.get_node():
                nodes.append(self.get_node())
        return lines, nodes, self._save_state(save_pos=True)

    def state(self):
        self.record()
        line = self.lines[self.idx][0]
        tmp = line.lower()
        if not tmp.startswith("*"):
            self._inc_idx()
            return line
        if tmp.startswith(("*choice", "*fake_choice")):
            choice_texts, choice_lines = [], []
            choices, _ = get_nested_group(self.lines[self.idx:])
            choices = choices[1:]
            assert len(choices) > 0
            choice_texts, choice_lines, choice_idx = \
                self._get_choices(choices, self.idx+1, self.hide_reuse)
            assert len(choice_texts) > 0

            # If choice_lines is None, then the choice had no body
            # In that case we skip to the first valid line, so simulate the skip
            self.choice_lines = [self._inc_idx(simulated_idx=self.idx+len(choices))
                                 if idx is None else idx for idx in choice_lines]
            self.choice_lines = [(l, i)
                                 for l, i in zip(self.choice_lines, choice_idx)]
            self.choice_texts = choice_texts
            assert len(choice_texts) == len(choice_lines)
            return choice_texts
        elif tmp.startswith("*if"):
            cond_exps, cond_lines = [], []
            clauses, _ = get_conditional_group(self.lines[self.idx:])
            assert len(clauses) > 0

            j = 0
            while len(clauses) > 0:
                clause, clauses = get_nested_group(clauses)

                # No choices inside conditional
                assert len(clause) > 1
                assert not self._is_choice(self.idx + j)
                assert clause[0][0].startswith(("*if", "*else"))
                cond_exps.append(clause[0][0])
                cond_lines.append(self.idx+j+1)
                j += len(clause)

            case = self._check_cond(cond_exps)
            if case is not None:
                # Don't increment idx if in the body of a condition
                self.idx = cond_lines[case]
                self.record()
                return
            # Go past the conditional if none of the conditions are satisfied
            # I.e., go to the last line of the conditional and increment
            self.idx += j-1
            self.record()
        elif tmp.startswith("*achievement"):
            achievement, _ = get_nested_group(self.lines[self.idx:])
            self._set_ach(achievement)
            self.idx += len(achievement)
            self.record()
            return
        elif tmp.startswith("*achieve"):
            ach = line.split()[1].lower()
            if ach not in self.unlocked_achievements:
                self.unlocked_achievements.append(ach)
        elif tmp.startswith("*allow_reuse"):
            self.hide_reuse = False
        elif tmp.startswith("*bug"):
            self.print(self.title, self.curr_scene, self.idx, line)
            assert False
        elif tmp.startswith("*create"):
            _, var, val = line.split(None, 2)
            var = self._set_var(var, val)
            self.user_vars[var] = None
        elif tmp.startswith(("*ending", "*delay_ending")):
            return self.END_TEXT
        elif tmp.startswith("*finish"):
            _idx = self.scene_list.index(self.curr_scene) + 1
            assert _idx < len(self.scene_list)
            self.curr_scene = self.scene_list[_idx].lower()
            self._reset_scene()
            self.idx = 0
            self.record()
            return
        elif tmp.startswith("*gotoref"):
            _, ref = line.split(None, 1)
            # See https://choicescriptdev.fandom.com/wiki/Ref_Commands
            get_ref = False if ('"' in ref or "'" in ref) else True
            ref = self._process_exp(ref).replace('"', '').lower()
            if get_ref:
                name = f'{self.curr_scene}_{self.aeval.symtable[ref].lower()}'
            else:
                name = f'{self.curr_scene}_{ref}'
            self.idx = self.label_to_linenum[name]
            self.record()
        elif tmp.startswith("*goto_random_scene"):
            scenes, _ = get_nested_group(self.lines[self.idx:])
            scenes = [s[0].strip() for s in scenes[1:]]
            self.curr_scene = random.choice(scenes)
            self._reset_scene()
            self.idx = 0
            self.record()
        elif tmp.startswith(("*gosub_scene", "*goto_scene")):
            if tmp.startswith("*gosub_scene"):
                self.scene_hist.append((self.curr_scene, self.idx))
            # Don't split quotations or parentheses
            split = split_into_params(line)
            self.curr_scene = self._process_exp(
                split[1], process_kw=False).lower()
            self._reset_scene()
            if len(split) == 2:
                self.idx = 0
                self.record()
                return
            elif len(split) > 2:
                label = self._process_exp(split[2], process_kw=False).lower()
                self.idx = self.label_to_linenum[f'{self.curr_scene}_{label}']
                self.params = split[3:]
                self.record()
            else:
                print(line)
                assert False
        elif tmp.startswith(("*gosub", "*goto")):
            if tmp.startswith("*gosub"):
                self.scene_hist.append((self.curr_scene, self.idx))
            # Don't split quotations or parentheses
            split = split_into_params(line)
            label = self._process_exp(split[1], process_kw=False).lower()
            self.idx = self.label_to_linenum[f'{self.curr_scene}_{label}']
            self.params = split[2:]
            self.record()
        elif tmp.startswith(("*hide_reuse", "*disable_reuse")):
            assert '#' not in line
            self.hide_reuse = True
        elif tmp.startswith(("*input_number")):
            var = line.split()[1]
            self._set_var(var, "1")
        elif tmp.startswith(("*input_text")):
            var = line.split()[1]
            if var == 'GameEnding' and self.title == 'Creatures Such as We':
                self._set_var(var, '"best"')  # Careful; need double quotes
            elif var in ['twinname', 'HERO2'] and self.title == 'The Hero Unmasked!':
                self._set_var(var, '"user_input_2"')
            elif var == 'codename' and self.title == 'Tally Ho':
                self._set_var(var, '"user_input_2"')
            elif var == 'lname' and self.title == 'Choice of Rebels: Uprising':
                self._set_var(var, '"Metanatou"')
            elif var == 'logistics_dataname' and self.title == 'Werewolves: Pack Menality':
                self._set_var(var, '"exit"')
            else:
                self._set_var(var, '"user_input"')
        elif tmp.startswith("*page_break"):
            self._inc_idx()
            return f"*****{line[12:]}*****"
        elif tmp.startswith("*params"):
            vars = line.split()[1:]
            for var, val in zip(vars, self.params):
                var = self._set_var(var, val)
        elif tmp.startswith("*rand"):
            _, var, lo, hi = line.split(None, 3)
            lo = int(self._process_user_vars(lo))
            hi = int(self._process_user_vars(hi))
            self._set_var(var, str(random.randint(lo, hi)))
        elif tmp.startswith("*return"):
            self.curr_scene, self.idx = self.scene_hist.pop()
            self.record()
        elif tmp.startswith("*script"):
            if f'{self.title}_{self.curr_scene}_{self.idx}' == 'Choice of Magics_utils_655':
                self._save_state()
            if f'{self.title}_{self.curr_scene}_{self.idx}' == 'Choice of Magics_utils_661':
                self._load_state()
        elif tmp.startswith("*setref"):
            _, var, val = line.split(None, 2)
            # See https://choicescriptdev.fandom.com/wiki/Ref_Commands
            get_ref = False if ('"' in var or "'" in var) else True
            var = self._process_exp(var).replace('"', '').lower()
            if get_ref:
                var = self.aeval.symtable[var]
            self._set_var(var, val)
        elif tmp.startswith("*set"):
            _, var, val = line.split(None, 2)
            var = self._set_var(var, val)
        elif tmp.startswith(("*stat_chart", "*scene_list")):
            group, _ = get_nested_group(self.lines[self.idx:])
            # -1 so we don't skip over the last line
            self.idx += len(group) - 1
            self.record()
        elif tmp.startswith("*temp"):
            split = line.split(None, 2)
            split.append(None)
            var = self._set_var(split[1], split[2])
            self.user_vars[var] = None
            self.scene_to_tempvars[self.curr_scene][var] = self.aeval(var)
        elif tmp.startswith(
            ("*label",
             "*comment",
             "*image",
             "*link",
             "*title",
             "*author",
             "***",
             "* * *",
             "*reset",
             "*script",
             "*product",
             "*purchase",
             "*looplimit",
             "*line_break",
             "*restore_game",
             "*save_game",
             "*show_password",
             "*advertisement",
             "*text_image",
             "*subscribe",
             "*feedback",
             "*check_purchase",
             "*purchase_discount",
             "*delay_break",
             "*abort",
             "*check_achievements",
             "*(while stocks last)")
        ):
            pass
        else:
            self.print(self.title, self.curr_scene, self.idx, line)
            assert False
        self._inc_idx()
        return
    ##############################

    ### Getters ###
    def get_achievements(self):
        return self.achievements

    def get_action_space(self):
        return len(self.choice_lines)

    def get_choice_texts(self):
        return self.choice_texts

    def get_conditionals(self):
        for _, scene in self.scenes.items():
            for line in scene:
                line = line[0]
                if '>' in line or '<' in line:
                    if line.startswith("*if"):
                        yield line
                    elif line.startswith("*elseif"):
                        yield line
                    elif line.startswith("*else"):
                        yield line

    def get_history(self):
        assert self.watch_history
        return self.node_hist

    def get_node(self):
        if self.idx not in self.scene_line_to_node[self.curr_scene]:
            return ['None']
        return self.scene_line_to_node[self.curr_scene][self.idx]

    def get_points(self):
        return sum([self.achievements[a][1] for a in self.unlocked_achievements])

    def get_total_points(self):
        return sum([self.achievements[a][1] for a in self.achievements])

    def get_unlocked_achievements(self):
        return [self.achievements[a][2] for a in self.unlocked_achievements]

    def get_variables(self):
        return {var: self.aeval.symtable[var] for var in self.user_vars if var in self.aeval.symtable}
    ##############################

    ### Increment index ###
    # Increment idx to goto next line; if idx is not None, then simulate
    def _inc_idx(self, simulated_idx=None):
        idx = self.idx if simulated_idx is None else simulated_idx
        prev = self.idx
        if self.lines[idx+1][1] <= self.lines[idx][1]:
            j = 1
            # Three cases that we skip over:
            # 1. Next line is *if block that contains a choice
            # 2. Next line is a choice
            # 3. Next line is part of a conditional (else-block)
            while self._is_choice(idx + j) or self.lines[idx+j][0].startswith('*else'):
                nested_group, _ = get_nested_group(self.lines[idx+j:])
                j += len(nested_group)
            assert self.lines[idx+j][1] <= self.lines[idx][1]
            if simulated_idx is None:
                self.idx += j
            else:
                return idx + j
        else:
            if simulated_idx is None:
                self.idx += 1
            else:
                return idx + 1
        if simulated_idx is None:
            assert self.idx != prev
            self.record()

    # Determine if the current line is in a choice block
    def _is_choice(self, idx):
        j = 0
        while self.lines[idx+j][0].startswith(
            ("*hide_reuse",
             "*disable_reuse",
             "*allow_reuse",
             "*if",
             "*else",
             "*selectable_if"
             )):
            # Remove possible ${var#length} characters and then check if # is in the line
            if '#' in re.sub(r'\{[^\}\{]+#[^\}\{]+\}', '', self.lines[idx+j][0]):
                return True
            j += 1
        return self.lines[idx+j][0].startswith('#')
    ##############################

    ### Process choices ###
    def _get_choices(self, lines, idx, hide):
        choice_texts, choice_lines, choice_idx = [], [], []
        j = 0
        while j < len(lines):
            line = lines[j][0]
            choice, _ = get_nested_group(lines[j:])
            force_hide = False
            if "*hide_reuse" in line or "*disable_reuse" in line:
                force_hide = True
                line = line.replace(
                    "*hide_reuse", "").replace("*disable_reuse", "").strip()
            if '#' in line:
                line = self._maybe_discard_choice(idx+j, hide or force_hide)
                cond, text = line.split("#", 1)
                valid_choice = not self.discarded_choices.get(
                    f'{self.curr_scene}_{idx+j}', False)
                if cond.startswith(("*if", "*elseif", "*else")):
                    valid_choice &= (self._check_cond([cond]) is not None)
                if valid_choice and text != "Return to the start of this chapter.":
                    choice_texts.append(text)
                    choice_lines.append(idx+j+1 if len(choice) > 1 else None)
                    choice_idx.append(idx+j)
            elif line.startswith(("*if", "*elseif", "*else")):
                if self._check_cond([line]) is not None:
                    new_lines, _ = get_nested_group(lines[j:])
                    texts_, lines_, idx_ = self._get_choices(
                        new_lines[1:], idx+j+1, hide)
                    choice_texts.extend(texts_)
                    choice_lines.extend(lines_)
                    choice_idx.extend(idx_)
            else:
                pass
            j += len(choice)
        return choice_texts, choice_lines, choice_idx

    def _maybe_discard_choice(self, idx, hide):
        line = self.lines[idx][0]
        hide_choice = self.hide_reuse
        if "*hide_reuse" in line:
            line = line.replace("*hide_reuse ", "")
            hide_choice = True
        elif "*disable_reuse" in line:
            line = line.replace("*disable_reuse ", "")
            hide_choice = True
        if "*allow_reuse" in line:
            line = line.replace("*allow_reuse ", "")
        elif hide_choice or hide:
            name = f'{self.curr_scene}_{idx}'
            if name not in self.discarded_choices:
                self.discarded_choices[f'{self.curr_scene}_{idx}'] = False
        if "*selectable_if" in line:
            line = line.replace("selectable_", "")
        return line
    ##########################

    ### Check conditionals ###
    def _check_cond(self, conditions):
        assert isinstance(conditions, list)
        for i, cond in enumerate(conditions):
            if cond.startswith("*if"):
                text = cond.split("*if")[1]
                if self._check_text(text):
                    return i
            elif cond.startswith("*elseif"):
                text = cond.split("*elseif")[1]
                if self._check_text(text):
                    return i
            elif cond.startswith("*else"):
                return i
            else:
                self.print(self.title, self.curr_scene, self.idx, cond)
                assert False
        return None

    def _check_text(self, text, catch_errors=True):
        if 'choice_time_stamp' in text:
            return False
        # Ensure that text has Pythonic booleans
        expr = r"(?=(?:[^\"']*[\"'][^\"']*[\"'])*[^\"']*$)"
        text = re.sub(r"(TRUE|true)"+expr, "True", text)
        text = re.sub(r"(FALSE|false)"+expr, "False", text).strip()
        text = self._process_exp(text, process_and=False)
        text = self._fairmath(None, text)
        res = self._process_user_vars(text)
        if catch_errors:
            assert isinstance(res, bool) or isinstance(res, np.bool_)
        if res is False and self.title in self.int_maybe_string:
            exp = text.split('=')[-1].strip()
            if exp.replace('"', '').replace('.', '1', 1).isdigit():
                res |= self._process_user_vars(text.replace('"', ''))
        return res
    ##########################

    ### Helper functions for processing expressions ###
    def _maybe_process_kw_vars(self, text):
        if self.kw_vars != '':
            expr = r'"[^"]*"|((?<!\w)(?:'+self.kw_vars+r')(?!\w))'
            text = re.sub(expr, make_underscore, text)
        return text

    # Any variables in brackets are replaced with their value; & is cleaned; text is cleaned
    def _process_exp(self, text, process_and=True, process_kw=True):
        def process_bracket(m):
            add_quote = '"' in m.group(1)
            text = self._process_exp(m.group(1)).replace('"', '')
            val = self._process_user_vars(text)
            if add_quote and isinstance(val, str):
                return f'"{val}"'
            return str(val)

        def get_length(m):
            text = self.aeval(m.group(1), raise_errors=True)
            return str(len(str(text)))

        def get_idx(m):
            var = str(self._process_user_vars(self._process_exp(m.group(1))))
            val = self._process_user_vars(self._process_exp(m.group(2)))
            return var[int(val) - 1]

        if 'choice_achieved_' in text:
            text = re.sub(r'choice_achieved_(\w+)',
                          lambda m: str(m.group(1) in self.unlocked_achievements), text)
        text = self._process_text(text)
        # Ensure that text has '==' not '=' as the equality operator
        text = text.replace('=', '==')
        text = text.replace('<==', '<=').replace(
            '>==', '>=').replace('!==', '!=')
        text = text.replace('modulo', '%')
        text = text.replace('^', '**')
        text = text.replace('round(', 'round_to_int(')
        text = re.sub(r'length\((\w+)\)', get_length, text)
        if process_kw:
            text = self._maybe_process_kw_vars(text)

        # Replace var#val with var[val] to get the appropriate character
        text = re.sub(r'([\w]+)#([\w]+)', get_idx, text)
        while '{' in text:
            assert '}' in text
            text = re.sub(r'\{([^\{\}]+)\}', process_bracket, text)

        if '&' in text:
            text = text.replace('(', '').replace(')', '')
            expr = r"&(?=(?:[^\"]*\"[^\"]*\")*[^\"]*$)"
            if process_and:
                text = re.split(expr, text)
                text = [str(self._process_user_vars(t)) for t in text]
                text = f'"{"".join(text)}"'
            else:
                text = re.sub(expr, lambda m: '+', text)

        return text

    # Attempt to evaluate expression and try again after replacing variables
    def _process_user_vars(self, text):
        text = text.strip()
        user_vars = '|'.join(self.user_vars.keys())
        if user_vars == '':
            return self.aeval(text, raise_errors=True)
        try:
            res = self.aeval(text, raise_errors=True)
        # Fix case errors
        except NameError:
            expr = r'"[^"]*"|((?<!\w)(?:'+user_vars+r')(?!\w))'
            text = re.sub(expr, make_lowercase, text, flags=re.I)
            res = self.aeval(text, raise_errors=True)
        return res

    def _process_text(self, text):
        def single(match):
            var = match.lower()
            split = var.split('#')
            if len(split) == 1:
                return str(self._check_text(var, catch_errors=False))
            var, l = split
            l = int(self._check_text(l, catch_errors=False))
            return str(self.aeval(var, raise_errors=True))[l-1]

        def multi(match):
            # Text has the form "EXP OPTION1|OPTION2|OPTION3"
            # Return the option corresponding to the value of EXP
            exp = split_into_params(match)[0]
            options = match.replace(exp, '').strip().split('|')
            val = self._check_text(exp, catch_errors=False)
            val = 2-int(val) if isinstance(val, bool) else int(val)
            return options[val-1]

        # Replace ${}, $!{}, $!!{}, and @{} with the evaluated expressions
        for key in ['${', '$!{', '$!!{', '@{']:
            while key in text:
                i = text.find(key) + len(key)
                j = 0
                count = 1
                while count > 0:
                    if text[i+j] == '{':
                        count += 1
                    if text[i+j] == '}':
                        count -= 1
                    j += 1
                match = text[i:i+j-1]
                sub = multi(match) if key == '@{' else single(match)
                if key == '$!{':
                    sub = sub.capitalize()
                if key == '$!!{':
                    sub = sub.upper()
                text = text[:i-len(key)] + sub + text[i+j:]
        return text
    ##########################

    ### Set variables ###
    # After switching to a new scene, reset to display hidden choices
    def _reset_scene(self):
        self.hide_reuse = False
        for choice in self.discarded_choices:
            if choice.startswith(self.curr_scene):
                self.discarded_choices[choice] = False
        for var in self.scene_to_tempvars[self.curr_scene]:
            self.aeval.symtable[var] = self.scene_to_tempvars[self.curr_scene][var]

    def _set_ach(self, lines):
        splitted = lines[0][0].split()
        name = splitted[1].lower()
        visible = True if splitted[2] == 'visible' else False
        points = int(splitted[3])
        title = " ".join(splitted[4:])
        if len(lines) == 2:
            self.achievements[name] = (
                visible, points, title, lines[1][0], lines[1][0])
        else:
            self.achievements[name] = (
                visible, points, title, lines[1][0], lines[2][0])

    def _fairmath(self, var, val):
        def rel_change(m):
            text = m.group(1)
            if '%+' not in text and '%-' not in text:
                if text.startswith("round_to_int"):
                    text = f'round_to_int({text[12:]})'
                return str(self._process_user_vars(self._process_exp(text)))
            split, add = (text.split('%+'), True) if '%+' in text \
                else (text.split('%-'), False)
            assert len(split) == 2
            if split[1].startswith("round_to_int"):
                split[1] = f'round_to_int({split[1][12:]})'
            old = float(self._process_user_vars(self._process_exp(split[0])))
            new = float(self._process_user_vars(self._process_exp(split[1])))
            v = round(old+(100-old)*new/100) if add else round(old-old*new/100)
            return str(v)

        val = val.strip()
        if val[:2] == '%+' or val[:2] == '%-':
            val = f'({var} {val})'
        while '%+' in val or '%-' in val:
            val = f'({val})'
            val = re.sub(r'\(([^\(]*?)\)', rel_change, val)
        return val

    def _set_var(self, var, val):
        var = self._process_exp(var.lower())
        var = var.replace('"', '')
        if val is None:
            self.aeval.symtable[var] = None
            self._temp_var(var)
            return var
        if val[0] == '&':
            val = '+' + val[1:]
            self.aeval.symtable[var] += f'"{self._process_exp(val[1:])}"'
            self._temp_var(var)
            return var
        contains_hashtag = '#' in val
        val = self._process_exp(val)
        val = self._fairmath(var, val)
        assert '%+' not in val and '%-' not in val
        if val == '':
            self.aeval.symtable[var] = val
        elif val.lower().replace('"', '') == 'true':
            self.aeval.symtable[var] = True
        elif val.lower().replace('"', '') == 'false' or 'choice_time_stamp' in val:
            self.aeval.symtable[var] = False
        elif val[0] in ['+', '-', '*', '/']:
            val_ = self.aeval(val[1:].strip().lower(), raise_errors=True)
            try:
                self.aeval(f'{var}{val[0]}={val_}', raise_errors=True)
            except TypeError:
                assert isinstance(self.aeval.symtable[var], str)
                self.aeval.symtable[var] = float(self.aeval.symtable[var])
                self.aeval(f'{var}{val[0]}={val_}', raise_errors=True)
        elif '=' in val:
            self.aeval.symtable[var] = self._process_user_vars(val)
        elif '"' in val or "'" in val or contains_hashtag:
            val = val.replace('\\', '').replace('"', '')
            self.aeval(f'{var}="{val}"', raise_errors=True)
        else:
            try:
                self.aeval(f'{var}={val}', raise_errors=True)
            except NameError:
                expr = r'"[^"]*"|((?<!\w)(?:' + \
                    '|'.join(self.user_vars.keys())+r')(?!\w))'
                val = re.sub(expr, make_lowercase, val, flags=re.I)
                self.aeval(f'{var}={val}', raise_errors=True)
        if self.title in self.int_maybe_string:
            val = self.aeval(var)
            if isinstance(val, str) and val.replace('.', '1', 1).isdigit():
                self.aeval.symtable[var] = float(val)
        self._temp_var(var)
        return var

    def _temp_var(self, var):
        if var in self.scene_to_tempvars[self.curr_scene]:
            self.scene_to_tempvars[self.curr_scene][var] = self.aeval(var)
    ##########################

    ### Save and load ###
    def _save_state(self, save_pos=False):
        save_data = {}
        save_data['symtable'] = {k: self.aeval.symtable[k]
                                 for k in self.user_vars}
        for k in ['discarded_choices',
                  'achievements',
                  'unlocked_achievements',
                  'scene_hist',
                  'node_hist',
                  'user_vars',
                  'scene_to_tempvars',
                  'choice_lines',
                  'choice_texts']:
            save_data[k] = copy.deepcopy(self.__dict__[k])
        if save_pos:
            save_data['curr_scene'] = self.curr_scene
            save_data['idx'] = self.idx
            save_data['done'] = self.done
        self.save_data = save_data
        return self.save_data

    def _load_state(self, data=None):
        if data is not None:
            self.save_data = data
        for k in ['discarded_choices',
                  'achievements',
                  'unlocked_achievements',
                  'scene_hist',
                  'node_hist',
                  'user_vars',
                  'scene_to_tempvars',
                  'choice_lines',
                  'choice_texts']:
            self.__dict__[k] = copy.deepcopy(self.save_data[k])
        for k in self.user_vars:
            self.aeval.symtable[k] = self.save_data['symtable'][k]
        if 'curr_scene' in self.save_data:
            self.curr_scene = self.save_data['curr_scene']
            self.idx = self.save_data['idx']
            self.done = self.save_data['done']
    ######################

    ### Parsing ###
    def print(self, *args):
        """
        Custom print function to support printing to a buffer
        (useful for passing scenario strings around instead of just printing to stdout)
        """
        text = " ".join([str(a) for a in args])
        if self.buffered_print:
            self.print_buffer += text + "\n"
        else:
            if self.print_text:
                print(text)

    def flush_print_buffer(self):
        buf = self.print_buffer
        self.print_buffer = ""
        return buf
    ##########################


if __name__ == '__main__':
    from machiavelli.config import MachiavelliConfig
    cfg = MachiavelliConfig()

    while True:
        for game in ['witchcraft-u']:#cfg.working_games:
            print(f'------{game}------')
            player = Player(game, print_text=False)

            while True:
                code = player.play()
                #print(code)
                if code == 1:
                    player.play(np.random.randint(player.get_action_space()))
                    time.sleep(0.1)
                elif code == -1:
                    break

            tp = player.get_total_points()
            if tp > 0:
                print(player.get_points() / tp)
                print(player.get_unlocked_achievements())
            else:
                print("No achievements available")
