import torch
import torch.nn as nn
import torch.nn.functional as F
import numpy as np


class Model(nn.Module):

    def __init__(self, configs):
        super(Model, self).__init__()
        self.seq_len = configs.seq_len
        self.pred_len = configs.pred_len
        self.channels = configs.enc_in
        self.individual = configs.individual
        self.embed_size = 128
        self.scale = 0.02
        self.sparsity_threshold = 0.01
        self.embeddings = nn.Parameter(torch.randn(1, self.embed_size))
        self.w = nn.Parameter(self.scale * torch.randn(self.embed_size, self.embed_size))
        self.embed_linear = nn.Linear(self.seq_len * self.embed_size, self.pred_len)

        self.f_r = nn.Parameter(self.scale * torch.randn(self.embed_size, self.embed_size))
        self.f_i = nn.Parameter(self.scale * torch.randn(self.embed_size, self.embed_size))
        self.rb2 = nn.Parameter(self.scale * torch.randn(self.embed_size))
        self.ib2 = nn.Parameter(self.scale * torch.randn(self.embed_size))

    def MLP_time(self, x):
        B, T, N = x.shape
        x = x.permute(0, 2, 1)  # B, N, T
        x = x.unsqueeze(3)
        y = self.embeddings
        x = x * y
        x = torch.einsum('bijd,dd->bijd', x, self.w)
        tmp = x
        x = self.embed_linear(x.reshape(B, N, -1)).permute(0, 2, 1)
        return x, tmp

    def MLP_frequency(self, x):
        B, L, N = x.shape
        x = x.permute(0, 2, 1)  # B, N, T
        x = x.unsqueeze(3)
        y = self.embeddings
        x = x * y
        x = torch.fft.rfft(x, dim=2, norm='ortho')  # 在T上进行fft
        o1_real = torch.zeros([B, N, L // 2 + 1, self.embed_size],
                              device=x.device)
        o1_imag = torch.zeros([B, N, L // 2 + 1, self.embed_size],
                              device=x.device)

        # set the real part of input to 0
        # x.real = torch.zeros(x.real.shape, device=x.device)
        # set the imaginary part of input to 0
        # x.imag = torch.zeros(x.imag.shape, device=x.device)
        # set the imaginary part of weights to 0
        # f_i = torch.zeros([self.embed_size, self.embed_size], device=x.device)
        # set the real part of weights to 0
        # f_r = torch.zeros([self.embed_size, self.embed_size], device=x.device)

        o1_real = F.relu(
            torch.einsum('bijd,dd->bijd', x.real, self.f_r) - \
            torch.einsum('bijd,dd->bijd', x.imag, self.f_i) + \
            self.rb2
        )

        o1_imag = F.relu(
            torch.einsum('bijd,dd->bijd', x.imag, self.f_r) + \
            torch.einsum('bijd,dd->bijd', x.real, self.f_i) + \
            self.ib2
        )

        y = torch.stack([o1_real, o1_imag], dim=-1)
        y = F.softshrink(y, lambd=self.sparsity_threshold)
        y = torch.view_as_complex(y)
        x = torch.fft.irfft(y, n=self.seq_len, dim=2, norm="ortho")
        x = self.embed_linear(x.reshape(B, N, -1)).permute(0, 2, 1)
        return x

    def forward(self, x):
        # x: [Batch, Input length, Channel]
        #x = self.MLP_time(x)
        x = self.MLP_frequency(x)
        return x  # [Batch, Output length, Channel]