import argparse
import os
import json
import hashlib
from diffusers import StableDiffusionPipeline
import torch
import random

def parse_args():
    parser = argparse.ArgumentParser(description="Generate images using Stable Diffusion with optional LoRA fine-tuning.")
    parser.add_argument("--prompt_file", type=str, required=True, help="Path to the prompt file with one prompt per line.")
    parser.add_argument("--model_path", type=str, default=None, help="Path to the LoRA fine-tuned model. If not provided, only the base model will be used.")
    parser.add_argument("--output_folder", type=str, required=True, help="Path to the output folder where images and JSON file will be saved.")
    return parser.parse_args()

def generate_image(pipe, prompt, seed, generator, output_folder, image_hash):
    # Generate image
    image = pipe(prompt, num_inference_steps=30, guidance_scale=7.5, generator=generator).images[0]
    
    # Save image
    image_path = os.path.join(output_folder, "images", f"{image_hash}.png")
    image.save(image_path)
    
    return image_path

def main():
    args = parse_args()
    
    # Create output folder and images subfolder
    os.makedirs(args.output_folder, exist_ok=True)
    os.makedirs(os.path.join(args.output_folder, "images"), exist_ok=True)
    
    # Load base model
    pipe = StableDiffusionPipeline.from_pretrained("stable-diffusion-v1-5/stable-diffusion-v1-5", torch_dtype=torch.float16)
    pipe.to("cuda")
    
    # Load LoRA fine-tuned model if provided
    if args.model_path:
        pipe.unet.load_attn_procs(args.model_path)
    
    # Read prompts from file
    with open(args.prompt_file, "r") as f:
        prompts = [line.strip() for line in f.readlines()]
    
    # Initialize random seed and generator
    seed = 42
    torch.manual_seed(seed)
    random.seed(seed)
    generator = torch.Generator("cuda").manual_seed(seed)
    
    # List to store JSON data
    json_data = []
    
    for prompt in prompts:
        # Generate unique hash for each prompt
        image_hash = hashlib.sha256(prompt.encode()).hexdigest()[:10]
        
        # Generate image
        image_path = generate_image(pipe, prompt, seed, generator, args.output_folder, image_hash)
        
        # Append JSON data
        json_data.append({
            "prompt": prompt,
            "image_path": os.path.join("images", f"{image_hash}.png")
        })
    
    # Save JSON file
    json_file_path = os.path.join(args.output_folder, "output.json")
    with open(json_file_path, "w") as json_file:
        json.dump(json_data, json_file, indent=4)
    
    print(f"Images and JSON file saved to {args.output_folder}")

if __name__ == "__main__":
    main()
