#!/bin/bash
# IMLP Ablation Study Script
# Tests attention=false (once) and attention=true with different window sizes

export CUDA_LAUNCH_BLOCKING=1
export PYTORCH_CUDA_ALLOC_CONF=max_split_size_mb:128

# Create ablation study directory
ABLATION_DIR="ablation_study_imlp_$(date +%Y%m%d_%H%M%S)"
mkdir -p "$ABLATION_DIR"

LOG_FILE="$ABLATION_DIR/ablation_log.log"
exec > >(tee -a "$LOG_FILE") 2>&1
echo "Starting IMLP ablation study at $(date)"
echo "Results will be saved in: $ABLATION_DIR"

# Tasks to test
TASKS=(146206 146607 146820 14951 14952 14964 14969 14970 167141 167211 168908 168909 168910 168912 190410 2074 28 32 3481 3510 3686 3711 3735 3889 3896 3897 3904 3950 3954 43 45 9952 9960 9985 9986 9987)

# Ablation parameters
WINDOW_SIZES=(1 5 10 15 20 25)
SEEDS=(7 42 101)
DEVICE="cuda"

# Base config file
BASE_CONFIG="configs/experiments/imlp.yaml"

# Function to create temporary config file
create_temp_config() {
    local window_size=$1
    local use_attention=$2
    local temp_config="$ABLATION_DIR/temp_imlp_ws${window_size}_att${use_attention}.yaml"
    
    cat > "$temp_config" << EOF
model: "IMLP"
mode: "segmental"
hyperparameters:
  learning_rate: 0.001
  epochs: 100
  batch_size: 128
  patience: 30
  window_size: ${window_size}
  use_attention: ${use_attention}
EOF
    
    echo "$temp_config"
}

# Create summary file
SUMMARY_FILE="$ABLATION_DIR/ablation_summary.csv"
echo "task,window_size,use_attention,seed,status" > "$SUMMARY_FILE"

# Total number of experiments
total_experiments=$((${#TASKS[@]} * ${#SEEDS[@]} + ${#TASKS[@]} * ${#WINDOW_SIZES[@]} * ${#SEEDS[@]}))
current_experiment=0

echo "Total experiments: $total_experiments"

# First, run all experiments with attention=false (window_size doesn't matter)
echo "================================================"
echo "Phase 1: Running experiments with attention=false"
echo "================================================"

for task in "${TASKS[@]}"; do
    for seed in "${SEEDS[@]}"; do
        current_experiment=$((current_experiment + 1))
        
        echo "------------------------------------------------"
        echo "Experiment $current_experiment/$total_experiments"
        echo "Task: $task, Attention: false, Seed: $seed"
        echo "------------------------------------------------"
        
        # Create temporary config (window_size=10 is arbitrary)
        temp_config=$(create_temp_config 10 false)
        
        # Set output directory
        export ABLATION_OUTPUT_DIR="$ABLATION_DIR/results/no_attention"
        
        # Run the experiment
        if python src/cli.py --experiment "$temp_config" --task $task --device $DEVICE --seed $seed --no_tuning; then
            echo "$task,N/A,false,$seed,success" >> "$SUMMARY_FILE"
            echo "✓ Completed successfully"
        else
            echo "$task,N/A,false,$seed,failed" >> "$SUMMARY_FILE"
            echo "✗ Failed"
        fi
        
        # Small delay to avoid overwhelming the system
        sleep 2
    done
done

# Then, run experiments with attention=true and varying window sizes
echo "================================================"
echo "Phase 2: Running experiments with attention=true"
echo "================================================"

for task in "${TASKS[@]}"; do
    for window_size in "${WINDOW_SIZES[@]}"; do
        for seed in "${SEEDS[@]}"; do
            current_experiment=$((current_experiment + 1))
            
            echo "------------------------------------------------"
            echo "Experiment $current_experiment/$total_experiments"
            echo "Task: $task, Window Size: $window_size, Attention: true, Seed: $seed"
            echo "------------------------------------------------"
            
            # Create temporary config
            temp_config=$(create_temp_config $window_size true)
            
            # Set output directory
            export ABLATION_OUTPUT_DIR="$ABLATION_DIR/results/ws${window_size}_atttrue"
            
            # Run the experiment
            if python src/cli.py --experiment "$temp_config" --task $task --device $DEVICE --seed $seed --no_tuning; then
                echo "$task,$window_size,true,$seed,success" >> "$SUMMARY_FILE"
                echo "✓ Completed successfully"
            else
                echo "$task,$window_size,true,$seed,failed" >> "$SUMMARY_FILE"
                echo "✗ Failed"
            fi
            
            # Small delay to avoid overwhelming the system
            sleep 2
        done
    done
done

echo "================================================"
echo "Ablation study completed at $(date)"
echo "Results saved in: $ABLATION_DIR"
echo "Summary available at: $SUMMARY_FILE"

# Generate quick statistics
echo ""
echo "Quick Statistics:"
echo "-----------------"
echo "Total experiments: $total_experiments"
echo "Successful: $(grep -c ",success" "$SUMMARY_FILE")"
echo "Failed: $(grep -c ",failed" "$SUMMARY_FILE")"

# Clean up temporary configs
rm -f "$ABLATION_DIR"/temp_imlp_*.yaml