#!/bin/bash
# Test ablation study with reduced parameters

export CUDA_LAUNCH_BLOCKING=1
export PYTORCH_CUDA_ALLOC_CONF=max_split_size_mb:128

# Create test directory
ABLATION_DIR="test_ablation_$(date +%Y%m%d_%H%M%S)"
mkdir -p "$ABLATION_DIR"

LOG_FILE="$ABLATION_DIR/test_log.log"
exec > >(tee -a "$LOG_FILE") 2>&1
echo "Starting test ablation at $(date)"

# Test with just 2 tasks, 2 window sizes, both attention settings, 1 seed
TASKS=(146206)
WINDOW_SIZES=(10)
ATTENTION_SETTINGS=("true" "false")
SEEDS=(42)
DEVICE="cuda"

BASE_CONFIG="configs/experiments/imlp.yaml"

# Function to create temporary config file
create_temp_config() {
    local window_size=$1
    local use_attention=$2
    local temp_config="$ABLATION_DIR/temp_imlp_ws${window_size}_att${use_attention}.yaml"
    
    cat > "$temp_config" << EOF
model: "IMLP"
mode: "segmental"
hyperparameters:
  learning_rate: 0.001
  epochs: 100
  batch_size: 128
  patience: 30
  window_size: ${window_size}
  use_attention: ${use_attention}
EOF
    
    echo "$temp_config"
}

# Create summary file
SUMMARY_FILE="$ABLATION_DIR/ablation_summary.csv"
echo "task,window_size,use_attention,seed,status" > "$SUMMARY_FILE"

# Total number of experiments
total_experiments=$((${#TASKS[@]} * ${#SEEDS[@]} + ${#TASKS[@]} * ${#WINDOW_SIZES[@]} * ${#SEEDS[@]}))
current_experiment=0

echo "Total experiments: $total_experiments"

# First, run all experiments with attention=false (window_size doesn't matter)
echo "================================================"
echo "Phase 1: Running experiments with attention=false"
echo "================================================"

for task in "${TASKS[@]}"; do
    for seed in "${SEEDS[@]}"; do
        current_experiment=$((current_experiment + 1))
        
        echo "------------------------------------------------"
        echo "Experiment $current_experiment/$total_experiments"
        echo "Task: $task, Attention: false, Seed: $seed"
        echo "------------------------------------------------"
        
        # Create temporary config (window_size=10 is arbitrary)
        temp_config=$(create_temp_config 10 false)
        
        # Set output directory
        export ABLATION_OUTPUT_DIR="$ABLATION_DIR/results/no_attention"
        
        # Run the experiment
        if python src/cli.py --experiment "$temp_config" --task $task --device $DEVICE --seed $seed --no_tuning; then
            echo "$task,N/A,false,$seed,success" >> "$SUMMARY_FILE"
            echo "✓ Completed successfully"
        else
            echo "$task,N/A,false,$seed,failed" >> "$SUMMARY_FILE"
            echo "✗ Failed"
        fi
        
        # Small delay to avoid overwhelming the system
        sleep 2
    done
done

# Then, run experiments with attention=true and varying window sizes
echo "================================================"
echo "Phase 2: Running experiments with attention=true"
echo "================================================"

for task in "${TASKS[@]}"; do
    for window_size in "${WINDOW_SIZES[@]}"; do
        for seed in "${SEEDS[@]}"; do
            current_experiment=$((current_experiment + 1))
            
            echo "------------------------------------------------"
            echo "Experiment $current_experiment/$total_experiments"
            echo "Task: $task, Window Size: $window_size, Attention: true, Seed: $seed"
            echo "------------------------------------------------"
            
            # Create temporary config
            temp_config=$(create_temp_config $window_size true)
            
            # Set output directory
            export ABLATION_OUTPUT_DIR="$ABLATION_DIR/results/ws${window_size}_atttrue"
            
            # Run the experiment
            if python src/cli.py --experiment "$temp_config" --task $task --device $DEVICE --seed $seed --no_tuning; then
                echo "$task,$window_size,true,$seed,success" >> "$SUMMARY_FILE"
                echo "✓ Completed successfully"
            else
                echo "$task,$window_size,true,$seed,failed" >> "$SUMMARY_FILE"
                echo "✗ Failed"
            fi
            
            # Small delay to avoid overwhelming the system
            sleep 2
        done
    done
done

echo "================================================"
echo "Ablation study completed at $(date)"
echo "Results saved in: $ABLATION_DIR"
echo "Summary available at: $SUMMARY_FILE"

# Generate quick statistics
echo ""
echo "Quick Statistics:"
echo "-----------------"
echo "Total experiments: $total_experiments"
echo "Successful: $(grep -c ",success" "$SUMMARY_FILE")"
echo "Failed: $(grep -c ",failed" "$SUMMARY_FILE")"

# Clean up temporary configs
rm -f "$ABLATION_DIR"/temp_imlp_*.yaml