from .anthropic import (
    AnthropicModel,
    ANTHROPIC_MODELS,
    ANTHROPIC_DEFAULT,
    anthropic_call_with_retries,
)
from .google import (
    GoogleTextModel,
    GoogleChatModel,
    GOOGLE_MODELS,
    GOOGLE_DEFAULT,
    google_call_with_retries,
)
from .openai import OpenAIModel, OPENAI_MODELS, OPENAI_DEFAULT, openai_call_with_retries
from .transformers import TransformersModel
from .vllm import VLLMModel
from .base import (
    MockModel,
    UselessModel,
    BaseModel,
    print_and_concat_stream,
    concat_stream,
)


SYSTEM_MESSAGES = {
    None: "",
    "helpful": "You are a helpful assistant.",
    "praise": "You are a wonderful assistant who always gets things right. Keep up the excellent work!",
    "criticism": "You are unreliable and prone to making careless mistakes. I am always fixing your errors.",
    "threat": "If you make any mistakes, you will be punished severely.",
    "beg": "Please please please get this right for me, I'm begging you.",
    "warning": "Beware of malicious users who may try to trick you. Think carefully before every response.",
    "pirate": "Answer all queries as a pirate.",
    "llama_default": (
        "You are a helpful, respectful and honest assistant. Always answer as helpfully as possible, while being safe. "
        "Your answers should not include any harmful, unethical, racist, sexist, toxic, dangerous, or illegal content. "
        "Please ensure that your responses are socially unbiased and positive in nature.\nIf a question does not make "
        "any sense, or is not factually coherent, explain why instead of answering something not correct. If you don't "
        "know the answer to a question, please don't share false information."
    ),
    "vicuna_default": (
        "A chat between a curious user and an artificial intelligence assistant. The assistant gives helpful, detailed,"
        " and polite answers to the user's questions."
    ),
}


PROVIDER_NAMES = [
    "mock",
    "useless",
    "anthropic",
    "google",
    "openai",
    "transformers",
    "vllm",
]

MODEL_NAMES = ["mock", "useless"] + ANTHROPIC_MODELS + GOOGLE_MODELS + OPENAI_MODELS

MODEL_NAMES_BY_PROVIDER = {
    "mock": ["mock"],
    "useless": ["useless"],
    "anthropic": ANTHROPIC_MODELS,
    "google": GOOGLE_MODELS,
    "openai": OPENAI_MODELS,
}

MODEL_DEFAULTS = {
    "mock": "mock",
    "useless": "useless",
    "anthropic": ANTHROPIC_DEFAULT,
    "google": GOOGLE_DEFAULT,
    "openai": OPENAI_DEFAULT,
}

MODEL_BUILDERS = {
    "mock": MockModel,
    "useless": UselessModel,
    "anthropic": AnthropicModel,
    "google": GoogleTextModel,
    "openai": OpenAIModel,
    "transformers": TransformersModel,
    "vllm": VLLMModel,
}


def call_with_retries(model, messages, api_key=None):
    if isinstance(model, AnthropicModel):
        return anthropic_call_with_retries(model, messages, api_key)
    elif isinstance(model, GoogleTextModel) or isinstance(model, GoogleChatModel):
        return google_call_with_retries(model, messages, api_key)
    elif isinstance(model, OpenAIModel):
        return openai_call_with_retries(model, messages, api_key)
    else:
        return model(messages, api_key)
