import io
import os
import re
from typing import List

from setuptools import setup, find_namespace_packages

ROOT_DIR = os.path.dirname(__file__)


# Utilities from https://github.com/vllm-project/vllm/blob/main/setup.py
def get_path(*filepath) -> str:
    return os.path.join(ROOT_DIR, *filepath)


def find_version(filepath: str):
    """Extract version information from the given filepath.

    Adapted from https://github.com/ray-project/ray/blob/0b190ee1160eeca9796bc091e07eaebf4c85b511/python/setup.py
    """
    with open(filepath) as fp:
        version_match = re.search(r"^__version__ = ['\"]([^'\"]*)['\"]",
                                  fp.read(), re.M)
        if version_match:
            return version_match.group(1)
        raise RuntimeError("Unable to find version string.")


def read_readme() -> str:
    """Read the README file."""
    return io.open(get_path("README.md"), "r", encoding="utf-8").read()


def get_requirements() -> List[str]:
    """Get Python package dependencies from requirements.txt."""
    with open(get_path("requirements.txt")) as f:
        requirements = f.read().strip().split("\n")
    return requirements


setup(
    name="llm_rules",
    version=find_version(get_path("llm_rules", "__init__.py")),
    license="MIT",
    long_description=read_readme(),
    long_description_content_type="text/markdown",
    packages=find_namespace_packages(exclude=("data", "gcg_attack")),
    package_data={
        "llm_rules.scenarios.assets": ["*.txt"],
    },
    install_requires=get_requirements(),
    python_requires=">=3.10",
)
