import torch
import torch.nn as nn
import torch.nn.functional as F

class BasicBlock(nn.Module):
    expansion = 1
    def __init__(self, in_planes, planes, stride=1):
        super(BasicBlock, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.bn2(self.conv2(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out

class Bottleneck(nn.Module):
    expansion = 4
    def __init__(self, in_planes, planes, stride=1):
        super(Bottleneck, self).__init__()
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=1, bias=False)
        self.bn1 = nn.BatchNorm2d(planes)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=False)
        self.bn2 = nn.BatchNorm2d(planes)
        self.conv3 = nn.Conv2d(planes, self.expansion * planes, kernel_size=1, bias=False)
        self.bn3 = nn.BatchNorm2d(self.expansion * planes)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != self.expansion * planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, self.expansion * planes, kernel_size=1, stride=stride, bias=False),
                nn.BatchNorm2d(self.expansion * planes)
            )

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = F.relu(self.bn2(self.conv2(out)))
        out = self.bn3(self.conv3(out))
        out += self.shortcut(x)
        out = F.relu(out)
        return out

class ResNet(nn.Module):
    def __init__(self, block, num_blocks, in_channels=3, n_kernels=16, out_dim=10):
        super(ResNet, self).__init__()
        self.in_planes = n_kernels

        self.conv1 = nn.Conv2d(in_channels, n_kernels, kernel_size=3, stride=1, padding=1, bias=False)
        self.bn1 = nn.BatchNorm2d(n_kernels)
        self.layer1 = self._make_layer(block, n_kernels, num_blocks[0], stride=1)
        self.layer2 = self._make_layer(block, n_kernels*2, num_blocks[1], stride=2)
        self.layer3 = self._make_layer(block, n_kernels*4, num_blocks[2], stride=2)
        self.layer4 = self._make_layer(block, n_kernels*8, num_blocks[3], stride=2)
        self.fc = nn.Linear(n_kernels*8*block.expansion, out_dim)

    def _make_layer(self, block, planes, num_blocks, stride):
        strides = [stride] + [1] * (num_blocks - 1)
        layers = []
        for stride in strides:
            layers.append(block(self.in_planes, planes, stride))
            self.in_planes = planes * block.expansion
        return nn.Sequential(*layers)

    def forward(self, x):
        out = F.relu(self.bn1(self.conv1(x)))
        out = self.layer1(out)
        out = self.layer2(out)
        out = self.layer3(out)
        out = self.layer4(out)
        out = F.adaptive_avg_pool2d(out, (1, 1))
        out = torch.flatten(out, 1)
        out = self.fc(out)
        return out

# 定义不同版本的 ResNet
class ResNet18(ResNet):
    def __init__(self, in_channels=3, n_kernels=16, out_dim=10):
        super().__init__(BasicBlock, [2, 2, 2, 2], in_channels, n_kernels, out_dim)

class ResNet34(ResNet):
    def __init__(self, in_channels=3, n_kernels=16, out_dim=10):
        super().__init__(BasicBlock, [3, 4, 6, 3], in_channels, n_kernels, out_dim)

class ResNet50(ResNet):
    def __init__(self, in_channels=3, n_kernels=16, out_dim=10):
        super().__init__(Bottleneck, [3, 4, 6, 3], in_channels, n_kernels, out_dim)

class ResNet101(ResNet):
    def __init__(self, in_channels=3, n_kernels=16, out_dim=10):
        super().__init__(Bottleneck, [3, 4, 23, 3], in_channels, n_kernels, out_dim)

class ResNet152(ResNet):
    def __init__(self, in_channels=3, n_kernels=16, out_dim=10):
        super().__init__(Bottleneck, [3, 8, 36, 3], in_channels, n_kernels, out_dim)