import numpy as np
import pandas as pd
from sklearn.linear_model import LinearRegression
from sklearn.neural_network import MLPRegressor
from sklearn.linear_model import Ridge
from sklearn.linear_model import Lasso
from sklearn.linear_model import BayesianRidge
from sklearn.svm import SVR
from sklearn.tree import DecisionTreeRegressor
from sklearn.ensemble import (
    RandomForestRegressor,
    GradientBoostingRegressor,
    AdaBoostRegressor,
)
from sklearn.metrics import mean_absolute_error
from utils import Console
from typing import Union

# Dictionary of regressors
regressors = {
    "AdaBoost Regressor": AdaBoostRegressor(
        estimator=DecisionTreeRegressor(max_depth=3), n_estimators=100, random_state=48
    ),
    "Bayesian Ridge Regressor": BayesianRidge(),
    "Decision Tree Regressor": DecisionTreeRegressor(max_depth=5, max_leaf_nodes=4),
    "Gradient Boosting Regressor": GradientBoostingRegressor(
        n_estimators=100, random_state=48
    ),
    "Lasso Regressor": Lasso(alpha=0.1),
    "Linear Regressor": LinearRegression(),
    "MLP Regressor": MLPRegressor(
        hidden_layer_sizes=(150, 100, 50),
        max_iter=500,
        activation="relu",
        solver="adam",
    ),
    "Random Forest Regressor": RandomForestRegressor(n_estimators=100, random_state=48),
    "Ridge Regressor": Ridge(alpha=1.0),
    "Support Vector Regressor": SVR(kernel="linear", C=1, epsilon=0.1),
}


class Regressor:
    # Initializes the regressor
    def __init__(self, *, name: str, verbose=True):
        self.console = Console(verbose=verbose)
        self.mse = float("inf")
        self.name = name
        self.regressor = regressors.get(self.name)

    # print regressor attributes
    def __str__(self):
        str = self.console.string("Regressor information", endl=True)
        str += self.console.string("Name", self.name, endl=True)
        str += self.console.string("Mean absolute error", self.error, endl=True)
        return str

    # Train regressor from instances and labels
    def train(self, *, instances: pd.DataFrame, labels: pd.DataFrame) -> float:
        self.console.log("Train regressor")
        X = instances.to_numpy()
        Y = labels.values.ravel()
        self.regressor.fit(X, Y)
        P = self.regressor.predict(X)
        self.error = mean_absolute_error(Y, P)
        return self.error

    # Predict the value of single instance or values of multiple instances
    def predict(self, X) -> np.ndarray:
        self.console.log("Predict")
        # Predict from 1 or 2 dimensions
        if X.ndim == 1:
            return self.regressor.predict([X])
        else:
            return self.regressor.predict(X)
