import numpy as np
import pandas as pd
from sklearn.metrics import mean_squared_error, root_mean_squared_error


class Console:
    def __init__(self, *, decimals=3, length=40, verbose=True):
        self.decimals = decimals
        self.length = length
        self.verbose = verbose

    def log(self, first, second=None, endl=False):
        if not self.verbose:
            return
        if isinstance(first, str):
            print(self.string(first, second, endl))
        else:
            if isinstance(first, dict):
                print(self.dictionary(first))

    def dictionary(self, obj):
        str = ""
        for first, second in obj.items():
            str += self.string(f"{first}", f"{second}", endl=True)
        return str

    def string(self, first, second=None, endl=False):
        if second is None:
            str = first.ljust(self.length, ".")
        else:
            str = first.ljust(self.length - 1, ".") + ": "
            if isinstance(second, float):
                str += f"{second:.{self.decimals}f}"
            else:
                str += f"{second}"
        if endl:
            str += "\n"
        return str


class Convert:
    # Convert sparse vector to dictionary
    @staticmethod
    def vec_to_dict(vector: np.ndarray, columns: pd.Index) -> dict:
        (indices,) = np.nonzero(vector)
        dictionary = {}
        for j in indices:
            dictionary[columns[j]] = vector[j]
        return dictionary

    # Convert list of tuples (feature, val) to vector of vals
    @staticmethod
    def list_to_vec(list: list, columns: pd.Index) -> np.ndarray:
        vec = np.zeros(len(columns), dtype="float64")
        for pair in list:
            i = columns.get_loc(pair[0])
            vec[i] = pair[1]
        return vec


class Metrics:
    # Mean squared error using matrix of samples, vector of labels and solution
    @staticmethod
    def mse(
        *, samples: np.ndarray, labels: np.ndarray, solution: np.ndarray
    ) -> np.ndarray:
        A = samples
        b = labels
        w = solution
        p = np.dot(A, w)
        return mean_squared_error(b, p)

    @staticmethod
    def rmse(
        *, samples: np.ndarray, labels: np.ndarray, solution: np.ndarray
    ) -> np.ndarray:
        A = samples
        b = labels
        w = solution
        p = np.dot(A, w)
        return root_mean_squared_error(b, p)
