"""
Prototype tool-augmented procedural activity assistant

"""

from argparse import ArgumentParser
import json
import logging
from pathlib import Path
from pprint import pprint, pformat
from copy import deepcopy
from collections import defaultdict

from utils import (
    get_date,
    format_instruction,
    # check_prompt,
)

from utils_api import (
    estimate_cost,
    call_api_single,
    ANTHROPIC_USAGE_KEYS,
)

from tools import (
    call_function,
    load_tools,
)

from prompts import (
    format_initial_user_message,
    load_prompt_template,
)


def cut_in(
    args,
    example,
    template_components,
    count_turn,
    count_non_tool_output,
    flag_answer_predicted,
):
    block, block_text = None, ""

    logging.info("Check if cut-in is required")

    # max_turn = args.max_turn
    threshold_max_text_turn = args.threshold_max_text_turn
    threshold_continue = args.threshold_continue

    if flag_answer_predicted:
        if count_turn > threshold_continue:
            logging.info(
                f"{threshold_continue} turns have passed. No prolonging required."
            )
        else:
            if count_non_tool_output > 0:
                # encourage model to explore more if model's output is only answer
                logging.info(
                    "Answer is provided, but let's encourage the model to explore more."
                )
                block = {
                    "type": "text",
                    "text": template_components["user"]["continue"],
                }
                block_text = template_components["user"]["continue"]
            else:
                logging.info(
                    "tool_call detected. let model answer based on the tool result"
                )
                block_text = "Make sure to review the tool result and think before making any additional tool calls or answering the question."
                block = {"type": "text", "text": block_text}
    else:
        # encourage model to output answer in the specified format
        if count_turn > threshold_continue:
            logging.info("Let's encourage the model to output an answer")
            text = template_components["user"]["answer"].replace(
                "{question}", example["question"]
            )
            block = {"type": "text", "text": text}
            block_text = text
        else:
            if (
                count_non_tool_output >= threshold_max_text_turn
                and count_turn <= threshold_continue
            ):
                logging.info("Let's encourage the model to continue its exploration")
                # if some consecutive non-function outputs and not enough tool calling
                # encourage model to explore more
                block = {
                    "type": "text",
                    "text": template_components["user"]["continue"],
                }
                block_text = template_components["user"]["continue"]
            else:
                if count_non_tool_output == 0:
                    logging.info("Encourage model to think based on tool result.")
                    block_text = "Make sure to review the tool result and think before making any additional tool calls or answering the question."
                    block = {"type": "text", "text": block_text}
                else:
                    logging.info("No need right now")

    return block, block_text


def count_image_in_messages(messages):
    count = 0
    for message in messages:
        if isinstance(message["content"], list):
            for block in message["content"]:
                if block["type"] == "tool_result":
                    for content in block["content"]:
                        if content["type"] == "image":
                            count += 1
    return count


def truncate_images_if_needed(args, messages):
    """
    truncate initial images if #images reached max_#images

    """

    # 1. check #images
    total_images = count_image_in_messages(messages)

    # 2. truncation if needed
    if total_images <= args.max_total_images:
        logging.info(f"#images: {total_images} (not changed)")
        return messages

    output = []
    images_to_remove = total_images - args.max_total_images
    images_removed = 0
    logging.info(f"[debug] remove {images_to_remove} images")

    for message in messages:
        if isinstance(message["content"], list):
            new_message = {"role": message["role"], "content": []}

            for block in message["content"]:
                if block["type"] == "tool_result":
                    new_block = {
                        "type": block["type"],
                        "tool_use_id": block["tool_use_id"],
                        "content": [],
                    }

                    for content in block["content"]:
                        if (
                            content["type"] == "image"
                            and images_removed < images_to_remove
                        ):
                            new_block["content"].append(
                                {
                                    "type": "text",
                                    "text": "<image>Truncated because #images exceeds the max</image>",
                                }
                            )
                            images_removed += 1
                            logging.info(
                                f"debug: {images_removed}/{images_to_remove} images removed"
                            )
                        else:
                            new_block["content"].append(content)
                else:
                    new_block = block

                new_message["content"].append(new_block)
        else:
            new_message = message

        output.append(new_message)

    final_count = count_image_in_messages(output)
    logging.info(f"#image: {total_images} -> {final_count} (reduced)")

    return output


def run_single_example(args, example, template_components, tools, instructions):
    count_turn = 0  # count conversational turns
    count_non_tool_output = 0
    conversation_history = []
    total_usage = defaultdict(int)

    system_developer_message = template_components["system"]
    initial_user_message, initial_user_message_text = format_initial_user_message(
        args.model_id, template_components, example
    )
    # register turn 0
    conversation_history.append(
        [["system", system_developer_message], ["user", initial_user_message_text]]
    )

    messages = [initial_user_message]
    flag_answer_predicted = False
    answer = ""
    flag_end_conversation = False
    while count_turn < args.max_turn:
        logging.info(f"Next: turn {count_turn + 1}")

        tool_choice = "auto"

        try:
            response, usage = call_api_single(
                args, system_developer_message, messages, tools, tool_choice
            )
        except Exception:
            logging.exception("Error in run_single_example")
            conversation_history.append(["Error"])
            continue

        for key in ANTHROPIC_USAGE_KEYS:
            total_usage[key] += usage[key]

        if len(response.content) > 1:  # sanity check
            logging.info(
                "[Debug] "
                f"Multiple blocks in response: {[x.type for x in response.content]}"
            )

        # prev_message_types = []
        latest_assistant_response = ""
        current_turn_assistant, current_turn_user = [], []
        message_types_in_response = []
        # note: diff from openai. tool_use under content for anthropic
        assistant_content, user_content = [], []
        for block in response.content:
            message_types_in_response.append(block.type)
            match block.type:
                case "tool_use":
                    # append model's function call message
                    assistant_content.append(
                        {
                            "id": block.id,
                            "type": block.type,
                            "name": block.name,
                            "input": block.input,
                        }
                    )
                    current_turn_assistant.append(f"{block.name} {block.input}")

                    # function call
                    tool_result_content, tool_result_content_text, _ = call_function(
                        args,
                        example,
                        template_components,
                        instructions,
                        block,
                    )
                    user_content.append(
                        {
                            "type": "tool_result",
                            "tool_use_id": block.id,
                            "content": tool_result_content,
                        }
                    )
                    current_turn_user.append(tool_result_content_text)
                case "text":
                    assistant_content.append(
                        {
                            "type": block.type,
                            "text": block.text,
                        }
                    )
                    current_turn_assistant.append(block.text)

                    latest_assistant_response = block.text
                case "thinking":
                    assistant_content.append(
                        {
                            "type": block.type,
                            "thinking": block.thinking,
                            "signature": block.signature,
                        }
                    )
                    current_turn_assistant.append(block.thinking)
                case _:
                    logging.error(f"Undefined {block.type=}")

        # register as message(s)
        messages.append({"role": "assistant", "content": assistant_content})
        current_turn = [["assistant", current_turn_assistant]]
        logging.info("[assistant]")
        logging.info(current_turn_assistant)

        # check if answer is produced
        if "<answer>" in latest_assistant_response:
            flag_answer_predicted = True
            answer = latest_assistant_response.split("<answer>")[-1].replace(
                "</answer>", ""
            )

        # check output types
        if "tool_use" in message_types_in_response:
            count_non_tool_output = 0
        else:
            count_non_tool_output += 1

        # manual cut-in to prolong or end conversation
        block_cut_in, block_cut_in_text = cut_in(
            args,
            example,
            template_components,
            count_turn,
            count_non_tool_output,
            flag_answer_predicted,
        )
        if block_cut_in:
            user_content.append(block_cut_in)
            current_turn_user.append(block_cut_in_text)

        # create a user message possibly including function_call_output and
        if user_content:
            messages.append({"role": "user", "content": user_content})
            logging.info("[user]")
            logging.info(current_turn_user)
            current_turn.append(["user", current_turn_user])

        # register conv history in text
        conversation_history.append(current_turn)

        # update messages if #images > max_#images
        messages = truncate_images_if_needed(args, messages)

        # termination condition
        if count_turn >= args.max_turn:
            logging.info(f"Reached {args.max_turn=}. Terminate this process.")
            flag_end_conversation = True
        else:
            if flag_answer_predicted:
                if count_turn > args.threshold_continue:
                    if count_non_tool_output > 0:
                        logging.info(
                            "Answer predicted & Latest response contains no tool call. "
                            "So finish now."
                        )
                        flag_end_conversation = True
                    else:
                        logging.info(
                            f"Although turn {args.threshold_continue} reached, "
                            "as a tool is called, continue this process"
                        )
                else:
                    pass
            else:
                pass

        if flag_end_conversation:
            break

        count_turn += 1
    cost = estimate_cost(args.model_id, total_usage)

    return messages, conversation_history, answer, cost


def check_if_prediction_exists(example, examples_prev):
    corresponding_example = None
    if "examples" in examples_prev:
        for example_prev in examples_prev["examples"]:
            if (
                "question" in example_prev
                and example["question"] == example_prev["question"]
            ):
                if (
                    "prediction" in example_prev
                    and example_prev["prediction"]["answer"].strip()
                ):
                    corresponding_example = deepcopy(example_prev)

    return corresponding_example


def main(args):
    # load data
    with open(args.filepath_qa, "r") as f:
        examples = json.load(f)

    # load instruction
    toy2instruction = format_instruction(
        args.filepath_instruction,
        args.dirpath_instruction_image,
        args.dirpath_parts_image,
    )

    # load prompt template
    template_components = load_prompt_template(args.filepath_template)

    # load tools
    tools = load_tools(args)
    logging.info(pformat(tools, width=100))

    new_examples, total_cost = [], 0

    # output file
    filepath_output = args.dirpath_output / f"{args.model_id}_{args.reasoning}.json"
    if filepath_output.exists():
        with open(filepath_output, "r") as f:
            examples_prev = json.load(f)
        logging.info(
            f"Prev attempt exists. Restart from {len(examples_prev['examples'])}"
        )
        total_cost = float(examples_prev["metadata"]["cost"])
    else:
        logging.info("Initial attempt")
        examples_prev = []

    metadata = {
        "data-created": get_date(),
        "args": {k: str(v) if isinstance(v, Path) else v for k, v in vars(args).items()},
    }

    # start inference
    logging.info("Prediction starts ...")
    for idx, example in enumerate(examples):
        logging.info(f"[Example {idx}]")

        # check if prediction already exists
        corresponding_example_prev = check_if_prediction_exists(example, examples_prev)
        if corresponding_example_prev:
            logging.info("Previous prediction exists. Skip.")
            new_examples.append(corresponding_example_prev)
        else:
            try:
                messages, conversation_history, answer, cost = run_single_example(
                    args,
                    example,
                    template_components,
                    tools,
                    toy2instruction[example["toy_id"]],
                )
                new_example = deepcopy(example)
                new_example["prediction"] = {
                    "conversation_history": conversation_history,
                    "answer": answer,
                    "cost": cost,
                }
                new_examples.append(new_example)
                total_cost += cost
            except Exception:
                logging.exception("An error occurred")
                new_examples.append("Error")

        # save output every time
        metadata["cost"] = f"{total_cost:.3f}"
        final_output = {"metadata": metadata, "examples": new_examples}
        with open(filepath_output, "w") as f:
            json.dump(final_output, f, indent=4)
            f.write("\n")

    logging.info(f"{total_cost=}")


if __name__ == "__main__":
    parser = ArgumentParser(description="prototype")
    parser.add_argument("--filepath_qa", type=Path, help="filepath for qa input")
    parser.add_argument(
        "--filepath_instruction", type=Path, help="filepath for instruction"
    )
    parser.add_argument(
        "--dirpath_instruction_image", type=Path, help="dirpath for instruction (image)"
    )
    parser.add_argument(
        "--dirpath_parts_image", type=Path, help="dirpath for parts (image)"
    )
    parser.add_argument("--dirpath_frame", type=Path, help="dirpath for frames")
    parser.add_argument(
        "--filepath_template", type=Path, help="filepath for prompt template"
    )
    parser.add_argument(
        "--dirpath_intermediate_output",
        type=Path,
        help="dirpath to store intermediate outputs",
    )
    parser.add_argument("--filepath_tool", type=Path, help="filepath for tools")
    parser.add_argument("--dirpath_output", type=Path, help="filepath for output")
    parser.add_argument("--model_id", type=str, help="model id", default="")
    parser.add_argument("--temperature", type=float, help="temperature", default=0.1)
    parser.add_argument(
        "--max_tokens", type=int, help="max tokens to generate", default=2500
    )
    parser.add_argument("--max_frames", type=int, help="max frames to feed", default=30)
    parser.add_argument(
        "--max_total_images",
        type=int,
        help="max total frames to feed (100 is for anthropic api, but use 80 for 32MB limit)",
        default=70,
    )
    parser.add_argument("--max_turn", type=int, help="max turns", default=10)
    parser.add_argument(
        "--threshold_continue",
        type=int,
        help="the minimum turn to spend before generating answers for each example",
        default=5,
    )
    parser.add_argument(
        "--threshold_max_text_turn",
        type=int,
        help="the max #turn of non-function outputs",
        default=2,
    )
    parser.add_argument("--reasoning", action="store_true", help="Enable reasoning")
    parser.add_argument(
        "--reasoning_effort",
        type=str,
        help="reasoning level for openAI models",
        default="medium",
    )
    parser.add_argument(
        "--summary_type", type=str, help="summary type for openai api", default="auto"
    )
    parser.add_argument(
        "--budget_tokens",
        type=int,
        help="thinking budget tokens for anthropic api",
        default=2048,
    )
    parser.add_argument("--angle", type=str, help="angle", default="C10118_rgb")
    parser.add_argument("--resolution", type=str, help="resolution", default="360p")
    parser.add_argument("--color", type=str, help="color", default="rgb")
    parser.add_argument("--wait_time", type=int, help="API call wait time", default=10)
    parser.add_argument("--dirpath_log", type=Path, help="dirpath for log")

    args = parser.parse_args()

    if not args.dirpath_log.exists():
        args.dirpath_log.mkdir()

    if not args.dirpath_output.exists():
        args.dirpath_output.mkdir(parents=True)

    logging.basicConfig(
        format="%(asctime)s:%(levelname)s - %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        level=logging.INFO,
        handlers=[
            logging.StreamHandler(),
            logging.FileHandler(
                args.dirpath_log
                / f"prototype_{Path(args.model_id).name}_{get_date()}.log"
            ),
        ],
    )

    logging.info("Arguments:")
    pprint(vars(args))

    main(args)
