"""
Prototype tool-augmented procedural activity assistant

model: mimo
tools: text-based
"""

from argparse import ArgumentParser
import json
import logging
from pathlib import Path
from pprint import pformat
import sys
from copy import deepcopy

from utils import (
    get_date,
    format_instruction,
    # check_prompt,
)

from utils_api import (
    call_api_single,
)

from tools import (
    load_tools,
)
from tools_text import (
    call_function,
)

from prompts import (
    format_initial_user_message,
    load_prompt_template,
)


def cut_in(
    args,
    example,
    template_components,
    count_turn,
    count_non_tool_output,
    flag_answer_predicted,
):
    content = ""

    logging.info(f"Check if cut-in is required: {count_turn=}, {count_non_tool_output=}")

    # max_turn = args.max_turn
    threshold_max_text_turn = args.threshold_max_text_turn
    threshold_continue = args.threshold_continue

    # THINK: better to think more. create a diagram may help
    if flag_answer_predicted:
        if count_turn > threshold_continue:
            logging.info(
                f"{threshold_continue} turns have passed. No prolonging required."
            )
        else:
            if count_non_tool_output > 0:
                # encourage model to explore more if model's output is only answer
                logging.info(
                    "Answer is provided, but let's encourage the model to explore more."
                )
                content = template_components["user"]["continue"]
            else:
                logging.info(
                    "tool_call detected. let model answer based on the tool result"
                )
                logging.info("Encourage model to think based on tool result.")
                content = "Make sure to review the tool result and think before making any additional tool calls or answering the question (Format: <answer>your answer</answer>)."
    else:
        # encourage model to output answer in the specified format
        if count_turn > threshold_continue:
            logging.info("Let's encourage the model to output an answer")
            content = template_components["user"]["answer"].replace(
                "{question}", example["question"]
            )
        else:
            if (
                count_non_tool_output >= threshold_max_text_turn
                and count_turn <= threshold_continue
            ):
                logging.info("Let's encourage the model to continue its exploration")
                # if some consecutive non-function outputs and not enough tool calling
                # encourage model to explore more
                content = template_components["user"]["continue"]
            else:
                if count_non_tool_output == 0:
                    logging.info("Encourage model to think based on tool result.")
                    content = "Make sure to review the tool result and think before making any additional tool calls or answering the question (Format: <answer>your answer</answer>)."
                else:
                    logging.info("No need right now")

    if content:
        message = {"role": "user", "content": content}
        message_text = ["user", content]
    else:
        message = None
        message_text = ""

    return message, message_text


def run_single_example(args, example, template_components, tools, instructions):
    count_turn = 0  # count conversational turns
    count_non_tool_output = 0
    conversation_history = []
    messages = []

    # register turn 0
    system_developer_message = template_components["system"]
    messages.append({"role": "system", "content": system_developer_message})
    conversation_history.append(["system", system_developer_message])

    initial_user_message, initial_user_message_text = format_initial_user_message(
        args.model_id, template_components, example
    )
    messages.append(initial_user_message)
    conversation_history.append(["user", initial_user_message_text])
    message_types_in_response = ["init"]

    flag_answer_predicted = False
    answer = ""
    flag_end_conversation = False
    while count_turn < args.max_turn:
        logging.info(f"Next: turn {count_turn + 1}")

        # api call
        try:
            response, usage = call_api_single(args, "", messages, tools, "auto")
        except Exception:
            logging.exception("Error in run_single_example")
            conversation_history.append(["Error"])
            continue

        latest_assistant_response = ""
        current_turn = []
        message_types_in_response = []
        if len(response.choices) > 1:  # sanity check
            logging.warning("[Debug] Multiple choices/candidates in response")

        message = response.choices[0].message

        if message.content:
            messages.append({"role": "assistant", "content": message.content})
            current_turn.append(["assistant", message.content])
            latest_assistant_response = message.content
            message_types_in_response.append("text")

        if message.tool_calls:
            # list
            if len(message.tool_calls) > 1:
                logging.warning(f"Multiple tools are called: {message.tool_calls=}")
            tool_call = message.tool_calls[0]
            logging.info(f"debug {tool_call=}")
            messages.append(
                {
                    "role": "assistant",
                    "tool_calls": [
                        {
                            "id": tool_call.id,
                            "type": tool_call.type,
                            "function": tool_call.function,
                        }
                    ],
                }
            )
            current_turn.append(
                [
                    "assistant",
                    f"{tool_call.function.name} {tool_call.function.arguments}",
                ]
            )

            # function call
            tool_response_messages, tool_response_messages_text, _, _ = call_function(
                args,
                example,
                template_components,
                instructions,
                tool_call.function,
            )
            if tool_response_messages:
                messages.extend(tool_response_messages)
                current_turn.extend(tool_response_messages_text)
            else:
                logging.error("no output returned from function call")
                sys.exit("Force Quit (function call no output)")
            message_types_in_response.append("tool_call")

        logging.info(f"{message_types_in_response=}")

        # check if answer is produced
        if "<answer>" in latest_assistant_response:
            flag_answer_predicted = True
            answer = latest_assistant_response.split("<answer>")[-1].replace(
                "</answer>", ""
            )

        # check output types
        if "tool_call" in message_types_in_response:
            count_non_tool_output = 0
        else:
            count_non_tool_output += 1

        # manual cut-in to prolong or end conversation
        message_cut_in, message_cut_in_text = cut_in(
            args,
            example,
            template_components,
            count_turn,
            count_non_tool_output,
            flag_answer_predicted,
        )
        if message_cut_in:
            messages.append(message_cut_in)
            current_turn.append(message_cut_in_text)

        # register conv history in text
        conversation_history.append(current_turn)
        for role, content in current_turn:
            logging.info(f"{role=}")
            logging.info(f"{content=}")

        # termination condition
        if count_turn >= args.max_turn:
            logging.info(f"Reached {args.max_turn=}. Terminate this process.")
            flag_end_conversation = True
        else:
            if flag_answer_predicted:
                if count_turn > args.threshold_continue:
                    if count_non_tool_output > 0:
                        logging.info(
                            "Answer predicted & Latest response contains no tool call. "
                            "So finish now."
                        )
                        flag_end_conversation = True
                    else:
                        logging.info(
                            f"Although turn {args.threshold_continue} reached, "
                            "as a tool is called, continue this process"
                        )
                else:
                    pass
            else:
                pass

        if flag_end_conversation:
            if len(answer) < 1 and "</think>" not in latest_assistant_response:
                answer = "Error: No answer was generated."
            break

        count_turn += 1

    return messages, conversation_history, answer


def check_if_prediction_exists(example, examples_prev):
    corresponding_example = None
    if "examples" in examples_prev:
        for example_prev in examples_prev["examples"]:
            if (
                "question" in example_prev
                and example["question"] == example_prev["question"]
            ):
                if (
                    "prediction" in example_prev
                    and example_prev["prediction"]["answer"].strip()
                ):
                    corresponding_example = deepcopy(example_prev)

    return corresponding_example


def main(args):
    # load data
    with open(args.filepath_qa, "r") as f:
        examples = json.load(f)

    # load instruction
    toy2instruction = format_instruction(
        args.filepath_instruction,
        args.dirpath_instruction_image,
        args.dirpath_parts_image,
    )

    # load prompt template
    template_components = load_prompt_template(args.filepath_template)

    # load tools
    tools = load_tools(args)
    new_examples = []

    # output file
    filepath_output = (
        args.dirpath_output / f"{Path(args.model_id).name}_text_{args.reasoning}.json"
    )
    if filepath_output.exists():
        with open(filepath_output, "r") as f:
            examples_prev = json.load(f)
        logging.info(
            f"Prev attempt exists. Restart from {len(examples_prev['examples'])}"
        )
    else:
        logging.info("Initial attempt")
        examples_prev = []

    metadata = {
        "data-created": get_date(),
        "args": {k: str(v) if isinstance(v, Path) else v for k, v in vars(args).items()},
    }

    # start inference
    logging.info("Prediction starts ...")
    for idx, example in enumerate(examples):
        logging.info(f"[Example {idx}]")

        # check if prediction already exists
        corresponding_example_prev = check_if_prediction_exists(example, examples_prev)
        if corresponding_example_prev:
            logging.info("Previous prediction exists. Skip.")
            new_examples.append(corresponding_example_prev)
        else:
            try:
                messages, conversation_history, answer = run_single_example(
                    args,
                    example,
                    template_components,
                    tools,
                    toy2instruction[example["toy_id"]],
                )
                new_example = deepcopy(example)
                new_example["prediction"] = {
                    "conversation_history": conversation_history,
                    "answer": answer,
                }
                new_examples.append(new_example)
            except Exception:
                logging.exception("An error occurred")
                new_examples.append("Error")

        # save output every time
        final_output = {"metadata": metadata, "examples": new_examples}
        with open(filepath_output, "w") as f:
            json.dump(final_output, f, indent=4)
            f.write("\n")


if __name__ == "__main__":
    parser = ArgumentParser(description="prototype")
    parser.add_argument("--filepath_qa", type=Path, help="filepath for qa input")
    parser.add_argument(
        "--filepath_instruction", type=Path, help="filepath for instruction"
    )
    parser.add_argument(
        "--dirpath_instruction_image", type=Path, help="dirpath for instruction (image)"
    )
    parser.add_argument(
        "--dirpath_parts_image", type=Path, help="dirpath for parts (image)"
    )
    parser.add_argument("--dirpath_frame", type=Path, help="dirpath for frames")
    parser.add_argument(
        "--filepath_template", type=Path, help="filepath for prompt template"
    )
    parser.add_argument(
        "--dirpath_intermediate_output",
        type=Path,
        help="dirpath to store intermediate outputs",
    )
    parser.add_argument("--filepath_tool", type=Path, help="filepath for tools")
    parser.add_argument("--dirpath_output", type=Path, help="filepath for output")
    parser.add_argument("--model_id", type=str, help="model id", default="")
    parser.add_argument("--temperature", type=float, help="temperature", default=0.1)
    parser.add_argument(
        "--max_tokens", type=int, help="max tokens to generate", default=2560
    )
    parser.add_argument("--max_frames", type=int, help="max frames to feed", default=30)
    parser.add_argument(
        "--max_total_images",
        type=int,
        help="max total frames to feed (100 is for anthropic api, but use 80 for 32MB limit)",
        default=70,
    )
    parser.add_argument("--max_turn", type=int, help="max turns", default=10)
    parser.add_argument(
        "--threshold_continue",
        type=int,
        help="the minimum turn to spend before generating answers for each example",
        default=5,
    )
    parser.add_argument(
        "--threshold_max_text_turn",
        type=int,
        help="the max #turn of non-function outputs",
        default=2,
    )
    # default true
    parser.add_argument("--reasoning", action="store_true", help="Enable reasoning")
    parser.add_argument(
        "--reasoning_effort",
        type=str,
        help="reasoning level for openAI models",
        default="medium",
    )
    parser.add_argument(
        "--summary_type", type=str, help="summary type for openai api", default="auto"
    )
    parser.add_argument(
        "--budget_tokens",
        type=int,
        help="thinking budget tokens for anthropic api",
        default=2048,
    )
    parser.add_argument("--angle", type=str, help="angle", default="C10118_rgb")
    parser.add_argument("--resolution", type=str, help="resolution", default="360p")
    parser.add_argument("--color", type=str, help="color", default="rgb")
    parser.add_argument(
        "--api_key_for_qwen", type=str, help="dummy api key for qwen", default="dummpy"
    )
    parser.add_argument(
        "--base_url_qwen",
        type=str,
        help="port for qwen",
        default="http://localhost:8000/v1",
    )
    parser.add_argument(
        "--base_url_mimo",
        type=str,
        help="port for mimo",
        default="http://localhost:8088/v1",
    )
    parser.add_argument(
        "--base_url_internvl",
        type=str,
        help="port for internvl",
        default="http://localhost:8007/v1",
    )
    parser.add_argument("--wait_time", type=int, help="API call wait time", default=10)
    parser.add_argument("--dirpath_log", type=Path, help="dirpath for log")

    args = parser.parse_args()

    if not args.dirpath_log.exists():
        args.dirpath_log.mkdir()

    if not args.dirpath_output.exists():
        args.dirpath_output.mkdir(parents=True)

    logging.basicConfig(
        format="%(asctime)s:%(levelname)s - %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        level=logging.INFO,
        handlers=[
            logging.StreamHandler(),
            logging.FileHandler(
                args.dirpath_log / f"prototype_mimo_text_{get_date()}.log"
            ),
        ],
    )

    logging.info("Arguments:")
    logging.info(pformat(vars(args), indent=4, width=100))

    main(args)
