import os
import json

# Define base directory and output directory
base_dir = '.'  # Current directory
output_dir = os.path.join(base_dir, 'voting_results')
os.makedirs(output_dir, exist_ok=True)

"""
Assumed txt file format:
Each txt file contains multiple matrices separated by empty lines.
Each matrix section starts with a number followed by a colon (e.g., "1:").
After the header line, each line represents a row of the matrix with 0s and 1s.

Example:
1:
010
001
000

2:
0100
0011
0000
0000

...

1000:
010
001
000
"""

# Dictionary to store all matrices
all_matrices = {}

# Process each annotator folder
for folder_idx in range(11):  # annotator_0 to annotator_10
    folder = f'annotator_{folder_idx}'
    folder_path = os.path.join(base_dir, folder)

    # Process all txt files in the folder
    for file_idx in range(11):  # annotator_0.txt to annotator_10.txt
        txt_file = f'annotator_{file_idx}.txt'
        file_path = os.path.join(folder_path, txt_file)

        # Skip if file doesn't exist (silently)
        if not os.path.exists(file_path):
            continue

        # Process file content
        with open(file_path, 'r') as f:
            content = f.read()

        # Parse matrices from the file
        sections = content.split('\n\n')
        for section in sections:
            lines = section.strip().split('\n')
            if not lines:
                continue

            # Extract matrix number (e.g., "1:", "2:", etc.)
            matrix_num = None
            matrix_lines = []
            for line in lines:
                if ':' in line:
                    matrix_num = line.strip().rstrip(':')
                elif matrix_num is not None:  # Only collect matrix rows after finding the matrix number
                    matrix_lines.append(line)

            if matrix_num is None or not matrix_lines:
                continue

            # Create matrix
            matrix = []
            for line in matrix_lines:
                row = [int(char) for char in line]
                matrix.append(row)

            # Store matrix
            if matrix_num not in all_matrices:
                all_matrices[matrix_num] = []

            all_matrices[matrix_num].append(matrix)

# Perform voting on matrices
voted_results = {}

for matrix_num, matrices in all_matrices.items():
    # Determine maximum dimensions of the matrix
    max_height = max(len(matrix) for matrix in matrices)
    max_width = max(len(row) for matrix in matrices for row in matrix)

    # Create new matrix to store voting results
    voted_matrix = []
    for i in range(max_height):
        voted_row = []
        for j in range(max_width):
            # Count zeros at this position
            count_zeros = sum(1 for matrix in matrices
                              if i < len(matrix) and j < len(matrix[i]) and matrix[i][j] == 0)

            # Determine majority vote result
            # If the number of zeros is greater than or equal to the number of ones, result is 0, otherwise 1
            voted_value = 0 if count_zeros * 2 >= len(matrices) else 1
            voted_row.append(voted_value)
        voted_matrix.append(voted_row)

    voted_results[matrix_num] = voted_matrix

# Save results as JSON files
for matrix_num, matrix in voted_results.items():
    output_file = os.path.join(output_dir, f'{matrix_num}.json')
    with open(output_file, 'w') as f:
        json.dump(matrix, f, indent=2)

print(f"Voting completed. Results saved to {output_dir} directory")