/**
 * Copyright (c) 2014-present, Facebook, Inc. All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 *
 * 
 */

'use strict';

const chalk = require('chalk');
const BULLET = chalk.bold('\u25cf');
const DEPRECATION = `${BULLET} Deprecation Warning`;
const ERROR = `${BULLET} Validation Error`;
const WARNING = `${BULLET} Validation Warning`;

const format = value =>
typeof value === 'function' ?
value.toString() :
require('pretty-format')(value, { min: true });

class ValidationError extends Error {



  constructor(name, message, comment) {
    super();
    comment = comment ? '\n\n' + comment : '\n';
    this.name = '';
    this.message = chalk.red(chalk.bold(name) + ':\n\n' + message + comment);
    Error.captureStackTrace(this, () => {});
  }}


const logValidationWarning = (
name,
message,
comment) =>
{
  comment = comment ? '\n\n' + comment : '\n';
  console.warn(chalk.yellow(chalk.bold(name) + ':\n\n' + message + comment));
};

const createDidYouMeanMessage = (
unrecognized,
allowedOptions) =>
{
  const leven = require('leven');
  const suggestion = allowedOptions.find(option => {
    const steps = leven(option, unrecognized);
    return steps < 3;
  });

  return suggestion ?
  `Did you mean ${chalk.bold(format(suggestion))}?` :
  '';
};

module.exports = {
  DEPRECATION,
  ERROR,
  ValidationError,
  WARNING,
  createDidYouMeanMessage,
  format,
  logValidationWarning };