import torch
import torch.nn as nn
import torch.nn.functional as F
from copy import deepcopy
from collections import OrderedDict
from rotograd import RotoGrad


# task_branches = []


# def is_branches():


def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=dilation, groups=groups, bias=False,
                     dilation=dilation)


def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return nn.Conv2d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)


class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity

        return out


class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):
    def __init__(self, block, layers, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 norm_layer=None):
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm2d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv2d(1, 64, kernel_size=(7, 7), stride=(2, 2), padding=(3, 3), bias=False)

        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool2d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool2d((1, 1))
        self.fc = nn.Linear(512 * block.expansion, num_classes)

        self._shared_parameters = OrderedDict()

        # self.turn()

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, stride=1, dilate=False):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    def shared_modules(self):
        return self

    def shared_parameters(self):
        return self._shared_parameters

    # def turn(self, task_branches=[]):
    #     self._shared_parameters.clear()
    #
    #     if self.n_tasks > 1:
    #         for idx, m in self.named_modules():
    #             if idx in task_branches:
    #                 m.set_n_tasks(n_tasks=self.n_tasks)
    #
    #     task_branches = task_branches
    #
    #     # obtain the shared parameters
    #     for idx, m in self.named_modules():
    #         if '.m_list' in idx:
    #             idx = idx[:idx.index('.m_list')]
    #
    #         if idx not in task_branches:
    #             members = m._parameters.items()
    #             memo = set()
    #             for k, v in members:
    #                 if v is None or v in memo:
    #                     continue
    #                 memo.add(v)
    #                 name = idx + ('.' if idx else '') + k
    #                 self._shared_parameters[name] = v

    def _forward_impl(self, x):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = torch.flatten(x, 1)
        x = self.fc(x)

        return x

    def forward(self, x):
        return self._forward_impl(x)


def resnet18(num_classes=1000):
    return ResNet(BasicBlock, [2, 2, 2, 2], num_classes=num_classes)


class Model(nn.Module):
    def __init__(self, backbone, pred1, pred2):
        super(Model, self).__init__()
        self.backbone = backbone
        # self.ReLU = nn.ReLU(inplace=True)
        self.pred1 = pred1
        self.pred2 = pred2
        self._shared_parameters = OrderedDict()

    # def turn(self, task_branches=[]):
    #     self._shared_parameters.clear()
    #
    #     if self.n_tasks > 1:
    #         for idx, m in self.named_modules():
    #             if idx in task_branches:
    #                 m.set_n_tasks(n_tasks=self.n_tasks)
    #
    #     task_branches = task_branches
    #
    #     # obtain the shared parameters
    #     for idx, m in self.named_modules():
    #         if '.m_list' in idx:
    #             idx = idx[:idx.index('.m_list')]
    #
    #         if idx not in task_branches:
    #             members = m._parameters.items()
    #             memo = set()
    #             for k, v in members:
    #                 if v is None or v in memo:
    #                     continue
    #                 memo.add(v)
    #                 name = idx + ('.' if idx else '') + k
    #                 self._shared_parameters[name] = v

    def shared_parameters(self):
        return self._shared_parameters

    def forward(self, x):
        x = self.backbone(x)
        # x = self.ReLU(x)
        y1 = self.pred1(x)
        y2 = self.pred2(x)

        return y1, y2


def MnistResNet_RotoGrad(latent_size):
    backbone = nn.Sequential(resnet18(num_classes=100), nn.ReLU(inplace=True))
    # backbone = resnet18(num_classes=100)

    pred1 = nn.Linear(100, 10)
    pred2 = nn.Linear(100, 10)

    # rotoGrad = Model(backbone, pred1, pred2)
    rotoGrad = RotoGrad(backbone=backbone, heads=[pred1, pred2], latent_size=latent_size, normalize_losses=True)

    return rotoGrad


# class MnistResNet_RotoGrad(torch.nn.Module):
#     def __init__(self, latent_size):
#         super(MnistResNet_RotoGrad, self).__init__()
#         # self.n_tasks = n_tasks
#
#         backbone = nn.Sequential(resnet18(num_classes=100), nn.ReLU(inplace=True))
#
#         pred1 = nn.Linear(100, 10)
#         pred2 = nn.Linear(100, 10)
#         self.rotoGrad = RotoGrad(backbone=backbone, heads=[pred1, pred2], latent_size=latent_size)
#
#     # def shared_parameters(self):
#     #     return self.feature_extractor.shared_parameters()
#     #
#     # def turn(self, task_branches):
#     #     self.feature_extractor.turn(task_branches)
#     #
#     # def zero_grad_shared_modules(self):
#     #     for name, p in self.shared_parameters().items():
#     #         if p.grad is not None:
#     #             if p.grad.grad_fn is not None:
#     #                 p.grad.detach_()
#     #             else:
#     #                 p.grad.requires_grad_(False)
#     #             p.grad.zero_()
#     #
#     # def model_size(self):
#     #     param_size = 0
#     #     for param in self.parameters():
#     #         param_size += param.nelement() * param.element_size()
#     #
#     #     size_all_mb = param_size / 1024 ** 2
#     #     return size_all_mb
#
#     def forward(self, x):
#         x = self.rotoGrad(x)
#         return x


def get_layer_dict(m):
    name_list = []
    for i, mm in enumerate(m.shared_modules()):
        for name, p in mm.named_parameters():
            name_list.append(name)

    layer_dict = {}
    for i, name in enumerate(name_list):
        if '.weight' in name:
            name = name.replace('.m_list.0.weight', '')
        elif '.bias' in name:
            name = name.replace('.m_list.0.bias', '')

        if name not in layer_dict:
            layer_dict[name] = [i]
        else:
            layer_dict[name].append(i)

    return layer_dict


if __name__ == '__main__':
   net = MnistResNet_RotoGrad()
   input = torch.rand(4, 1, 36, 36)
   output = net(input)
   print('finished!')