import torch
import torch.nn as nn
import torch.nn.functional as F
from copy import deepcopy
from collections import OrderedDict
import json


# task_branches = []


# def is_branches():


class Linear_fw(nn.Module):  # used in MAML to forward input with fast weight
    def __init__(self, in_features, out_features, n_tasks=1):
        super(Linear_fw, self).__init__()
        self.n_tasks = n_tasks
        self.m_list = nn.ModuleList([nn.Linear(in_features, out_features) for i in range(n_tasks)])

    def set_n_tasks(self, n_tasks=1):
        if n_tasks >= self.n_tasks:
            gap = n_tasks - self.n_tasks
            self.n_tasks = n_tasks
            for i in range(gap):
                module = deepcopy(self.m_list[0])
                self.m_list.append(module)
        else:
            raise ValueError('Can not decrease the number of tasks in fw module')


    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        if len(x) == 1 and self.n_tasks > 1:
            for i, ln in enumerate(self.m_list):
                o = ln(x[0])
                out.append(o)
        elif len(x) > 1 and self.n_tasks == 1:
            for i, x_i in enumerate(x):
                o = self.m_list[0](x_i)
                out.append(o)
        elif len(x) == self.n_tasks:
            for i, ln in enumerate(self.m_list):
                o = ln(x[i])
                out.append(o)
        else:
            raise ValueError('Error')

        return out

class Conv2d_fw(nn.Module):  # used in MAML to forward input with fast weight
    def __init__(self, in_channels, out_channels, kernel_size, stride=1, padding=0, bias=True, groups=1, dilation=1, n_tasks=1):
        super(Conv2d_fw, self).__init__()
        self.n_tasks = n_tasks
        self.m_list = nn.ModuleList(
            [nn.Conv2d(in_channels, out_channels, kernel_size, stride=stride, padding=padding, dilation=dilation, groups=groups, bias=bias) for i in range(n_tasks)])
    
    def set_n_tasks(self, n_tasks=1):
        if n_tasks >= self.n_tasks:
            gap = n_tasks - self.n_tasks
            self.n_tasks = n_tasks
            for i in range(gap):
                module = deepcopy(self.m_list[0])
                self.m_list.append(module)
        else:
            raise ValueError('Can not decrease the number of tasks in fw module')
    
    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        if len(x) == 1 and self.n_tasks > 1:
            for i, conv in enumerate(self.m_list):
                o = conv(x[0])
                out.append(o)
        elif len(x) > 1 and self.n_tasks == 1:
            for i, x_i in enumerate(x):
                o = self.m_list[0](x_i)
                out.append(o)
        elif len(x) == self.n_tasks:
            for i, conv in enumerate(self.m_list):
                o = conv(x[i])
                out.append(o)
        else:
            raise ValueError('Error')
        return out

class BatchNorm2d_fw(nn.Module):  # used in MAML to forward input with fast weight
    def __init__(self, num_features, n_tasks=1):
        super(BatchNorm2d_fw, self).__init__()
        self.n_tasks = n_tasks
        self.m_list = nn.ModuleList([nn.BatchNorm2d(num_features) for i in range(n_tasks)])

    def set_n_tasks(self, n_tasks=1):
        if n_tasks >= self.n_tasks:
            gap = n_tasks - self.n_tasks
            self.n_tasks = n_tasks
            for i in range(gap):
                module = deepcopy(self.m_list[0])
                self.m_list.append(module)
        else:
            raise ValueError('Can not decrease the number of tasks in fw module')

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        if len(x) == 1 and self.n_tasks > 1:
            for i, bn in enumerate(self.m_list):
                o = bn(x[0])
                out.append(o)
        elif len(x) > 1 and self.n_tasks == 1:
            for i, x_i in enumerate(x):
                o = self.m_list[0](x_i)
                out.append(o)
        elif len(x) == self.n_tasks:
            for i, bn in enumerate(self.m_list):
                o = bn(x[i])
                out.append(o)
        else:
            raise ValueError('Error')

        return out

class ReLU_fw(nn.ReLU):
    def __init__(self, inplace=False):
        super(ReLU_fw, self).__init__(inplace)

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]
            
        out = []
        for x_i in x:
            out.append(super(ReLU_fw, self).forward(x_i))

        return out

class Sigmoid_fw(nn.Sigmoid):
    def __init__(self, inplace=False):
        super(Sigmoid_fw, self).__init__(inplace)

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        for x_i in x:
            out.append(super(Sigmoid_fw, self).forward(x_i))

        return out

class MaxPool2d_fw(nn.MaxPool2d):
    def __init__(self, kernel_size, stride=None, padding=0, dilation=1, return_indices=False, ceil_mode=False):
        super(MaxPool2d_fw, self).__init__(kernel_size, stride, padding, dilation, return_indices, ceil_mode)

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        for x_i in x:
            out.append(super(MaxPool2d_fw, self).forward(x_i))

        return out

class AdaptiveAvgPool2d_fw(nn.AdaptiveAvgPool2d):
    def __init__(self, output_size):
        super(AdaptiveAvgPool2d_fw, self).__init__(output_size)

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        out = []
        for x_i in x:
            out.append(super(AdaptiveAvgPool2d_fw, self).forward(x_i))

        return out

def conv3x3(in_planes, out_planes, stride=1, groups=1, dilation=1):
    """3x3 convolution with padding"""
    return Conv2d_fw(in_planes, out_planes, kernel_size=3, stride=stride, padding=dilation, groups=groups, bias=False, dilation=dilation)

def conv1x1(in_planes, out_planes, stride=1):
    """1x1 convolution"""
    return Conv2d_fw(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)

class BasicBlock(nn.Module):
    expansion = 1

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = BatchNorm2d_fw
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = ReLU_fw(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        if not isinstance(x, list):
            x = [x]

        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        if len(identity) == len(out):
            for i in range(len(identity)):
                out[i] += identity[i]
        elif len(identity) == 1 and len(out) > 1:
            for i in range(len(identity)):
                out[i] += identity[0]
        elif len(identity) > 1 and len(out) == 1:
            o = []
            for i in range(len(identity)):
                o.append(out[0]+identity[i])
            out = o

        out = self.relu(out)

        return out

class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion = 4

    def __init__(self, inplanes, planes, stride=1, downsample=None, groups=1,
                 base_width=64, dilation=1, norm_layer=None):
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = BatchNorm2d_fw
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = ReLU_fw(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x):
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out

class ResNet_fw(nn.Module):
    def __init__(self, block, layers, num_classes=1000, zero_init_residual=False,
                 groups=1, width_per_group=64, replace_stride_with_dilation=None,
                 norm_layer=None, n_tasks=1, branched='empty', topK=1, ablation_file=None):
        super(ResNet_fw, self).__init__()
        if norm_layer is None:
            norm_layer = BatchNorm2d_fw
        self._norm_layer = norm_layer

        self.n_tasks = n_tasks
        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = Conv2d_fw(1, 64, kernel_size=(7, 7), stride=(2, 2), padding=(3, 3), bias=False)

        self.bn1 = norm_layer(self.inplanes)
        self.relu = ReLU_fw(inplace=True)
        self.maxpool = MaxPool2d_fw(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = AdaptiveAvgPool2d_fw((1, 1))
        self.fc = Linear_fw(512 * block.expansion, num_classes)

        self._shared_parameters = OrderedDict()

        if branched == 'empty':
            task_branches = []
        if branched == 'bmtas':
            task_branches = [
                'layer4.0.bn1',
                'layer4.0.conv1',
                'layer3.0.bn2',
                'layer3.1.bn1',
                'layer3.0.downsample.1',
                'layer3.1.bn2',
                'layer2.1.bn1',
                'layer4.1.bn1',
                'layer4.1.conv1',
                'layer4.0.conv2',
                'layer3.1.conv2',
                'layer4.0.downsample.0',
                'layer2.1.bn2',
                'layer3.0.bn1',
                'layer3.0.downsample.0',
                'layer3.1.conv1',
                'layer2.0.bn2',
                'layer4.0.downsample.1',
                'layer3.0.conv2',
                'layer2.0.bn1',
                'layer4.0.bn2',
                'layer2.0.downsample.1',
                'layer1.1.bn1',
                'fc',
                'layer4.1.bn2',
                'layer4.1.conv2',
                'layer1.1.bn2',
                'layer3.0.conv1',
                'layer2.1.conv2',
                'layer2.0.downsample.0',
                'layer2.1.conv1',
                'layer1.0.bn1',
                'layer2.0.conv1',
                'layer2.0.conv2',
                'layer1.0.bn2',
                'layer1.1.conv1',
                'layer1.1.conv2',
                'layer1.0.conv1',
                'layer1.0.conv2'
            ]
        elif branched == 'ablation':
            with open(ablation_file, "r") as fp:
                task_branches = json.load(fp)
        elif branched == 'branch':
            # seed0 cagrad SGD 40 epoch lw_cos L=-0.1
            task_branches = ['bn1',
'layer1.0.bn1',
'layer1.1.bn1',
'layer1.0.bn2',
'layer1.1.bn2',
'layer2.0.bn1',
'layer2.1.bn1',
'conv1',
'layer2.0.bn2',
'layer2.0.downsample.1',
'layer2.1.bn2',
'layer3.0.bn1',
'layer1.0.conv1',
'layer3.1.bn1',
'layer4.1.bn1',
'layer4.0.bn1',
'layer3.0.bn2',
'layer3.0.downsample.1',
'layer4.1.bn2',
'layer3.1.bn2',
'layer4.0.downsample.1',
'layer4.0.bn2',
'layer1.0.conv2',
'layer1.1.conv1',
'layer1.1.conv2',
'fc',
'layer2.0.downsample.0',
'layer2.0.conv1',
'layer4.1.conv2',
'layer2.0.conv2',
'layer2.1.conv2',
'layer3.0.conv1',
'layer3.0.downsample.0',
'layer2.1.conv1',
'layer4.0.conv1',
'layer3.1.conv2',
'layer4.1.conv1',
'layer3.1.conv1',
'layer4.0.downsample.0',
'layer3.0.conv2',
'layer4.0.conv2',
]

            # seed0 cagrad SGD 40 epoch S=0.0
            # task_branches = [
            #     'layer4.0.bn1',
            #     'layer4.0.conv1',
            #     'layer3.0.bn2',
            #     'layer3.1.bn1',
            #     'layer3.0.downsample.1',
            #     'layer3.1.bn2',
            #     'layer2.1.bn1',
            #     'layer4.1.bn1',
            #     'layer4.1.conv1',
            #     'layer4.0.conv2',
            #     'layer3.1.conv2',
            #     'layer4.0.downsample.0',
            #     'layer2.1.bn2',
            #     'layer3.0.bn1',
            #     'layer3.0.downsample.0',
            #     'layer3.1.conv1',
            #     'layer2.0.bn2',
            #     'layer4.0.downsample.1',
            #     'layer3.0.conv2',
            #     'layer2.0.bn1',
            #     'layer4.0.bn2',
            #     'layer2.0.downsample.1',
            #     'layer1.1.bn1',
            #     'fc',
            #     'layer4.1.bn2',
            #     'layer4.1.conv2',
            #     'layer1.1.bn2',
            #     'layer3.0.conv1',
            #     'layer2.1.conv2',
            #     'layer2.0.downsample.0',
            #     'layer2.1.conv1',
            #     'layer1.0.bn1',
            #     'layer2.0.conv1',
            #     'layer2.0.conv2',
            #     'bn1',
            #     'layer1.0.bn2',
            #     'layer1.1.conv1',
            #     'layer1.1.conv2',
            #     'layer1.0.conv1',
            #     'layer1.0.conv2',
            #     'conv1'
            # ]

            # seed0 cagrad SGD 40 epoch S=-0.05
            # task_branches = [
            #     'bn1',
            #     'layer1.1.bn1',
            #     'layer1.0.bn1',
            #     'layer1.0.bn2',
            #     'layer1.1.bn2',
            #     'layer2.1.bn1',
            #     'layer2.0.bn1',
            #     'layer2.0.bn2',
            #     'layer2.0.downsample.1',
            #     'conv1',
            #     'layer2.1.bn2',
            #     'layer3.1.bn1',
            #     'layer3.0.bn1',
            #     'layer1.0.conv1',
            #     'layer4.1.bn1',
            #     'layer3.0.downsample.1',
            #     'layer3.0.bn2',
            #     'layer4.0.bn1',
            #     'layer3.1.bn2',
            #     'layer4.1.bn2',
            #     'layer4.0.downsample.1',
            #     'layer4.0.bn2',
            #     'layer1.0.conv2',
            #     'layer1.1.conv1',
            #     'layer4.1.conv2',
            #     'fc'
            #     'layer1.1.conv2',
            #     'layer2.0.downsample.0',
            #     'layer4.1.conv1',
            #     'layer4.0.conv1',
            #     'layer3.0.downsample.0',
            #     'layer2.0.conv1',
            #     'layer4.0.conv2',
            #     'layer2.0.conv2',
            #     'layer2.1.conv2',
            #     'layer3.1.conv1',
            #     'layer4.0.downsample.0',
            #     'layer2.1.conv1',
            #     'layer3.1.conv2',
            #     'layer3.0.conv1',
            #     'layer3.0.conv2'
            # ]

            # seed0 cagrad SGD 40 epoch S=-0.15
            # task_branches = [
            #     'bn1',
            #     'layer1.0.bn1',
            #     'layer1.1.bn1',
            #     'layer1.0.bn2',
            #     'layer1.1.bn2',
            #     'conv1',
            #     'layer2.0.bn1',
            #     'layer2.1.bn1',
            #     'layer2.0.bn2',
            #     'layer2.0.downsample.1',
            #     'layer2.1.bn2',
            #     'layer1.0.conv1',
            #     'layer3.0.bn1',
            #     'layer3.1.bn1',
            #     'layer4.1.bn1',
            #     'layer4.1.bn2',
            #     'layer3.0.bn2',
            #     'layer3.0.downsample.1',
            #     'layer4.0.bn1',
            #     'layer4.0.downsample.1',
            #     'layer1.0.conv2',
            #     'layer4.0.bn2',
            #     'layer3.1.bn2',
            #     'layer1.1.conv1',
            #     'layer1.1.conv2',
            #     'layer2.0.conv1',
            #     'layer2.0.conv2',
            #     'fc',
            #     'layer2.0.downsample.0',
            #     'layer2.1.conv1',
            #     'layer2.1.conv2',
            #     'layer3.0.conv1',
            #     'layer3.0.downsample.0',
            #     'layer3.0.conv2',
            #     'layer4.0.conv1',
            #     'layer3.1.conv1',
            #     'layer3.1.conv2',
            #     'layer4.0.downsample.0',
            #     'layer4.1.conv1',
            #     'layer4.0.conv2',
            #     'layer4.1.conv2',
            # ]

            # seed0 cagrad SGD 40 epoch S=-0.2
            # task_branches = [
            #     'bn1',
            #     'layer1.0.bn1',
            #     'layer1.1.bn1',
            #     'layer1.0.bn2',
            #     'layer1.1.bn2',
            #     'conv1',
            #     'layer2.0.bn1',
            #     'layer2.1.bn1',
            #     'layer2.0.downsample.1',
            #     'layer2.0.bn2',
            #     'layer2.1.bn2',
            #     'layer1.0.conv1',
            #     'layer3.0.bn1',
            #     'layer3.0.downsample.1',
            #     'layer4.1.bn2',
            #     'layer1.0.conv2',
            #     'layer1.1.conv1',
            #     'layer3.0.bn2',
            #     'layer4.1.bn1',
            #     'layer1.1.conv2',
            #     'layer3.1.bn1',
            #     'layer4.0.bn2',
            #     'layer4.0.downsample.1',
            #     'layer4.0.bn1',
            #     'layer3.1.bn2',
            #     'layer2.0.conv1',
            #     'layer2.0.conv2',
            #     'layer2.0.downsample.0',
            #     'fc',
            #     'layer2.1.conv1',
            #     'layer2.1.conv2',
            #     'layer3.0.conv1',
            #     'layer3.0.conv2',
            #     'layer3.0.downsample.0',
            #     'layer3.1.conv1',
            #     'layer3.1.conv2',
            #     'layer4.0.conv1',
            #     'layer4.0.conv2',
            #     'layer4.0.downsample.0',
            #     'layer4.1.conv1',
            #     'layer4.1.conv2',
            #
            # ]

            # task_branches = ['conv1', 'bn1', 'layer1.0.bn1', 'layer1.0.bn2', 'layer1.0.conv1', 'layer1.0.conv2',
            #                  'layer1.1.bn1', 'layer1.1.bn2', 'layer1.1.conv1', 'layer1.1.conv2', 'layer2.0.bn1',
            #                  'layer2.0.bn2',
            #                  'layer2.0.downsample.1', 'layer2.1.bn1', 'layer2.1.bn2', 'layer3.0.bn1', 'layer3.0.bn2',
            #                  'layer3.0.downsample.1', 'layer3.1.bn1',
            #                  'layer3.1.bn2', 'layer4.0.bn1', 'layer4.0.bn2', 'layer4.1.bn1', 'layer4.1.bn2',
            #                  'layer4.0.downsample.1']

            # lw_task_cos: flood
            # task_branches = ['bn1', 'layer1.0.bn1', 'layer1.1.bn1', 'layer1.1.bn2', 'layer1.0.bn2', 'layer2.1.bn1',
            #                  'layer2.0.bn1', 'conv1', 'layer2.0.downsample.1', 'layer2.0.bn2', 'layer2.1.bn2',
            #                  'layer3.0.bn1', 'layer3.1.bn1', 'layer1.0.conv1', 'layer3.0.downsample.1',
            #                  'layer3.0.bn2', 'layer3.1.bn2', 'layer1.0.conv2', 'layer4.1.bn1', 'layer1.1.conv1', 'fc']

            # lw_cos: avg_angle between task (< 0)
            # task_branches = ['layer4.0.downsample.0', 'layer3.0.conv2', 'layer3.1.conv1', 'layer3.0.conv1', 'layer2.1.conv1',
            #                  'layer3.1.conv2', 'layer2.1.conv2', 'layer3.0.downsample.0', 'layer3.0.bn2', 'layer4.0.conv2']

            # lw_cos: flood
            # task_branches = ['layer4.0.downsample.0', 'layer3.0.bn1', 'layer3.1.bn2', 'layer3.1.bn1', 'layer2.0.bn2',
            #                  'layer3.0.bn2', 'layer2.1.bn2', 'layer2.1.bn1', 'layer2.0.bn1', 'layer2.0.downsample.1',
            #                  'layer3.0.conv1', 'layer3.0.conv2', 'layer3.0.downsample.1']

            # # lw_task_cos_flood:Adam
            # task_branches = ['layer3.1.bn1', 'layer4.0.bn1', 'layer4.1.bn1', 'layer4.1.conv1', 'layer1.0.bn1', 'bn1',
            #                  'layer1.1.bn1', 'layer2.1.bn1', 'conv1', 'layer1.1.bn2', 'layer1.0.bn2', 'layer3.1.conv2',
            #                  'layer4.0.conv1', 'layer2.0.bn1', 'layer2.0.bn2', 'fc']


            # # lw_cos_avg:Adam
            # task_branches = ['layer4.1.conv1', 'fc', 'layer4.1.bn1', 'layer4.0.downsample.0', 'layer4.0.conv2', 'layer4.1.conv2',
            #                  'layer4.0.bn1', 'layer2.1.bn1', 'conv1', 'layer1.1.bn2', 'layer1.0.bn2', 'layer3.1.conv2']

            # task_branches = task_branches[:topK]

            # task_branches = ['bn1', 'layer1.0.bn1', 'layer1.0.bn2',
            #                  'layer1.1.bn1', 'layer1.1.bn2', 'layer2.0.bn1',
            #                  'layer2.0.bn2', 'layer2.1.bn1', 'layer2.1.bn2', 'layer3.0.bn1', 'layer3.0.bn2', 'layer3.1.bn1',
            #                  'layer3.1.bn2', 'layer4.0.bn1', 'layer4.0.bn2', 'layer4.1.bn1', 'layer4.1.bn2']

            # task_branches = ['layer1.1.bn2', 'layer1.1.bn1', 'layer2.0.bn2', 'layer2.0.bn1', 'bn1',
            #                  'layer1.0.bn2', 'layer2.1.bn2', 'layer1.0.bn1', 'layer3.0.bn1', 'layer2.0.downsample.1', 'conv1',
            #                  'layer2.1.bn1', 'layer1.0.conv1', 'layer2.0.conv2', 'layer1.1.conv2', 'layer1.0.conv2']

            task_branches = task_branches[:topK]

        self.turn(task_branches)

        for m in self.modules():
            if isinstance(m, nn.Conv2d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm2d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)

    def _make_layer(self, block, planes, blocks, stride=1, dilate=False):
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    def shared_modules(self):
        return self

    def shared_parameters(self):
        return self._shared_parameters

    def turn(self, task_branches=[]):
        self._shared_parameters.clear()

        if self.n_tasks > 1:
            for idx, m in self.named_modules():
                if idx in task_branches:
                    m.set_n_tasks(n_tasks=self.n_tasks)

        task_branches = task_branches

        # obtain the shared parameters
        for idx, m in self.named_modules():
            if '.m_list' in idx:
                idx = idx[:idx.index('.m_list')]

            if idx not in task_branches:
                members = m._parameters.items()
                memo = set()
                for k, v in members:
                    if v is None or v in memo:
                        continue
                    memo.add(v)
                    name = idx + ('.' if idx else '') + k
                    self._shared_parameters[name] = v

    def _forward_impl(self, x):
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)

        x = self.avgpool(x)
        x = [torch.flatten(x_i, 1) for x_i in x]
        x = self.fc(x)

        return x

    def forward(self, x):
        return self._forward_impl(x)

def resnet18_fw(num_classes=1000, n_tasks=1, branched='empty', topK=1, ablation_file=None):
    return ResNet_fw(BasicBlock, [2, 2, 2, 2], num_classes=num_classes, n_tasks=n_tasks, branched=branched, topK=topK, ablation_file=ablation_file)

class MnistResNet_fw(torch.nn.Module):
    def __init__(self, n_tasks=2, branched='empty', topK=1, ablation_file=None):
        super(MnistResNet_fw, self).__init__()
        # self.n_tasks = n_tasks
        self.feature_extractor = resnet18_fw(num_classes=100, n_tasks=n_tasks, branched=branched, topK=topK, ablation_file=ablation_file)

        self.ReLU = ReLU_fw(inplace=True)
        self.pred = Linear_fw(100, 10, n_tasks=n_tasks)

    def shared_parameters(self):
        return self.feature_extractor.shared_parameters()

    def turn(self, task_branches):
        self.feature_extractor.turn(task_branches)

    def zero_grad_shared_modules(self):
        for name, p in self.shared_parameters().items():
            if p.grad is not None:
                if p.grad.grad_fn is not None:
                    p.grad.detach_()
                else:
                    p.grad.requires_grad_(False)
                p.grad.zero_()

    def model_size(self, unit='MB'):
        param_size = 0
        for param in self.parameters():
            param_size += param.nelement() * param.element_size()

        if unit == 'MB':
            size_all_out = param_size / 1024 ** 2
        elif unit == 'B':
            size_all_out = param_size
        else:
            raise ValueError(f'Error: Do not support unit {unit}')

        return size_all_out

    def forward(self, x):
        x = self.feature_extractor(x)
        x = self.ReLU(x)
        x = self.pred(x)

        return x


if __name__ == '__main__':
    # net = MnistResNet_fw(branched='branch')
    # sz = net.model_size()
    # print(f'branch: {sz} MB')

    # net = MnistResNet_fw(branched='empty')
    # normal_sz = net.model_size()
    # print(f'Normal: {normal_sz:.2f} MB')
    #
    # net = MnistResNet_fw(n_tasks=1, branched='empty')
    # sz = net.model_size()
    # print(f'STL: {sz * 2:.2f} MB')
    #
    # net = MnistResNet_fw(branched='branch', topK=25)
    # sz = net.model_size()
    # print(f'Recon: {sz:.2f} MB')
    #
    # net = MnistResNet_fw(branched='bmtas')
    # sz = net.model_size()
    # print(f'BMTAS: {sz:.2f} MB')
    #
    # n_tasks = 2
    # roto_param = nn.Parameter(torch.eye(100))
    # roto_param_size = roto_param.nelement() * roto_param.element_size() * n_tasks / 1024 ** 2
    #
    # print(f'Roto: {roto_param_size + normal_sz:.2f}')


    def parameter_sz(p_list, unit='B'):
        param_size = 0
        for param in p_list:
            param_size += param.nelement() * param.element_size()

        if unit == 'MB':
            size = param_size / 1024 ** 2
        elif unit == 'B':
            size = param_size
        else:
            raise ValueError(f'Error: Do not support unit {unit}')

        return size
    #
    #
    def get_layer_dict(m):
        shared_parameters = m.shared_parameters()
        name_list = list(shared_parameters.keys())
        param_list = list(shared_parameters.values())

        layer_dict = {}
        for i, name in enumerate(name_list):
            if '.weight' in name:
                name = name.replace('.weight', '')
            elif '.bias' in name:
                name = name.replace('.bias', '')

            if name not in layer_dict:
                layer_dict[name] = [param_list[i]]
            else:
                layer_dict[name].append(param_list[i])

        return layer_dict

    import numpy as np
    import random


    seed = 0
    np.random.seed(seed)
    random.seed(seed)
    #
    # randomly select layers and turn them to task-specific parameters
    net = MnistResNet_fw(branched='empty')
    layer_dict = get_layer_dict(net)
    layer_names = list(layer_dict.keys())

    # fist K layers
    topK = 25
    # first_K_layers = layer_names[:topK]
    # saved_file = f'./saved/seed{seed}_first_{topK}Layers.json'
    # with open(saved_file, "w") as fp:
    #     json.dump(first_K_layers, fp)
    #
    # last_K_layers = layer_names[-topK:]
    # saved_file = f'./saved/seed{seed}_last_{topK}Layers.json'
    # with open(saved_file, "w") as fp:
    #     json.dump(last_K_layers, fp)

    saved_file = f'./saved/seed{seed}_first_{topK}Layers.json'
    with open(saved_file, "r") as fp:
        b = json.load(fp)
    net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    print(f'first{topK} sz: {net.model_size():.2f}')

    saved_file = f'./saved/seed{seed}_last_{topK}Layers.json'
    with open(saved_file, "r") as fp:
        b = json.load(fp)
    net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)

    print(f'last{topK} sz: {net.model_size():.2f}')

    #
    # # fix the number of layers
    # topK=25
    # n_layers = len(layer_names)
    # permute = np.random.permutation(n_layers)
    # layer_name_permute = [layer_names[p] for p in permute]
    # random_layer_names = layer_name_permute[:topK]
    #
    # saved_file = f'./saved/seed{seed}_random_select_{topK}Layers.json'
    #
    # with open(saved_file, "w") as fp:
    #     json.dump(random_layer_names, fp)
    #
    # #--------------------------------------------------------------------------------------------------------------------------
    # # fix the size of models
    # branch_net = MnistResNet_fw(branched='branch', topK=25)
    # branch_net_sz = branch_net.model_size(unit='B')
    # sz = net.model_size(unit='B')
    # sz_bound = branch_net_sz - sz
    # param_sz = [parameter_sz(layer_dict[name]) for name in layer_name_permute]
    #
    # p_sz_total = 0
    # n_select_layers = 0
    # for i, p_sz in enumerate(param_sz):
    #     p_sz_total += p_sz
    #     if p_sz_total > sz_bound:
    #         n_select_layers = i + 1
    #         break
    #
    # random_layer_names = layer_name_permute[:n_select_layers]
    # saved_file = f'./saved/seed{seed}_random_select_comparable_size.json'
    #
    # with open(saved_file, "w") as fp:
    #     json.dump(random_layer_names, fp)

    # saved_file = f'./saved/seed{0}_random_select_comparable_size.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')
    #
    # saved_file = f'./saved/seed{1}_random_select_comparable_size.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')
    #
    #
    # saved_file = f'./saved/seed{2}_random_select_comparable_size.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')
    #
    # saved_file = f'./saved/seed{0}_random_select_25Layers.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')
    #
    # saved_file = f'./saved/seed{1}_random_select_25Layers.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')
    #
    # saved_file = f'./saved/seed{2}_random_select_25Layers.json'
    # with open(saved_file, "r") as fp:
    #     b = json.load(fp)
    # net = MnistResNet_fw(branched='ablation', ablation_file=saved_file)
    # print(f'saved_file sz: {net.model_size():.2f}')

    print('finished!')

