
import tensorflow as tf
import numpy as np
import scipy.sparse as sp


# pullaway_loss

def pullaway_loss(embeddings):
    """
    Pull Away loss calculation
    :param embeddings: The embeddings to be orthogonalized for varied faces. Shape [batch_size, embeddings_dim]
    :return: pull away term loss
    """
    norm = tf.sqrt(tf.reduce_sum(tf.square(embeddings), 1, keep_dims=True))
    normalized_embeddings = embeddings / norm
    similarity = tf.matmul(
        normalized_embeddings, normalized_embeddings, transpose_b=True)
    batch_size = tf.cast(tf.shape(embeddings)[0], tf.float32)
    pt_loss = (tf.reduce_sum(similarity) - batch_size) / (batch_size * (batch_size - 1))
    return pt_loss

### initial function
def kaiming_normal(shape):
    if len(shape) == 2:
        fan_in, fan_out = shape[0], shape[1]
    elif len(shape) == 4:
        fan_in, fan_out = np.prod(shape[:3]), shape[3]
    return tf.random_normal(shape) * np.sqrt(2.0 / fan_in) 



### leaky_relu activation

def leaky_relu(x, alpha=0.01):
    """Compute the leaky ReLU activation function.
    
    Inputs:
    - x: TensorFlow Tensor with arbitrary shape
    - alpha: leak parameter for leaky ReLU
    
    Returns:
    TensorFlow Tensor with the same shape as x
    """
    # TODO: implement leaky ReLU
    return tf.maximum(x,0) - tf.maximum(-alpha * x, 0) #


### GUMBEL refer https://github.com/vithursant/VAE-Gumbel-Softmax

def sample_gumbel(shape, eps=1e-20):
    U = tf.random_uniform(shape, minval=0, maxval=1)
    return -tf.log(-tf.log(U + eps) + eps)

def gumbel_softmax(logits, temperature, hard=False):
    gumbel_softmax_sample = logits + sample_gumbel(tf.shape(logits))
    y = tf.nn.softmax(gumbel_softmax_sample / temperature)

    if hard:
        k = tf.shape(logits)[-1]
        y_hard = tf.cast(tf.equal(y, tf.reduce_max(y, 1, keep_dims=True)),
                         y.dtype)
        y = tf.stop_gradient(y_hard - y) + y

    return y


"""
Combine fake nodes into original graph
"""

def combine_nodes(adj, features, y_train, y_val, y_test, train_mask, val_mask, test_mask, \
                  generated_adj, generated_features):
    '''
        injecting nodes to adj, features, and assign labels to the injected nodes
    '''
    enlarged_features = enlarged_features = sp.vstack((features, sp.coo_matrix(generated_features)))
    B = generated_adj.shape[0]
    part1_adj = sp.vstack((adj, sp.coo_matrix(generated_adj)))
    part2_adj = np.vstack((generated_adj.T, np.eye(B)))
    enlarged_adj = sp.hstack((part1_adj, part2_adj))
    enlarged_y_train =  np.vstack( (y_train, np.zeros((B, y_train.shape[1]))))
    enlarged_y_val =  np.vstack( (y_val, np.zeros((B, y_val.shape[1]))))
    enlarged_y_test =  np.vstack( (y_test, np.zeros((B, y_test.shape[1]))))

    enlarged_train_mask = np.hstack( (train_mask,np.array([False] * B) ))
    enlarged_val_mask = np.hstack( (val_mask,np.array([False] * B) ))
    enlarged_test_mask = np.hstack( (test_mask,np.array([False] * B) ))
    return enlarged_adj, enlarged_features, enlarged_y_train, enlarged_y_val, enlarged_y_test, enlarged_train_mask, enlarged_val_mask, enlarged_test_mask
    
"""
Generate random uniform noise from -1 to 1.
"""

def sample_noise(batch_size, dim):
    """
    
    Inputs:
    - batch_size: integer giving the batch size of noise to generate
    - dim: integer giving the dimension of the the noise to generate
    
    Returns:
    TensorFlow Tensor containing uniform noise in [-1, 1] with shape [batch_size, dim]
    """
    random_noise = tf.random_uniform([batch_size, dim], maxval= 1, minval= -1)
    return random_noise
    

"""
generate feature for fake nodes
"""
def generator_covariate(z, d, hidden_layer = 256 , name = "generator_cov"):
    with tf.variable_scope(name):
        initializer = tf.contrib.layers.xavier_initializer(uniform=True)
        fc1_output = tf.layers.dense(z, hidden_layer, activation = tf.nn.relu,
                                 kernel_initializer=initializer)
        sy_covariates = tf.layers.dense(fc1_output, d, activation = tf.nn.relu, ## tf.nn.relu can make the ouput sparse
                                 kernel_initializer=initializer)
        return sy_covariates

"""
generate link relation for fake nodes
"""
def generator_link(z, n, hidden_layer = 256, name = "generator_link"):
    with tf.variable_scope(name, reuse=tf.AUTO_REUSE):
        initializer = tf.contrib.layers.xavier_initializer(uniform=True)
        fc1_output = tf.layers.dense(z, hidden_layer, activation = tf.nn.relu,
                                 kernel_initializer=initializer)
        sy_logits_na = tf.layers.dense(fc1_output, n, activation = tf.nn.relu6, 
                                 kernel_initializer = initializer) /6
        return sy_logits_na



