import os
import numpy as np
import pandas as pd
import torch
from torch.utils.data import Dataset, DataLoader
from sklearn.preprocessing import StandardScaler
from utils.timefeatures import time_features
import warnings

warnings.filterwarnings('ignore')


class Dataset_ETT_hour(Dataset):
    def __init__(self, root_path, flag='train', size=None,
                 features='S', data_path='ETTh1.csv',
                 target='OT', scale=True, timeenc=0, freq='h', seasonal_patterns=None):
        # size [seq_len, label_len, pred_len]
        # info
        if size == None:
            self.seq_len = 24 * 4 * 4
            self.label_len = 24 * 4
            self.pred_len = 24 * 4
        else:
            self.seq_len = size[0]
            self.label_len = size[1]
            self.pred_len = size[2]
        # init
        assert flag in ['train', 'test', 'val']
        type_map = {'train': 0, 'val': 1, 'test': 2}
        self.set_type = type_map[flag]

        self.features = features
        self.target = target
        self.scale = scale
        self.timeenc = timeenc
        self.freq = freq

        self.root_path = root_path
        self.data_path = data_path
        self.__read_data__()

    def __read_data__(self):
        self.scaler = StandardScaler()
        df_raw = pd.read_csv(os.path.join(self.root_path,
                                          self.data_path))

        border1s = [0, 12 * 30 * 24 - self.seq_len, 12 * 30 * 24 + 4 * 30 * 24 - self.seq_len]
        border2s = [12 * 30 * 24, 12 * 30 * 24 + 4 * 30 * 24, 12 * 30 * 24 + 8 * 30 * 24]
        border1 = border1s[self.set_type]
        border2 = border2s[self.set_type]

        if self.features == 'M' or self.features == 'MS':
            cols_data = df_raw.columns[1:]
            df_data = df_raw[cols_data]
        elif self.features == 'S':
            df_data = df_raw[[self.target]]

        if self.scale:
            train_data = df_data[border1s[0]:border2s[0]]
            self.scaler.fit(train_data.values)
            data = self.scaler.transform(df_data.values)
        else:
            data = df_data.values

        df_stamp = df_raw[['date']][border1:border2]
        df_stamp['date'] = pd.to_datetime(df_stamp.date)
        if self.timeenc == 0:
            df_stamp['month'] = df_stamp.date.apply(lambda row: row.month, 1)
            df_stamp['day'] = df_stamp.date.apply(lambda row: row.day, 1)
            df_stamp['weekday'] = df_stamp.date.apply(lambda row: row.weekday(), 1)
            df_stamp['hour'] = df_stamp.date.apply(lambda row: row.hour, 1)
            data_stamp = df_stamp.drop(['date'], 1).values
        elif self.timeenc == 1:
            dt = pd.to_datetime(df_stamp['date'].values)
            # year, month, weekday, day, hour, minute
            year = dt.year.to_numpy(dtype=np.float64)
            month = dt.month.to_numpy(dtype=np.float64)
            weekday = dt.weekday.to_numpy(dtype=np.float64)
            day = dt.day.to_numpy(dtype=np.float64)
            hour = dt.hour.to_numpy(dtype=np.float64)
            minute = dt.minute.to_numpy(dtype=np.float64)
            time_stamp = [year, month, weekday, day, hour, minute]
            data_stamp_abs = np.stack(time_stamp, axis=-1)

            stamp_list, n = [], 1
            # numerical hour_of_day
            steps_per_weak = 7
            steps_per_day, T = 24, len(df_raw)
            tod = [i % steps_per_day / steps_per_day for i in range(T)]
            tod_tiled = np.tile(tod, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(tod_tiled)
            # numerical hour_of_week
            hour_of_week = [(i // steps_per_day) % steps_per_weak / steps_per_weak for i in range(T)]
            hour_of_week_tiled = np.tile(hour_of_week, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(hour_of_week_tiled)

            data_stamp_rlt = np.concatenate(stamp_list, axis=1).squeeze(-1)
            data_stamp_rlt = data_stamp_rlt[border1:border2]
            data_stamp = np.concatenate([data_stamp_abs, data_stamp_rlt], axis=-1)

        self.data_x = data[border1:border2]
        self.data_y = data[border1:border2]
        self.data_stamp = data_stamp

    def __getitem__(self, index):
        s_begin = index
        s_end = s_begin + self.seq_len
        r_begin = s_end - self.label_len
        r_end = r_begin + self.label_len + self.pred_len

        seq_x = self.data_x[s_begin:s_end]
        seq_y = self.data_y[r_begin:r_end]
        seq_x_mark = self.data_stamp[s_begin:s_end]
        seq_y_mark = self.data_stamp[r_begin:r_end]

        return seq_x, seq_y, seq_x_mark, seq_y_mark

    def __len__(self):
        return len(self.data_x) - self.seq_len - self.pred_len + 1

    def inverse_transform(self, data):
        return self.scaler.inverse_transform(data)


class Dataset_ETT_minute(Dataset):
    def __init__(self, root_path, flag='train', size=None,
                 features='S', data_path='ETTm1.csv',
                 target='OT', scale=True, timeenc=0, freq='t', seasonal_patterns=None):
        # size [seq_len, label_len, pred_len]
        # info
        if size == None:
            self.seq_len = 24 * 4 * 4
            self.label_len = 24 * 4
            self.pred_len = 24 * 4
        else:
            self.seq_len = size[0]
            self.label_len = size[1]
            self.pred_len = size[2]
        # init
        assert flag in ['train', 'test', 'val']
        type_map = {'train': 0, 'val': 1, 'test': 2}
        self.set_type = type_map[flag]

        self.features = features
        self.target = target
        self.scale = scale
        self.timeenc = timeenc
        self.freq = freq

        self.root_path = root_path
        self.data_path = data_path
        self.__read_data__()

    def __read_data__(self):
        self.scaler = StandardScaler()
        df_raw = pd.read_csv(os.path.join(self.root_path,
                                          self.data_path))

        border1s = [0, 12 * 30 * 24 * 4 - self.seq_len, 12 * 30 * 24 * 4 + 4 * 30 * 24 * 4 - self.seq_len]
        border2s = [12 * 30 * 24 * 4, 12 * 30 * 24 * 4 + 4 * 30 * 24 * 4, 12 * 30 * 24 * 4 + 8 * 30 * 24 * 4]
        border1 = border1s[self.set_type]
        border2 = border2s[self.set_type]

        if self.features == 'M' or self.features == 'MS':
            cols_data = df_raw.columns[1:]
            df_data = df_raw[cols_data]
        elif self.features == 'S':
            df_data = df_raw[[self.target]]

        if self.scale:
            train_data = df_data[border1s[0]:border2s[0]]
            self.scaler.fit(train_data.values)
            data = self.scaler.transform(df_data.values)
        else:
            data = df_data.values

        df_stamp = df_raw[['date']][border1:border2]
        df_stamp['date'] = pd.to_datetime(df_stamp.date)
        if self.timeenc == 0:
            df_stamp['month'] = df_stamp.date.apply(lambda row: row.month, 1)
            df_stamp['day'] = df_stamp.date.apply(lambda row: row.day, 1)
            df_stamp['weekday'] = df_stamp.date.apply(lambda row: row.weekday(), 1)
            df_stamp['hour'] = df_stamp.date.apply(lambda row: row.hour, 1)
            df_stamp['minute'] = df_stamp.date.apply(lambda row: row.minute, 1)
            df_stamp['minute'] = df_stamp.minute.map(lambda x: x // 15)
            data_stamp = df_stamp.drop(['date'], 1).values
        elif self.timeenc == 1:
            # data_stamp = time_features(pd.to_datetime(df_stamp['date'].values), freq=self.freq)
            # data_stamp = data_stamp.transpose(1, 0)

            dt = pd.to_datetime(df_stamp['date'].values)
            # year, month, weekday, day, hour, minute
            year = dt.year.to_numpy(dtype=np.float64)
            month = dt.month.to_numpy(dtype=np.float64)
            weekday = dt.weekday.to_numpy(dtype=np.float64)
            day = dt.day.to_numpy(dtype=np.float64)
            hour = dt.hour.to_numpy(dtype=np.float64)
            minute = dt.minute.to_numpy(dtype=np.float64) / 15.
            time_stamp = [year, month, weekday, day, hour, minute]
            data_stamp_abs = np.stack(time_stamp, axis=-1)

            stamp_list, n = [], 1
            # numerical hour_of_day
            steps_per_weak = 7
            steps_per_day, T = 96, len(df_raw)
            tod = [i % steps_per_day / steps_per_day for i in range(T)]
            tod_tiled = np.tile(tod, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(tod_tiled)
            # numerical hour_of_week
            hour_of_week = [(i // steps_per_day) % steps_per_weak / steps_per_weak for i in range(T)]
            hour_of_week_tiled = np.tile(hour_of_week, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(hour_of_week_tiled)

            data_stamp_rlt = np.concatenate(stamp_list, axis=1).squeeze(-1)
            data_stamp_rlt = data_stamp_rlt[border1:border2]
            data_stamp = np.concatenate([data_stamp_abs, data_stamp_rlt], axis=-1)

        self.data_x = data[border1:border2]
        self.data_y = data[border1:border2]
        self.data_stamp = data_stamp

    def __getitem__(self, index):
        s_begin = index
        s_end = s_begin + self.seq_len
        r_begin = s_end - self.label_len
        r_end = r_begin + self.label_len + self.pred_len

        seq_x = self.data_x[s_begin:s_end]
        seq_y = self.data_y[r_begin:r_end]
        seq_x_mark = self.data_stamp[s_begin:s_end]
        seq_y_mark = self.data_stamp[r_begin:r_end]

        return seq_x, seq_y, seq_x_mark, seq_y_mark

    def __len__(self):
        return len(self.data_x) - self.seq_len - self.pred_len + 1

    def inverse_transform(self, data):
        return self.scaler.inverse_transform(data)


class Dataset_Custom(Dataset):
    def __init__(self, root_path, flag='train', size=None,
                 features='S', data_path='ETTh1.csv',
                 target='OT', scale=True, timeenc=0, freq='h', seasonal_patterns=None):
        # size [seq_len, label_len, pred_len]
        # info
        if size == None:
            self.seq_len = 24 * 4 * 4
            self.label_len = 24 * 4
            self.pred_len = 24 * 4
        else:
            self.seq_len = size[0]
            self.label_len = size[1]
            self.pred_len = size[2]
        # init
        assert flag in ['train', 'test', 'val']
        type_map = {'train': 0, 'val': 1, 'test': 2}
        self.set_type = type_map[flag]

        if 'weather' in data_path:
            self.steps_per_day = 144
        else:
            self.steps_per_day = 24
        self.features = features
        self.target = target
        self.scale = scale
        self.timeenc = timeenc
        self.freq = freq

        self.root_path = root_path
        self.data_path = data_path
        self.__read_data__()


    def __read_data__(self):
        self.scaler = StandardScaler()
        df_raw = pd.read_csv(os.path.join(self.root_path,
                                          self.data_path))

        '''
        df_raw.columns: ['date', ...(other features), target feature]
        '''
        cols = list(df_raw.columns)
        cols.remove(self.target)
        cols.remove('date')
        df_raw = df_raw[['date'] + cols + [self.target]]
        num_train = int(len(df_raw) * 0.7)
        num_test = int(len(df_raw) * 0.2)
        num_vali = len(df_raw) - num_train - num_test
        border1s = [0, num_train - self.seq_len, len(df_raw) - num_test - self.seq_len]
        border2s = [num_train, num_train + num_vali, len(df_raw)]
        border1 = border1s[self.set_type]
        border2 = border2s[self.set_type]

        if self.features == 'M' or self.features == 'MS':
            cols_data = df_raw.columns[1:]
            df_data = df_raw[cols_data]
        elif self.features == 'S':
            df_data = df_raw[[self.target]]

        if self.scale:
            train_data = df_data[border1s[0]:border2s[0]]
            self.scaler.fit(train_data.values)
            data = self.scaler.transform(df_data.values)
        else:
            data = df_data.values

        df_stamp = df_raw[['date']][border1:border2]
        df_stamp['date'] = pd.to_datetime(df_stamp.date)
        if self.timeenc == 0:
            df_stamp['month'] = df_stamp.date.apply(lambda row: row.month, 1)
            df_stamp['day'] = df_stamp.date.apply(lambda row: row.day, 1)
            df_stamp['weekday'] = df_stamp.date.apply(lambda row: row.weekday(), 1)
            df_stamp['hour'] = df_stamp.date.apply(lambda row: row.hour, 1)
            data_stamp = df_stamp.drop(['date'], 1).values
        elif self.timeenc == 1:
            # data_stamp = time_features(pd.to_datetime(df_stamp['date'].values), freq=self.freq)
            # data_stamp = data_stamp.transpose(1, 0)

            dt = pd.to_datetime(df_stamp['date'].values)
            # year, month, weekday, day, hour, minute
            year = dt.year.to_numpy(dtype=np.float64)
            month = dt.month.to_numpy(dtype=np.float64)
            weekday = dt.weekday.to_numpy(dtype=np.float64)
            day = dt.day.to_numpy(dtype=np.float64)
            hour = dt.hour.to_numpy(dtype=np.float64)
            minute = dt.minute.to_numpy(dtype=np.float64) / 10.
            time_stamp = [year, month, weekday, day, hour, minute]
            data_stamp_abs = np.stack(time_stamp, axis=-1)

            stamp_list, n = [], 1
            # numerical hour_of_day
            steps_per_weak = 7
            steps_per_day, T = self.steps_per_day, len(df_raw)
            tod = [i % steps_per_day / steps_per_day for i in range(T)]
            tod_tiled = np.tile(tod, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(tod_tiled)
            # numerical hour_of_week
            hour_of_week = [(i // steps_per_day) % steps_per_weak / steps_per_weak for i in range(T)]
            hour_of_week_tiled = np.tile(hour_of_week, [1, n, 1]).transpose((2, 1, 0))
            stamp_list.append(hour_of_week_tiled)

            data_stamp_rlt = np.concatenate(stamp_list, axis=1).squeeze(-1)
            data_stamp_rlt = data_stamp_rlt[border1:border2]
            data_stamp = np.concatenate([data_stamp_abs, data_stamp_rlt], axis=-1)

        self.data_x = data[border1:border2]
        self.data_y = data[border1:border2]
        self.data_stamp = data_stamp

    def __getitem__(self, index):
        s_begin = index
        s_end = s_begin + self.seq_len
        r_begin = s_end - self.label_len
        r_end = r_begin + self.label_len + self.pred_len

        seq_x = self.data_x[s_begin:s_end]
        seq_y = self.data_y[r_begin:r_end]

        seq_x_mark = self.data_stamp[s_begin:s_end]
        seq_y_mark = self.data_stamp[r_begin:r_end]

        return seq_x, seq_y, seq_x_mark, seq_y_mark

    def __len__(self):
        return len(self.data_x) - self.seq_len - self.pred_len + 1

    def inverse_transform(self, data):
        return self.scaler.inverse_transform(data)


class Dataset_PEMS(Dataset):
    def __init__(self, root_path, flag='train', size=None,
                 features='S', data_path='ETTh1.csv', steps_per_day=288, steps_per_week=7,
                 target='OT', scale=True, timeenc=0, freq='h', seasonal_patterns=None):
        # size [seq_len, label_len, pred_len]
        # info
        self.seq_len = size[0]
        self.label_len = size[1]
        self.pred_len = size[2]
        # init
        assert flag in ['train', 'test', 'val']
        type_map = {'train': 0, 'val': 1, 'test': 2}
        self.set_type = type_map[flag]

        self.steps_per_day = steps_per_day
        self.steps_per_week = steps_per_week
        self.features = features
        self.target = target
        self.scale = scale
        self.timeenc = timeenc
        self.freq = freq

        self.root_path = root_path
        self.data_path = data_path
        self.__read_data__()

    def __read_data__(self):
        self.scaler = StandardScaler()
        data_file = os.path.join(self.root_path, self.data_path)
        data = np.load(data_file, allow_pickle=True)
        data = data['data'][:, :, 0]

        stamp_list, n = [], 1
        # numerical hour_of_day
        steps_per_weak = self.steps_per_week  # 7
        steps_per_day, T = self.steps_per_day, len(data)  # 24
        tod = [i % steps_per_day / steps_per_day for i in range(T)]
        tod_tiled = np.tile(tod, [1, n, 1]).transpose((2, 1, 0))
        stamp_list.append(tod_tiled)
        # numerical hour_of_week
        hour_of_week = [(i // steps_per_day) % steps_per_weak / steps_per_weak for i in range(T)]
        hour_of_week_tiled = np.tile(hour_of_week, [1, n, 1]).transpose((2, 1, 0))
        stamp_list.append(hour_of_week_tiled)

        stamp = np.concatenate(stamp_list, axis=1).squeeze(-1)

        train_ratio = 0.6
        valid_ratio = 0.2
        train_data = data[:int(train_ratio * len(data))]
        valid_data = data[int(train_ratio * len(data)): int((train_ratio + valid_ratio) * len(data))]
        test_data = data[int((train_ratio + valid_ratio) * len(data)):]
        total_data = [train_data, valid_data, test_data]

        train_stamp = stamp[:int(train_ratio * len(data))]
        valid_stamp = stamp[int(train_ratio * len(data)): int((train_ratio + valid_ratio) * len(data))]
        test_stamp = stamp[int((train_ratio + valid_ratio) * len(data)):]
        total_stamp = [train_stamp, valid_stamp, test_stamp]

        data = total_data[self.set_type]
        self.stamp = total_stamp[self.set_type]

        if self.scale:
            self.scaler.fit(train_data)
            data = self.scaler.transform(data)

        df = pd.DataFrame(data)
        df = df.fillna(method='ffill', limit=len(df)).fillna(method='bfill', limit=len(df)).values

        self.data_x = df
        self.data_y = df

    def __getitem__(self, index):
        s_begin = index
        s_end = s_begin + self.seq_len
        r_begin = s_end - self.label_len
        r_end = r_begin + self.label_len + self.pred_len

        seq_x = self.data_x[s_begin:s_end]
        seq_y = self.data_y[r_begin:r_end]

        seq_x_mark = self.stamp[s_begin:s_end]
        seq_y_mark = self.stamp[r_begin:r_end]

        return seq_x, seq_y, seq_x_mark, seq_y_mark

    def __len__(self):
        return len(self.data_x) - self.seq_len - self.pred_len + 1

    def inverse_transform(self, data):
        return self.scaler.inverse_transform(data)


class Dataset_Solar(Dataset):
    def __init__(self, root_path, flag='train', size=None,
                 features='S', data_path='ETTh1.csv', steps_per_day=144, steps_per_week=7,
                 target='OT', scale=True, timeenc=0, freq='h', seasonal_patterns=None):
        # size [seq_len, label_len, pred_len]
        # info
        self.seq_len = size[0]
        self.label_len = size[1]
        self.pred_len = size[2]
        # init
        assert flag in ['train', 'test', 'val']
        type_map = {'train': 0, 'val': 1, 'test': 2}
        self.set_type = type_map[flag]

        self.steps_per_day = steps_per_day
        self.steps_per_week = steps_per_week
        self.features = features
        self.target = target
        self.scale = scale
        self.timeenc = timeenc
        self.freq = freq

        self.root_path = root_path
        self.data_path = data_path
        self.__read_data__()

    def __read_data__(self):
        self.scaler = StandardScaler()
        df_raw = []
        with open(os.path.join(self.root_path, self.data_path), "r", encoding='utf-8') as f:
            for line in f.readlines():
                line = line.strip('\n').split(',')
                data_line = np.stack([float(i) for i in line])
                df_raw.append(data_line)
        df_raw = np.stack(df_raw, 0)
        df_raw = pd.DataFrame(df_raw)

        stamp_list, n = [], 1
        # numerical hour_of_day
        steps_per_weak = self.steps_per_week
        steps_per_day, T = self.steps_per_day, len(df_raw)
        tod = [i % steps_per_day / steps_per_day for i in range(T)]
        tod_tiled = np.tile(tod, [1, n, 1]).transpose((2, 1, 0))
        stamp_list.append(tod_tiled)
        # numerical hour_of_week
        hour_of_week = [(i // steps_per_day) % steps_per_weak / steps_per_weak for i in range(T)]
        hour_of_week_tiled = np.tile(hour_of_week, [1, n, 1]).transpose((2, 1, 0))
        stamp_list.append(hour_of_week_tiled)

        stamp = np.concatenate(stamp_list, axis=1).squeeze(-1)

        num_train = int(len(df_raw) * 0.7)
        num_test = int(len(df_raw) * 0.2)
        num_valid = int(len(df_raw) * 0.1)
        border1s = [0, num_train - self.seq_len, len(df_raw) - num_test - self.seq_len]
        border2s = [num_train, num_train + num_valid, len(df_raw)]
        border1 = border1s[self.set_type]
        border2 = border2s[self.set_type]

        df_data = df_raw.values

        if self.scale:
            train_data = df_data[border1s[0]:border2s[0]]
            self.scaler.fit(train_data)
            data = self.scaler.transform(df_data)
        else:
            data = df_data

        self.data_x = data[border1:border2]
        self.data_y = data[border1:border2]

        self.stamp = stamp[border1:border2]

    def __getitem__(self, index):
        s_begin = index
        s_end = s_begin + self.seq_len
        r_begin = s_end - self.label_len
        r_end = r_begin + self.label_len + self.pred_len

        seq_x = self.data_x[s_begin:s_end]
        seq_y = self.data_y[r_begin:r_end]
        seq_x_mark = self.stamp[s_begin:s_end]
        seq_y_mark = self.stamp[r_begin:r_end]

        return seq_x, seq_y, seq_x_mark, seq_y_mark

    def __len__(self):
        return len(self.data_x) - self.seq_len - self.pred_len + 1

    def inverse_transform(self, data):
        return self.scaler.inverse_transform(data)
