import json
import os
import matplotlib.pyplot as plt
import seaborn as sns
from datetime import datetime
from typing import Dict, List
import numpy as np


class AutomatedVisualizationEngine:
    
    def __init__(self, output_dir: str = "output/visualizations"):
        self.output_dir = output_dir
        os.makedirs(output_dir, exist_ok=True)
        
        plt.style.use('default')
        sns.set_palette("husl")
    
    # Generate metric distribution visualization
    def generate_metric_distribution(self, results: Dict) -> str:
        try:
            fig, ax = plt.subplots(figsize=(10, 6))
            
            metrics_data = results.get('aggregate_metrics', {}).get('batch_statistics', {}).get('per_metric', {})
            
            if metrics_data:
                metrics = list(metrics_data.keys())
                scores = [metrics_data[m].get('mean', 0) for m in metrics]
                
                bars = ax.bar(metrics, scores)
                ax.set_title('Metric Score Distribution')
                ax.set_ylabel('Score')
                ax.set_ylim(0, 1)
                
                for bar, score in zip(bars, scores):
                    ax.text(bar.get_x() + bar.get_width()/2, bar.get_height() + 0.01, 
                           f'{score:.3f}', ha='center', va='bottom')
            
            plt.xticks(rotation=45)
            plt.tight_layout()
            
            filename = f"metric_distribution_{datetime.now().strftime('%Y%m%d_%H%M%S')}.png"
            filepath = os.path.join(self.output_dir, filename)
            plt.savefig(filepath, dpi=300, bbox_inches='tight')
            plt.close()
            
            return filepath
            
        except Exception as e:
            print(f"Error generating metric distribution: {e}")
            return ""
    
    # Generate HTML dashboard
    def generate_dashboard_html(self, results: Dict) -> str:
        html_content = f"""
<!DOCTYPE html>
<html>
<head>
    <title>MIMIC-Eye Evaluation Dashboard</title>
    <style>
        body {{ font-family: Arial, sans-serif; margin: 20px; }}
        .metric {{ background: #f0f0f0; padding: 10px; margin: 10px 0; border-radius: 5px; }}
        .score {{ font-size: 24px; font-weight: bold; color: #2E8B57; }}
    </style>
</head>
<body>
    <h1>MIMIC-Eye Evaluation Dashboard</h1>
    <p>Generated: {datetime.now().isoformat()}</p>
    
    <div class="metric">
        <h3>Overall Performance</h3>
        <div class="score">{results.get('aggregate_metrics', {}).get('batch_overall_score', 0):.4f}</div>
    </div>
    
    <div class="metric">
        <h3>Processing Summary</h3>
        <p>Total Evaluations: {results.get('batch_summary', {}).get('total_evaluated', 0)}</p>
        <p>Success Rate: {results.get('batch_summary', {}).get('success_rate', 0)*100:.1f}%</p>
        <p>Processing Time: {results.get('batch_summary', {}).get('processing_time', 0):.2f}s</p>
    </div>
</body>
</html>
"""
        
        filename = f"dashboard_{datetime.now().strftime('%Y%m%d_%H%M%S')}.html"
        filepath = os.path.join(self.output_dir, filename)
        with open(filepath, 'w', encoding='utf-8') as f:
            f.write(html_content)
        
        return filepath

    # Generate all visualizations for automation results
    def generate_all_visualizations(self, results: Dict) -> Dict:
        metric_chart = self.generate_metric_distribution(results)
        dashboard = self.generate_dashboard_html(results)
        
        return {
            "metric_distribution": metric_chart,
            "dashboard": dashboard
        }


# Create visualization charts and HTML dashboard from evaluation results
def generate_visualizations(results: Dict) -> Dict:
    engine = AutomatedVisualizationEngine()
    return engine.generate_all_visualizations(results)