import numpy as np
from numba import jit


class RegressionFunc:

    def __init__(self, reg_fact, normalize, unnormalize_output, bias_entry=None):
        self._reg_fact = reg_fact
        self._normalize = normalize
        self._unnormalize_output = unnormalize_output
        self._bias_entry = bias_entry
        self._params = None
        self.o_std = None
        self._normalized_features = None
        self._normalized_outputs = None

    def __call__(self, inputs):
        if self._params is None:
            raise AssertionError("Model not trained yet")
        return self._feature_fn(inputs) @ self._params

    def _feature_fn(self, x):
        raise NotImplementedError

    def _normalize_features(self, features):
        mean = np.mean(features, axis=0, keepdims=True)
        std = np.std(features, axis=0, keepdims=True)
        # do not normalize bias
        if self._bias_entry is not None:
            mean[:, self._bias_entry] = 0.0
            std[:, self._bias_entry] = 1.0
            # mean[self._bias_entry] = 0.0        #orig, but if multidim array shape: (x,y) -> turns whole array to 0
            # std[self._bias_entry] = 1.0         #orig, but if multidim array shape: (x,y) -> turns whole array to 1
        features = (features - mean) / (std + 1e-20)
        return features, np.squeeze(mean, axis=0), np.squeeze(std, axis=0)

    def _normalize_outputs(self, outputs):
        mean = np.mean(outputs, axis=0, keepdims=True)
        # std = np.std(outputs)
        outputs_cov = np.cov(outputs.T, bias=True)
        if len(outputs_cov.shape) == 0:
            std = np.sqrt(outputs_cov)
        else:
            std = np.sqrt(np.diag(outputs_cov))
        outputs = (outputs - mean) / (std + 1e-20)
        return outputs, mean, std

    def _undo_normalization(self, params, f_mean, f_std, o_mean, o_std):
        if self._unnormalize_output:
            if len(params.shape) == 2:
                if len(f_mean.shape) == 1:
                    f_mean = f_mean.reshape((-1, 1))
                if len(o_std.shape) == 0:
                    o_std = o_std.reshape(1)
                if len(params.shape) == 1:
                    params = params.reshape((-1, 1))
                # params = np.dot(np.dot(np.diag(f_std), params), np.linalg.inv(np.diag(o_std)))
                tmp = params[self._bias_entry].copy()
                params = np.dot(np.dot(np.diag(1 / f_std).T, params), np.diag(o_std))
                params[self._bias_entry] = tmp.reshape((1, -1))
                params[self._bias_entry] = np.dot(params[self._bias_entry], np.diag(o_std)) + o_mean - np.dot(f_mean[:-1].T,
                                                                                                          params[:-1, :])

        # if self._unnormalize_output:
        #     tmp = params[self._bias_entry].copy()
        #     params = params*(1/f_std)*o_std
        #     params[self._bias_entry] = tmp*o_std + o_mean - f_mean[:-1].T@params[:-1]
        # if self._unnormalize_output:
            else:
                params *= (o_std / f_std)               # orig
                params[self._bias_entry] = params[self._bias_entry] - np.dot(params, f_mean) + o_mean         # orig
        else:
            params *= (1.0 / f_std)
            params[self._bias_entry] = params[self._bias_entry] - np.dot(params, f_mean)                  # orig
            # params[self._bias_entry] = params[self._bias_entry] - np.dot(params.T, f_mean)
        return params

    def fit(self, inputs, outputs, weights=None):
        if len(outputs.shape) > 1:
            outputs = np.squeeze(outputs)
        features = self._feature_fn(inputs)
        if self._normalize:
            features, f_mean, f_std = self._normalize_features(features)
            outputs, o_mean, o_std = self._normalize_outputs(outputs)
            self._normalized_features = features
            self._normalized_outputs = outputs
        if weights is not None:
            if len(weights.shape) == 1:
                weights = np.expand_dims(weights, 1)
            weighted_features = weights * features
            # self._normalized_features = weighted_features
        else:
            weighted_features = features

        # regression
        reg_mat = np.eye(weighted_features.shape[-1]) * self._reg_fact
        if self._bias_entry is not None:
            reg_mat[self._bias_entry, self._bias_entry] = 0.0
        try:
            self._params = np.linalg.solve(weighted_features.T @ features + reg_mat, weighted_features.T @ outputs)
            if self._normalize:
                self._params = self._undo_normalization(self._params, f_mean, f_std, o_mean, o_std)
                self.o_std = o_std
        except np.linalg.LinAlgError as e:
            print("Error during matrix inversion", e.what())


class LinFunc(RegressionFunc):

    def __init__(self, reg_fact, normalize, unnormalize_output):
        super().__init__(reg_fact, normalize, unnormalize_output, -1)

    def _feature_fn(self, x):
        return np.concatenate([x, np.ones([x.shape[0], 1], dtype=x.dtype)], 1)


class QuadFunc(RegressionFunc):
    # *Fits - 0.5 * x ^ T  Rx + x ^ T r + r_0 ** * /

    def __init__(self, reg_fact, normalize, unnormalize_output):
        super().__init__(reg_fact, normalize, unnormalize_output, bias_entry=-1)
        self.quad_term = None
        self.lin_term = None
        self.const_term = None

    @staticmethod
    @jit(nopython=True)
    def _feature_fn(x):
        num_quad_features = int(np.floor(0.5 * (x.shape[-1] + 1) * x.shape[-1]))
        num_features = num_quad_features + x.shape[-1] + 1
        features = np.ones((x.shape[0], num_features))
        write_idx = 0
        # quad features
        for i in range(x.shape[-1]):
            for j in range(x.shape[-1] - i):
                features[:, write_idx] = x[:, i] * x[:, j + i]
                write_idx += 1
        # linear features
        features[:, num_quad_features: -1] = x

        # last coloumn (bias) already 1
        return features

    def fit(self, inputs, outputs, weights=None, sample_mean=None, sample_chol_cov=None):
        if sample_mean is None:
            assert sample_chol_cov is None
        if sample_chol_cov is None:
            assert sample_mean is None

        # whithening
        if sample_mean is not None and sample_chol_cov is not None:
            inv_samples_chol_cov = np.linalg.inv(sample_chol_cov)
            inputs = (inputs - sample_mean) @ inv_samples_chol_cov.T

        dim = inputs.shape[-1]

        super().fit(inputs, outputs, weights)

        idx = np.triu(np.ones([dim, dim], np.bool))

        qt = np.zeros([dim, dim])
        qt[idx] = self._params[:- (dim + 1)]
        self.quad_term = - qt - qt.T

        self.lin_term = self._params[-(dim + 1): -1]
        self.const_term = self._params[-1]

        # unwhitening:
        if sample_mean is not None and sample_chol_cov is not None:
            self.quad_term = inv_samples_chol_cov.T @ self.quad_term @ inv_samples_chol_cov
            t1 = inv_samples_chol_cov.T @ self.lin_term
            t2 = self.quad_term @ sample_mean
            self.lin_term = t1 + t2
            self.const_term += np.dot(sample_mean, -0.5 * t2 - t1)

    # - 0.5 * x ^ T Rx + x ^ T r + r_0 ** * /
    def predict(self, inputs):
        quad_part = np.diag(inputs@self.quad_term@inputs.T)
        quad_part = -0.5*quad_part
        res = quad_part + inputs@self.lin_term + self.const_term
        return res.flatten()
        # feat = self._feature_fn(inputs)
        # return feat@self._params

class QuadFuncJoint(QuadFunc):
    # Fits function of the form R(x, y) = -0.5 * x^T R_xx x + x^T R_xy y - 0.5 * y^T R_yy y + l_x^T x + l_y^T y + c

    def __init__(self, x_dim, y_dim, reg_fact, normalize, unnormalize_output):
        super().__init__(reg_fact, normalize, unnormalize_output)
        self._x_dim = x_dim
        self._y_dim = y_dim
        self.quad_term_xx = None
        self.quad_term_yx = None
        self.quad_term_yy = None
        self.lin_term_x = None
        self.lin_term_y = None

    def fit(self, inputs, outputs, weights=None, sample_mean=None, sample_chol_cov=None):
        if isinstance(inputs, tuple) or isinstance(inputs, list):
            inputs = np.concatenate(inputs, axis=-1)
        super().fit(inputs, outputs, weights, sample_mean, sample_chol_cov)
        self.quad_term_xx = self.quad_term[:self._x_dim, :self._x_dim]
        self.quad_term_yx = - self.quad_term[self._x_dim:, :self._x_dim]
        self.quad_term_yy = self.quad_term[self._x_dim:, self._x_dim:]
        self.lin_term_x = self.lin_term[:self._x_dim]
        self.lin_term_y = self.lin_term[self._x_dim:]

    def predict(self, ctxts, samples):
        r_aa = self.quad_term_yy
        r_cc = self.quad_term_xx
        r_ac = self.quad_term_yx
        r_a = self.lin_term_y
        r_c = self.lin_term_x

        pred = -0.5*np.diag(samples @r_aa @samples.T) - 0.5*np.diag(ctxts@r_cc@ctxts.T) + np.diag(samples @r_ac @ ctxts.T) + samples@r_a + ctxts@r_c + self.const_term
        # pred = np.diag(samples @r_aa @samples.T) +np.diag(ctxts@r_cc@ctxts.T) + np.diag(samples @r_ac @ ctxts.T) + samples@r_a + ctxts@r_c + self.const_term
        return pred

    def predict_features(self, inputs):
        feat = self._feature_fn(inputs)
        return feat@self._params



if __name__ == "__main__":

    # Fits function of the form R(x, y) = -0.5 * x^T R_xx x + x^T R_xy y - 0.5 * y^T R_yy y + l_x^T x + l_y^T y + c

    # quad model of form:
    # x = ctxt
    # y = actions


    # ctxt_dim = 3
    # action_dim = 21
    # A_cc = np.zeros((ctxt_dim, ctxt_dim))
    # A_cc[0, 0] = 2
    # A_cc[0, 1] = 1
    # A_cc[1, 1] = 3
    # A_cc[1, 2] = 2
    # A_cc[2, 2] = 5
    # np.random.seed(0)
    # diag_aa = np.array([1, 3, 5, 2, 1.5, 6, 9, 11, 10, 8, 9, 7, 1, 8, 5.5, 3, 8, 3, 9, 11, 2.5])
    # A_aa = np.diag(diag_aa)
    # for i in range(A_aa.shape[0]-1):
    #     write_idx = i+1
    #     for j in range(A_aa.shape[0]-i-1):
    #         A_aa[i, write_idx+j] = np.random.randint(0, 20)
    #
    # A_ca = np.random.randint(-3, 3, size=(A_cc.shape[0], A_aa.shape[0]))
    # l_c = np.random.randint(-2, 2, size=(A_cc.shape[0]))
    # # l_c = np.zeros(l_c.shape)
    # l_a = np.random.randint(-2, 2, size=(A_aa.shape[0]))
    # b = 2
    #
    # triu_idx_ctxt = np.triu(np.ones([ctxt_dim, ctxt_dim], np.bool))
    # triu_idx_action = np.triu(np.ones([action_dim, action_dim], np.bool))
    #
    # # A_cc += A_cc.T
    # A_cc = np.zeros(A_cc.shape)
    #
    # A_aa += A_aa.T
    # n_samples = 1000
    # # contexts = np.random.uniform(-2, 2, size=(n_samples, A_cc.shape[0]))
    # # contexts_test = np.random.uniform(-2, 2, size=(int(n_samples*0.3), A_cc.shape[0]))
    # contexts = np.random.normal(0, 3, size=(n_samples, A_cc.shape[0]))
    # contexts_test = np.random.normal(0, 3, size=(int(n_samples*0.3), A_cc.shape[0]))
    # # actions = np.random.uniform(-3, 3, size=(n_samples, A_aa.shape[0]))
    # # actions_test = np.random.normal(-3, 3, size=(int(n_samples*0.3), A_aa.shape[0]))
    # actions = np.random.normal(0, 5, size=(n_samples, A_aa.shape[0]))
    # actions_test = np.random.normal(0, 3, size=(int(n_samples*0.3), A_aa.shape[0]))
    #
    # # quad multiplication_
    # # targets = np.sum((contexts@A_cc)*contexts, axis=1)
    # # targets_test = np.diag(contexts@A_cc@contexts.T)
    #
    # # Fits function of the form R(x, y) = -0.5 * x^T R_xx x + x^T R_xy y - 0.5 * y^T R_yy y + l_x^T x + l_y^T y + c
    # def func(contexts, actions):
    #
    #     targets = np.sum((contexts@A_cc)*contexts, axis=1) + np.sum((contexts@A_ca)*actions,axis=1) \
    #           + np.sum((actions@A_aa)*actions, axis=1) + contexts@l_c + actions@l_a + b + np.random.normal(0, 0.5, size=contexts.shape[0])
    #
    #     return targets
    #
    # targets = func(contexts, actions)
    # targets_test = func(contexts_test, actions_test)
    # # targets_test = np.sum((contexts_test@A_cc)*contexts_test, axis=1) + np.sum((contexts_test@A_ca)*actions_test,axis=1) \
    # #           + np.sum((contexts_test@A_aa)*actions_test, axis=1) + contexts_test@l_c + actions_test@l_a + b + np.random.normal(0, 0.0, size=contexts_test.shape[0])
    #
    # regressor = QuadFuncJoint(ctxt_dim, action_dim, reg_fact=1e-10, normalize=True, unnormalize_output=True)
    # samples = np.concatenate((contexts, actions), axis=-1)
    # samples_mean = np.mean(samples, axis=0)
    # samples_cov = np.cov(samples.T)
    # samples_chol = np.linalg.cholesky(samples_cov)
    # regressor.fit((contexts, actions), targets, sample_mean=samples_mean, sample_chol_cov=samples_chol)
    # regressor_cc = regressor.quad_term_xx
    # regressor_aa = regressor.quad_term_yy
    # regressor_ca = regressor.quad_term_yx.T
    # regressor_c = regressor.lin_term_x
    # regressor_a = regressor.lin_term_y
    # preds = regressor.predict(contexts, actions)
    # preds_test = regressor.predict(contexts_test, actions_test)
    # print("training error:", np.mean((preds-targets)**2))
    # print("testing error:", np.mean((preds_test - targets_test)**2))

    ####################################################################################################################
    contexts = np.load('contexts.npy')
    print(contexts.shape)
    samples = np.load('samples.npy')
    rewards = np.load('rewards.npy')
    is_weights = np.load('is_weights.npy')
    # sample_mean = np.load('sample_mean.npy')
    # samples_chol = np.load('sample_chol_cov.npy')
    ctxt_dim = contexts.shape[1]
    action_dim = samples.shape[1]

    contexts_train = contexts[:int(contexts.shape[0]*0.8), :]
    contexts_test = contexts[int(contexts.shape[0]*0.8):, :]

    samples_train = samples[:int(contexts.shape[0]*0.8), :]
    samples_test = samples[int(contexts.shape[0]*0.8):, :]

    rewards_train = rewards[:int(contexts.shape[0]*0.8)]
    rewards_test = rewards[int(contexts.shape[0]*0.8):]

    is_weights_train = is_weights[:int(contexts.shape[0]*0.8)]
    is_weights_test = is_weights[int(contexts.shape[0]*0.8):]

    is_weights_train = np.expand_dims(is_weights_train, -1)
    context_mean = np.sum(is_weights_train * contexts_train, axis=0)
    diff = contexts_train - context_mean
    context_covar = diff.T @ (is_weights_train * diff)
    lin_mat, bias = np.zeros((1, 21)), np.zeros(21)
    cond_covar = np.eye(samples_train.shape[1])
    joint_mean = np.concatenate([context_mean, bias + context_mean @ lin_mat])
    # See https://scicomp.stackexchange.com/questions/5050/cholesky-factorization-of-block-matrices
    try:
        cc_u = np.linalg.cholesky(context_covar)
    except np.linalg.LinAlgError:
        cc_u = np.linalg.cholesky(context_covar + 1e-6 * np.eye(context_covar.shape[0]))
    cc_s = np.linalg.solve(cc_u, context_covar @ lin_mat).T
    cc_l = np.linalg.cholesky(cond_covar)
    cc = np.concatenate([np.concatenate([cc_u, np.zeros([context_mean.shape[0], bias.shape[0]])], axis=-1),
                         np.concatenate([cc_s, cc_l], axis=-1)], axis=0)

    sample_mean = joint_mean
    samples_chol = cc

    regressor = QuadFuncJoint(ctxt_dim, action_dim, reg_fact=1e-10, normalize=True, unnormalize_output=True)
    regressor.fit((contexts_train, samples_train), rewards_train, sample_mean=sample_mean, sample_chol_cov=samples_chol, weights=is_weights_train)
    regressor_cc = regressor.quad_term_xx
    regressor_aa = regressor.quad_term_yy
    regressor_ca = regressor.quad_term_yx.T
    regressor_c = regressor.lin_term_x
    regressor_a = regressor.lin_term_y

    preds_train = regressor.predict(contexts_train, samples_train)
    preds_test = regressor.predict(contexts_test, samples_test)
    print('train error:', np.mean((preds_train-rewards_train)**2))
    print('test error:', np.mean((preds_test-rewards_test)**2))
    import matplotlib.pyplot as plt
    plt.figure("Train")
    plt.plot(contexts_train, rewards_train, 'ob')
    plt.plot(contexts_train, preds_train, 'or')
    plt.figure('test')
    plt.plot(contexts_test, rewards_test, 'ob')
    plt.plot(contexts_test, preds_test, 'or')

    ####################################################################################################################

    # np.random.seed(0)
    # def target_func(x):
    #     # - 0.5 * x ^ T Rx + x ^ T r + r_0 ** * /
    #     return 2*(x-5)**2 - 2*x - 100
    #     # return -2*x**2 + 3*x + 10
    #     # return 1*(x-5)**2 + 5
    #
    # x = np.random.normal(0, 3, size=[300, 1])
    # # x = np.linspace(-3, 13, 300)
    # # x = np.random.normal(0, 1, size=300)
    # # x = x.reshape((-1, 1))
    # y = target_func(x) + np.random.normal(0, 0.2, size=x.shape)
    # y = y.flatten()
    # x = x.reshape((-1, 1))
    # y = y.reshape((-1, 1))
    #
    # x_mean = np.mean(x, keepdims=True)
    # x_cov = np.cov(x, rowvar=False).reshape((1, 1))
    # x_chol = np.linalg.cholesky(x_cov)
    #
    # regressor = QuadFunc(1e-10, True, True)
    # regressor.fit(x, y, None, x_mean, x_chol)
    #
    # preds = regressor.predict(x)
    #
    # import matplotlib.pyplot as plt
    #
    # plt.figure()
    # plt.plot(x, y, 'or')
    # plt.plot(x, preds, 'ob')
    # plt.show()
    #
    # from sklearn.preprocessing import PolynomialFeatures
    # from sklearn.linear_model import LinearRegression
    #
    # poly = PolynomialFeatures(degree=2)
    # X_poly = poly.fit_transform(x)
    # lin = LinearRegression()
    # poly.fit(X_poly, y)
    #
    # lin.fit(X_poly, y)
    # preds_scipy = lin.predict(X_poly)
    #
    # plt.figure()
    # plt.plot(x, y, 'or')
    # plt.plot(x, preds_scipy, 'ob')



    # samples = np.load('comp_context_local_samples_list_component_0.npy')
    # rewards = np.load('comp_context_rewards_points_local_list_component_0.npy')
    #
    # samples_mean = np.mean(samples, axis=0, keepdims=True)
    # cov = np.cov(samples, rowvar=False).reshape((-1, 1))
    # samples_chol = np.linalg.cholesky(cov)
    #
    # regressor = QuadFunc(1e-10,  True, True)
    # regressor.fit(samples, rewards, None, samples_mean, samples_chol)
    #
    # preds = regressor.predict(samples)
    #
    # import matplotlib.pyplot as plt
    #
    # plt.plot(samples, rewards, 'ob')
    # plt.plot(samples, preds, 'or')
    # plt.show()

    # from sklearn.preprocessing import PolynomialFeatures
    # from sklearn.linear_model import LinearRegression

    # poly = PolynomialFeatures(degree=2)
    # X_poly = poly.fit_transform(samples)
    # lin = LinearRegression()
    # poly.fit(X_poly, rewards)
    #
    # lin.fit(X_poly, rewards)
    # preds_scipy = lin.predict(X_poly)
    #
    # plt.figure()
    # plt.plot(samples, rewards, 'or')
    # plt.plot(samples, preds_scipy, 'ob')




    # def target_func(x, y):
    #     a_yy = 2
    #     a_xy = 1
    #     a_xx = 2.5
    #     a_x = 3
    #     a_y = 2
    #     c = 2
    #     # -0.5 * x^T R_xx x + x^T R_xy y - 0.5 * y^T R_yy y + l_x^T x + l_y^T y + c
    #     return -0.5*a_xx*x**2 + -0.5*a_yy*y**2 + x*a_xy*y + a_x*x + a_y*y + c
    #
    # transf_mat = np.array([[2]])
    # y = np.random.normal(0, 0.5, size=((50, 1)))
    # x = y@transf_mat + np.random.normal(0, 0.1, size=y.shape)
    #
    # mean_x = np.mean(x, axis=0, keepdims=True)          # actions
    # mean_y = np.mean(y, axis=0, keepdims=True)          # contexts
    # cov_x = np.cov(x, rowvar=False).reshape((1, 1))
    # cov_y = np.cov(y, rowvar=False).reshape((1, 1))
    #
    # joint_distr_mean = np.concatenate((mean_y, mean_y@transf_mat), axis=0).squeeze()
    # lin_transf_cov = cov_y@transf_mat
    # quad_transf_cov = transf_mat.T@cov_y@transf_mat
    # joint_distr_cov = np.zeros((cov_y.shape[0]+lin_transf_cov.shape[0], lin_transf_cov.shape[1]+cov_y.shape[1]))
    # joint_distr_cov[:cov_y.shape[0], :cov_y.shape[1]] = cov_y
    # joint_distr_cov[cov_y.shape[0]:, :cov_y.shape[1]] = lin_transf_cov
    # joint_distr_cov[:cov_y.shape[0], cov_y.shape[1]:] = lin_transf_cov
    # joint_distr_cov[cov_y.shape[0]:, cov_y.shape[1]:] = cov_y + quad_transf_cov
    # joint_distr_chol = np.linalg.cholesky(joint_distr_cov)
    # xy = np.concatenate((x, y), axis=-1)
    # t = target_func(x, y)
    #
    # regressor = QuadFuncJoint(1, 1, 1e-10, True, True)
    # regressor.fit((y, x), t, None, joint_distr_mean, joint_distr_chol)
    #
    # plt_range = np.arange(-0.5, 0.5, 0.01)
    # plt_grid = np.stack(np.meshgrid(plt_range, plt_range), axis=-1)
    # flat_plt_samples = np.reshape(plt_grid, [-1, 2])
    #
    #
    # preds = regressor.predict_features(flat_plt_samples).reshape((-1, 1))
    # real_targets = target_func(flat_plt_samples[:, 0], flat_plt_samples[:, 1])
    #
    #
    # import matplotlib.pyplot as plt
    # from matplotlib import cm
    # X, Y = np.meshgrid(plt_range, plt_range)
    # fig = plt.figure('Predictions')
    # ax_fig = fig.add_subplot(111)
    # plt.scatter(flat_plt_samples[:, 0], flat_plt_samples[:, 1], preds, marker='.', c='b')
    # plt.scatter(flat_plt_samples[:, 0], flat_plt_samples[:, 1], real_targets, marker='.', c='r')
    #
    # a_yy = 2
    # a_xy = 1
    # a_xx = 2.5
    # a_x = 3
    # a_y = 2
    # c = 2
    #
    # print('target aa_yy:', a_yy)
    # print('model a_yy', regressor.quad_term_xx)
    # print('error:', np.linalg.norm(a_yy-regressor.quad_term_xx))
    # print("")
    # print('target aa_xx:', a_xx)
    # print('model a_yy', regressor.quad_term_yy)
    # print('error:', np.linalg.norm(a_xx - regressor.quad_term_yy))
    # print("")
    # print('target aa_xy:', a_xy)
    # print('model a_xy', regressor.quad_term_yx)
    # print('error:', np.linalg.norm(a_xy - regressor.quad_term_yx))
    # print("")
    # print('target aa_x:', a_x)
    # print('model a_x', regressor.lin_term_y)
    # print('error:', np.linalg.norm(a_x - regressor.lin_term_y))
    # print("")
    # print('target aa_y:', a_y)
    # print('model a_y', regressor.lin_term_x)
    # print('error:', np.linalg.norm(a_y - regressor.lin_term_x))
    # print("")
    # print('target bias:', c)
    # print('model bias', regressor.const_term)
    # print('error:', np.linalg.norm(c - regressor.const_term))
    # print("")
    #
    # print('prediction error on plot data:', np.linalg.norm(preds - real_targets))
