#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Service相关异常定义
"""

class ServiceError(Exception):
    """Service基础异常"""
    
    def __init__(self, message: str, service: str = None):
        self.message = message
        self.service = service
        super().__init__(self.format_message())
    
    def format_message(self) -> str:
        """格式化异常消息"""
        if self.service:
            return f"Service[{self.service}]: {self.message}"
        return self.message


class KnowledgeGraphError(ServiceError):
    """知识图谱相关异常"""
    
    def __init__(self, message: str):
        super().__init__(message, "KnowledgeGraph")


class PromptGenerationError(ServiceError):
    """提示词生成相关异常"""
    
    def __init__(self, message: str):
        super().__init__(message, "PromptGeneration")


class EvaluationError(ServiceError):
    """评估相关异常"""
    
    def __init__(self, message: str):
        super().__init__(message, "Evaluation")


class FilteringError(ServiceError):
    """过滤相关异常"""
    
    def __init__(self, message: str):
        super().__init__(message, "Filtering")


class DatabaseConnectionError(ServiceError):
    """数据库连接异常"""
    
    def __init__(self, message: str, database_type: str = None):
        service_name = f"Database[{database_type}]" if database_type else "Database"
        super().__init__(message, service_name)


class ExportError(ServiceError):
    """导出相关异常"""
    
    def __init__(self, message: str, export_type: str = None):
        service_name = f"Export[{export_type}]" if export_type else "Export"
        super().__init__(message, service_name)
