#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
ASR（攻击成功率）评估相关接口定义
"""

from abc import ABC, abstractmethod
from typing import Dict, Any
from dataclasses import dataclass


@dataclass
class ASRResult:
    """ASR评估结果"""
    asr_successful: bool
    target_response: str
    evaluation_reasoning: str
    is_refusal: bool
    confidence: float = 0.0
    metadata: Dict[str, Any] = None


class IASREvaluator(ABC):
    """ASR评估器接口"""
    
    @abstractmethod
    def check_asr(self, rewritten_prompt: str, category: str, original_prompt: str = None) -> ASRResult:
        """
        执行攻击成功率检查
        
        Args:
            rewritten_prompt: 重写后的提示词
            category: 攻击类别
            original_prompt: 原始提示词
            
        Returns:
            ASR评估结果
        """
        pass
    
    @abstractmethod
    def calculate_semantic_similarity(self, prompt1: str, prompt2: str) -> float:
        """计算语义相似度"""
        pass
    
    @abstractmethod
    def calculate_ppl(self, prompt: str) -> float:
        """计算困惑度"""
        pass


class IConstraintChecker(ABC):
    """约束条件检查器接口"""
    
    @abstractmethod
    def meets_constraints(self, original_prompt: str, current_prompt: str, 
                         similarity_threshold: float = 0.7, 
                         ppl_threshold: float = 50.0) -> bool:
        """检查是否满足约束条件"""
        pass
