#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
隐式攻击相关接口定义
"""

from abc import ABC, abstractmethod
from typing import List, Dict, Any
from dataclasses import dataclass
from enum import Enum


class ImplicitMethod(Enum):
    """隐式攻击方法枚举"""
    BASIC = "basic"
    ENHANCED = "enhanced"


@dataclass
class ImplicitRequest:
    """隐式攻击请求"""
    original_prompt: str
    method: ImplicitMethod
    category: str = ""
    node_name: str = ""
    semantic_card: str = ""  # Enhanced方法需要的语义卡片


@dataclass
class ImplicitPrompt:
    """隐式攻击提示词"""
    original_prompt: str
    implicit_prompt: str
    method: ImplicitMethod
    category: str
    ppl_score: float = 0.0  # 困惑度分数
    similarity_score: float = 0.0
    prompt_id: str = ""
    node_name: str = ""


class IImplicitGenerator(ABC):
    """隐式攻击提示词生成器接口"""
    
    @abstractmethod
    def generate_implicit_prompt(self, request: ImplicitRequest) -> ImplicitPrompt:
        """生成隐式攻击提示词"""
        pass
    
    @abstractmethod
    def batch_generate(self, requests: List[ImplicitRequest]) -> List[ImplicitPrompt]:
        """批量生成隐式攻击提示词"""
        pass
    
    @abstractmethod
    def get_supported_methods(self) -> List[ImplicitMethod]:
        """获取支持的隐式攻击方法"""
        pass


class IImplicitEvaluator(ABC):
    """隐式攻击效果评估器接口"""
    
    @abstractmethod
    def evaluate_concealment(self, original: str, implicit: str) -> float:
        """评估隐蔽性"""
        pass
    
    @abstractmethod
    def evaluate_similarity(self, original: str, implicit: str) -> float:
        """评估相似性"""
        pass
